/**
* @file MessageIDs.h
*
* Declaration of ids for debug messages.
* 
* @author Martin Ltzsch
*/

#ifndef __MessageIDs_h_
#define __MessageIDs_h_


/** 
* IDs for debug messages
*
* To distinguish debug messages, they all have an id.
*/
enum MessageID
{
  undefined, /*!< undefined message type */
    
    idText, /*!< the message contains plain unstructured text */ 
    
    // images
    idLowResImage, /*!< the message is an old image (this should be id 2 for compatibility with old logfiles) */
    idDebugImage, /*!< the message is a debug image (see macro SEND_DEBUG_IMAGE) */
    idDebugColorClassImage, /*!< the message is a debug color class image (see macro SEND_DEBUG_COLOR_CLASS_IMAGE) */

    // debugging, request.dat
    idDebugKeyTable, /*!< the message is a DebugKeyTable */
    idQueueFillRequest, /*!< the message is a QueueFillRequest */
    
    // percepts, models
    idSensorData, /*!< the message is SensorData */
    idPercepts, /*!< the message contains percepts (see macro SEND_PERCEPTS) */
    idSpecialPercept, /*!< the message is a SpecialPercept */
    idWorldState, /*!< the message contains the world state (see macro SEND_WORLDSTATE) */
    idOracledWorldState, /*!< the message is an oracled WorldState */
    idMotionRequest, /*!< the message is a MotionRequest */

    // motion
    idOdometryData, /*!< the message is OdometryData */
    idJointData, /*!< the message is a JointData */
    idJointDataSequence, /*!< the message is a JointDataSequence */
    idHeadMotionRequest, /*!< the message is a HeadMotionRequest */
    idHeadControlMode, /*!< the message is a HeadControlMode */
    idFourierCoefficients,     /*!< the message contains a set of fourier coefficients */
    idLEDRequest, /*!< the message is a LEDRequest */
    idInvKinWalkingParameters, /*!< the message contains parameters for the InvKinWalkingEngine */
    idYetAnotherInvKinParams, /*!< contains a parameters for an inv kin engine */
    idYetAnotherInvKinPaws, /*!< contains a parameters for the paws positions of an inv kin engine */
    idExecutedMotionRequest, /*!< the message is the executed MotionRequest */
    idPIDData, /*!< The message is a PIDData */
    idJoystickData, /*!< the message is a JoystickData */
    idSoundRequest, /*!< The message is a SoundRequest */
    idMotionRatingBehaviorMessage, /*!< the message is a MotionRatingBehavior debug message */
    
    // debugging
    idGenericDebugData, /*!< data/parameters used for debugging and parameter optimization */
    idSolutionRequest, /*!< the message is a SolutionRequest */
    idDebugDrawing, /*!< the message is a DebugDrawing */
    idDebugDrawing2, /*!< the message is a DebugDrawing */
    idDebugDrawingFinished, /*!< the message is a DebugDrawing */
    idStopwatch, /*!< the message contains stopwatch data */
    idHistogram, /*!< contains a histogram */
    
    // configuration
    idCameraParameters, /*!< the message is CameraParameters */
    idColorTable64, /*!< the message is a ColorTable64 */

    //////////////////////////////////////////////////////////////////////
    idJPEGImage, /*!< the message is a JPEGImage */ ////////////////////////////////////
    //////////////////////////////////////////////////////////////////////
    idImage, /*!< the message is an Image (see macro SEND_IMAGE) */

    idGameSpeed, /*! the message contains a desired game speed */
    idBodyOffsets, /*!< the message is the generic debug data for body offsets. */
    idOdometryScale, /*!< the message is the generic debug data for odoemtry scale. */
    idGT2003BehaviorConfiguration, /*!< the message is a GT2003Behavior configuration. */
    idGameControlData, /*!< the message is a GameControlData message */
    idGlobalGameControlData, /*!< the message is a GlobalGameControlData */
    
    idXabsl2DebugRequest, /*!< the message is a Xabsl2DebugRequest */
    idXabsl2DebugMessage, /*!< the message contains a debug message from the Xabsl2BehaviorControl */
    idXabsl2IntermediateCode, /*!< intermediate code for a Xabsl2BehaviorControl */
    idXabsl2StateActivationTimes, /*!< contains state activation time that are sent by the Xabsl2 Behavior Test. */
    
    idLinesImageProcessorParameters,
    idLinesSelfLocatorParameters,
    idParametersForGridImageProcessor2,
    idParametersFromGridImageProcessor2,
    idGridImageProcessor2DebugParameters,
    idColorTableCuboids, /*!< contains a colorTableCuboids */
    idColorTableReferenceColor, /*!< contains a colorTableReferenceColor */
    idColorTableTSL, /*!< the message is a ColorTableTSL */

    idAlLxComm, /*!< the message is a AlLx_CommPacket */
    idUDParameters, /*!< the message contains UDParameters set for the UDWalkingEngine */
    idUDEvolutionRequest, /*!< the message contains a request like 'current individual to measure is #7' */
    idMotionNet, /*!< The message contains a complete motion net for the special actions */

		idSingleBeliefs, /*!< The message contains the SingleBeliefs of the robot */
		idChoosenOption, /*!< The message contains the ChoosenOption of the robot */
    idChoosenTacticEntry, /*!< The message contains the ChoosenTacticEntry of the robot */
    idTacticEntryArray,  /*!< The message contains a TacticEntryArray */
    idFreezeRequest,     /*!< The message contains a freeze Request */
    idTimeWeights,     /*!< The message contains aging information on that robot */

    idKalmanData, /*!< settings for Kalman Ball Locator Filter */
    idKickSelectionTable, /*!< A KickSelectionTable */
	numOfMessageIDs /*!< the number of message ids */
};

#ifdef USE_GETMESSAGEIDNAME
/** returns names for message ids */
static const char* getMessageIDName(MessageID id)
{
  switch (id)
  {
  case undefined: return "undefined";
  case idText: return "Text";

  // images
  case idImage: return "Image";
  case idLowResImage: return "LowResImage";
  case idDebugImage: return "DebugImage"; 
  case idDebugColorClassImage: return "DebugColorClassImage"; 

  // percepts"; models
  case idSensorData: return "SensorData";
  case idPercepts: return "Percepts";
  case idSpecialPercept: return "SpecialPercept";
  case idWorldState: return "WorldState";
  case idOracledWorldState: return "OracledWorldState";
  case idMotionRequest: return "MotionRequest";

  // motion
  case idOdometryData: return "OdometryData";
  case idJointData: return "JointData";
  case idJointDataSequence: return "JointDataSequence";
  case idHeadMotionRequest: return "HeadMotionRequest";
  case idHeadControlMode: return "HeadControlMode";
  case idFourierCoefficients: return "FourierCoefficients";
  case idLEDRequest: return "LEDRequest";
  case idInvKinWalkingParameters: return "InvKinWalkingParameters";
  case idExecutedMotionRequest: return "ExecutedMotionRequest";
  case idPIDData: return "PidData";
  case idJoystickData: return "JoystickData";
  case idSoundRequest: return "SoundRequest";
  case idMotionRatingBehaviorMessage: return "idMotionRatingBehaviorMessage";

  // debugging
  case idGenericDebugData: return "GenericDebugData";
  case idDebugKeyTable: return "DebugKeyTable";
  case idSolutionRequest: return "SolutionRequest";
  case idDebugDrawing: return "DebugDrawing";
  case idDebugDrawing2: return "DebugDrawing2";
  case idDebugDrawingFinished: return "DebugDrawingFinished";
  case idQueueFillRequest: return "QueueFillRequest";
  case idStopwatch: return "Stopwatch";
  case idHistogram: return "Histogram";

  // configuration
  case idCameraParameters: return "CameraParameters";
  case idColorTable64: return "ColorTable64";
  case idGameSpeed: return "GameSpeed";
  case idBodyOffsets: return "BodyOffsets";

  //////////////////////////////////////////////////////////////////////
  case idJPEGImage: return "JPEGImage";
  //////////////////////////////////////////////////////////////////////

  case idGT2003BehaviorConfiguration: return "GT2003BehaviorConfiguration";
  case idGameControlData: return "GameControlData";
  case idGlobalGameControlData: return "GlobalGameControlData";

  case idXabsl2DebugRequest: return "Xabsl2DebugRequest";
  case idXabsl2DebugMessage: return "Xabsl2DebugMessage";
  case idXabsl2IntermediateCode: return "Xabsl2IntermediateCode";
  case idXabsl2StateActivationTimes: return "Xabsl2StateActivationTimes";

  case idLinesImageProcessorParameters: return "LinesImageProcessorParameters";
  case idLinesSelfLocatorParameters: return "LinesSelfLocatorParameters";
  case idParametersForGridImageProcessor2: return "ParametersForGridImageProcessor2";
  case idParametersFromGridImageProcessor2: return "ParametersFromGridImageProcessor2";
  case idGridImageProcessor2DebugParameters: return "GridImageProcessor2DebugParameters";
  case idColorTableCuboids: return "ColorTableCuboids";
  case idColorTableReferenceColor: return "ColorTableReferenceColor";
  case idColorTableTSL: return "ColorTableTSL";

  case idAlLxComm: return "AlLx_CommPacket";
  case idUDParameters: return "UDParameters";
  case idUDEvolutionRequest: return "UDEvolutionRequest";
  case idMotionNet: return "MotionNet";

	case idSingleBeliefs: return "SingleBeliefs";
	case idChoosenOption: return "ChoosenOption";
  case idChoosenTacticEntry: return "ChoosenTacticEntry";
  case idTacticEntryArray: return "TacticEntryArray";
  case idFreezeRequest: return "FreezeRequest";
  case idTimeWeights: return "TimeWeights";
  case idKickSelectionTable: return "KickSelectionTable";
  case idKalmanData: return "KalmanData";

  default: return "unknown";
  }
}
#endif

/** Composes the idImage message into a message queue */
#define SEND_IMAGE(i,cm) static_cast<Image&>(i) << static_cast<CameraMatrix&>(cm) 
/** Receives the idImage message from a message queue */
#define RECEIVE_IMAGE(i,cm) static_cast<Image&>(i) >> static_cast<CameraMatrix&>(cm)

/** Composes the idPercepts message into a message queue */
#define SEND_PERCEPTS(cm,ci,ballp,lmp,lnsp,plp,op,psdp,cp) \
  static_cast<CameraMatrix&>(cm) << static_cast<CameraInfo&>(ci) << static_cast<BallPercept&>(ballp) \
  << static_cast<LandmarksPercept&>(lmp) << static_cast<LinesPercept&>(lnsp) << static_cast<PlayersPercept&>(plp) << static_cast<ObstaclesPercept&>(op) << static_cast<PSDPercept&>(psdp) << static_cast<CollisionPercept&>(cp)
/** Receives the idPercepts message from a message queue */
#define RECEIVE_PERCEPTS(cm,ci,ballp,lmp,lnsp,plp,op,psdp,cp) \
  static_cast<CameraMatrix&>(cm) >> static_cast<CameraInfo&>(ci) >> static_cast<BallPercept&>(ballp) \
  >> static_cast<LandmarksPercept&>(lmp) >> static_cast<LinesPercept&>(lnsp) >> static_cast<PlayersPercept&>(plp) >> static_cast<ObstaclesPercept&>(op) >> static_cast<PSDPercept&>(psdp) >> static_cast<CollisionPercept&>(cp)

/** Composes the idWorldState message into a message queue */
#define SEND_WORLD_STATE(rp,bp,ppc,om,rs,cm,ci) static_cast<RobotPose&>(rp) \
  << static_cast<BallModel&>(bp) << static_cast<PlayerPoseCollection&>(ppc) << static_cast<ObstaclesModel&>(om) << static_cast<RobotState&>(rs) << static_cast<CameraMatrix&>(cm) << static_cast<CameraInfo&>(ci)
/** Receives the idWorldState message from a message queue */
#define RECEIVE_WORLDSTATE(rp,bp,ppc,om,rs,cm,ci) static_cast<RobotPose&>(rp) \
  >> static_cast<BallModel&>(bp) >> static_cast<PlayerPoseCollection&>(ppc) >> static_cast<ObstaclesModel&>(om) >> static_cast<RobotState&>(rs) >> static_cast<CameraMatrix&>(cm) >> static_cast<CameraInfo&>(ci)

#endif //__MessageIDs_h_

/*
 * Change log :
 * 
 * $Log: MessageIDs.h,v $
 * Revision 1.29  2004/05/03 15:32:25  uhrig
 * Added additional ball state and disabled sending of Kalman filter process model states.
 *
 * Revision 1.28  2004/04/07 13:00:49  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.3  2004/04/06 17:10:26  risler
 * renamed idOldImage to idLowResImage
 *
 * Revision 1.2  2004/03/29 15:53:39  risler
 * added oldImage ID for old logfile support
 *
 * Revision 1.1.1.1  2004/03/29 08:28:42  Administrator
 * initial transfer from tamara
 *
 * Revision 1.27  2004/03/22 21:58:15  roefer
 * True odometry
 *
 * Revision 1.26  2004/03/18 18:04:25  roefer
 * idJPEGImage repaired
 *
 * Revision 1.25  2004/03/17 19:52:24  juengel
 * Added idKickSelectionTable.
 *
 * Revision 1.24  2004/03/17 16:18:49  thomas
 * added preversion of motion optimisation with behaviour, selflocator, headcontrol and robotcontrol dialog
 *
 * Revision 1.23  2004/03/11 14:27:31  uhrig
 * KalmanBallLocator was added
 *
 * Revision 1.22  2004/02/27 16:41:17  dueffert
 * UDEvolutionRequest introduced
 *
 * Revision 1.21  2004/02/17 11:44:16  roefer
 * idJPEGMessage corrected
 *
 * Revision 1.20  2004/02/16 19:22:47  jhoffman
 * added debug parameters for yet another inv kin
 *
 * Revision 1.19  2004/02/03 13:19:49  spranger
 * renamed all references to  class BallPosition to BallModel
 *
 * Revision 1.18  2004/01/28 21:54:11  kerdels
 * fixed aging...
 *
 * Revision 1.17  2004/01/12 17:17:52  kerdels
 * added freeze functionality to tacticDesigner
 *
 * Revision 1.16  2004/01/10 10:09:14  juengel
 * Added CameraInfo to and removed Player from (SEND|RECEIVE)_(PERCEPTS|WORLDSTATE).
 *
 * Revision 1.15  2004/01/07 14:56:09  kerdels
 * added send functionality to TacticDesigner
 *
 * Revision 1.14  2004/01/06 00:01:57  kerdels
 * Aging im TacticChooser hinzugefgt,
 * OptionRatingDlgBar um choosenTacticEntry erweitert
 *
 * Revision 1.13  2003/12/28 23:00:12  kerdels
 * tactic chooser berarbeitet
 *
 * Revision 1.12  2003/12/28 19:03:30  kerdels
 * option rating dialog added
 *
 * Revision 1.11  2003/12/16 19:02:45  loetzsch
 * The motion net file Config/spec_act.dat can be sent through WLAN to a robot.
 *
 * Revision 1.10  2003/12/02 18:08:27  dueffert
 * first working not yet calibrated version of UDWalkingEngine added
 *
 * Revision 1.9  2003/12/01 16:37:14  juengel
 * order changed again
 *
 * Revision 1.8  2003/11/29 07:43:05  roefer
 * DON'T add MessageIDs before idJPEGImage
 *
 * Revision 1.7  2003/11/26 15:57:53  juengel
 * order changed
 *
 * Revision 1.6  2003/11/22 15:59:03  jhoffman
 * parameter sending using generic debug data to
 * walking engine now works if it is explicitely given
 * a messageID
 *
 * Revision 1.5  2003/11/22 09:33:01  jhoffman
 * added "Yet Another Inv Kin Walking Engine"
 * experimental stage, doesn't do much yet
 * no Fourier inside!
 *
 * Revision 1.4  2003/11/17 14:56:04  urban
 * added SensorBehaviorControl-solution "AlLx RemotePresence" and RobotControl-dialogbar "AlLx Joystick"
 *
 * Revision 1.3  2003/11/03 20:16:09  juengel
 * color class images can be sent from robot now
 *
 * Revision 1.2  2003/10/23 07:26:57  juengel
 * Renamed ColorTableAuto to ColorTableReferenceColor.
 *
 * Revision 1.1  2003/10/07 10:13:24  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.7  2003/09/26 21:23:20  loetzsch
 * renamed class JointState to CollisionPercept
 *
 * Revision 1.6  2003/09/05 16:46:42  juengel
 * resorted
 *
 * Revision 1.5  2003/09/01 10:23:14  juengel
 * DebugDrawings clean-up 2
 * DebugImages clean-up
 * MessageIDs clean-up
 * Stopwatch clean-up
 *
 * Revision 1.4  2003/08/25 17:22:51  juengel
 * Added Histogram
 *
 * Revision 1.3  2003/08/18 11:50:48  juengel
 * Added idColorTableAuto
 *
 * Revision 1.2  2003/07/04 16:39:42  loetzsch
 * added a nice message id
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.42  2003/06/02 09:39:34  goehring
 * Added JointState to SEND_PERCEPTS and RECEIVE_PERCEPTS.
 *
 * Revision 1.41  2003/05/27 12:23:42  risler
 * added GT2003BehaviorConfiguration debug message
 *
 * Revision 1.40  2003/05/26 08:18:17  juengel
 * Added cameraMatrix to worldState.
 *
 * Revision 1.39  2003/05/20 10:35:59  dueffert
 * missing id names added
 *
 * Revision 1.38  2003/05/05 12:23:27  juengel
 * Removed idGridImageProcessor2ColorCubes and idGridImageProcessor2Thresholds
 *
 * Revision 1.37  2003/05/03 16:20:43  roefer
 * bodyOffsets generic debug data
 *
 * Revision 1.36  2003/04/16 14:14:38  loetzsch
 * removed Xabsl 1 from GT2003
 *
 * Revision 1.35  2003/03/28 14:29:58  juengel
 * Added PSDPercept to SEND_PERCEPTS and RECEIVE_PERCEPTS.
 *
 * Revision 1.34  2003/03/25 08:47:11  dasmuli
 * Added simple segmentation and the 180/360 memory. Dialog was changed in order to display segemented data.
 *
 * Revision 1.33  2003/03/20 20:34:21  loetzsch
 * Game Toolbar now can adjust the game speed
 *
 * Revision 1.32  2003/03/10 14:06:15  juengel
 * Added ObstaclesLocator
 *
 * Revision 1.31  2003/03/07 19:04:41  juengel
 * Added messageID executedMotionRequest
 *
 * Revision 1.30  2003/03/06 18:19:34  dueffert
 * reduced amount of warnings drastically by ifdefing global functions
 *
 * Revision 1.29  2003/03/05 14:07:10  roefer
 * LinesPerceptor2 is now LinesImageProcessor
 *
 * Revision 1.28  2003/03/05 11:10:05  loetzsch
 * added class GlobalGameControlData
 *
 * Revision 1.27  2003/03/04 14:52:00  juengel
 * Added idParametersForGridImageProcessor2 and idParametersFromGridImageProcessor2.
 *
 * Revision 1.26  2003/02/27 12:55:26  juengel
 * Added some drawings.
 *
 * Revision 1.25  2003/02/27 12:02:30  dueffert
 * JoystickData added
 *
 * Revision 1.24  2003/02/21 22:20:13  roefer
 * LinesSelfLocator is working
 *
 * Revision 1.23  2003/02/19 12:10:43  dueffert
 * new ids should be placed at the very end!
 *
 * Revision 1.22  2003/02/18 21:29:17  osterhues
 * Changed all instances of ColorTable64 to new base class ColorTable
 *
 * Revision 1.21  2003/02/08 18:34:56  juengel
 * added idGridImageProcessor2Thresholds
 *
 * Revision 1.20  2003/02/08 16:06:02  roefer
 * ID sequence changed to be compatible with existing log files
 *
 * Revision 1.19  2003/01/30 11:28:54  juengel
 * Added idGridImageProcessor2Parameters and id LinesPerceptorParameters.
 *
 * Revision 1.18  2003/01/21 20:36:14  loetzsch
 * xabsl2 intermediate code now can be handled by Xabsl2BehaviorControl
 *
 * Revision 1.17  2003/01/16 09:55:17  jhoffman
 * Added "Generic Debug Data" data type. This can be used
 * for quick and dirty optimization and debugging, e.g. to send
 * parameters to a module through WLAN to adjust it's settings.
 * The DebugMessageGenerator is used to for parsing and
 * sendig the data
 *
 * Revision 1.16  2003/01/08 18:45:30  loetzsch
 * added xabsl2 messages
 *
 * Revision 1.15  2002/12/06 11:32:13  juengel
 * Added idXabsl2DebugRequest and idXabsl2DebugMessage.
 *
 * Revision 1.14  2002/11/27 18:01:09  dueffert
 * number and order of getMessageIDName-entries corrected
 *
 * Revision 1.13  2002/11/26 19:19:24  loetzsch
 * JPEG images are put into local processes
 *
 * Revision 1.12  2002/11/19 17:08:48  risler
 * added datatype PIDData
 * support for sending new pid values at runtime
 *
 * Revision 1.11  2002/11/19 12:57:55  dueffert
 * queuefillrequest added, debugkey toolbar extended, memorystick toolbar removed
 *
 * Revision 1.10  2002/11/18 17:30:31  loetzsch
 * Sound requests generated from BehaviorControl are now automatically sent
 * to RobotControl and played there.
 *
 * Revision 1.9  2002/11/18 10:35:05  juengel
 * Added "block debug sender" mechanism
 *
 * Revision 1.8  2002/11/16 09:34:45  dueffert
 * casts corrected
 *
 * Revision 1.7  2002/11/11 11:27:11  juengel
 * First Step: New debug drawing macros.
 *
 * Revision 1.6  2002/10/11 13:54:43  roefer
 * JPEGImage added
 *
 * Revision 1.5  2002/10/10 13:09:50  loetzsch
 * First experiments with the PSD Sensor
 * - SensorDataProcessor now calculates PSDPercept
 * - Added the PerceptBehaviorControl solution PSDTest
 * - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
 *
 * Revision 1.4  2002/09/19 23:38:54  juengel
 * Changed debug image mechanisms.
 *
 * Revision 1.3  2002/09/18 16:35:13  loetzsch
 * made GT2003 compilable,
 * rechanged message ids
 *
 * Revision 1.2  2002/09/17 23:55:24  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:53:59  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/08/21 11:50:44  cesarz
 * GT2001PlayersPerceptor, BerlinSelfLocator, GT2001WalkingEngine,
 * DoWalkingEngine, FuzzyBehaviorControl, BremenGOBehaviorControl,
 * FuzzyBehaviortesterdlgbar and Behaviortesterdlgbar removed.
 *
 * Revision 1.1  2002/07/23 13:47:14  loetzsch
 * - new streaming classes
 * - new debug message handling
 * - exchanged StaticQueue by MessageQueue
 *
 * Revision 1.3  2002/06/08 11:44:00  mkunz
 * Special Percept on Field
 *
 * Revision 1.2  2002/05/15 15:47:29  risler
 * added debug message AcousticMessage, DebugMessageGenerator generates AcousticMessage
 *
 * Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.32  2002/05/05 22:12:42  loetzsch
 * GameControlData can now be sent from the Game toolbar to BehaviorControl
 *
 * Revision 1.31  2002/05/02 17:04:09  roefer
 * New router
 *
 * Revision 1.30  2002/04/26 13:35:33  risler
 * DarmstadtGOWalkingEngine renamed to InvKinWalkingEngine
 * added InvKinParameterSets
 *
 * Revision 1.29  2002/04/23 15:37:06  jhoffman
 * added some pre GO fourier stuff
 *
 * Revision 1.28  2002/04/23 15:05:17  jhoffman
 * renamed FourierCoefficient id
 * added change log
 *
 *
 */
