/**
* @file LogPlayerWithSync.cpp
*
* Implementation of class LogPlayerWithSync
*
* @author Jrn Hamerla
*/

#include "LogPlayerWithSync.h"

#include "Bars/WorldstateAnalysisDlgBar.h"

#include "MessageIDs.h"
#include "../Representations/Cognition/RobotPose.h"
#include "../Representations/Cognition/RobotState.h"
#include "../Representations/Perception/ObstaclesPercept.h"
#include "../Representations/Perception/PSDPercept.h"
#include "../Representations/Cognition/ObstaclesModel.h"
#include "../Representations/Perception/CollisionPercept.h"
#include "../Representations/Perception/LandmarksPercept.h"
#include "../Representations/Perception/BallPercept.h"
#include "../Representations/Perception/PlayersPercept.h"
#include "../Representations/Perception/LinesPercept.h"
#include "../Representations/Cognition/PlayerPoseCollection.h"
#include "../Representations/Cognition/BallModel.h"
#include "../Representations/RoboCup/GameControlData.h"


LogPlayerWithSync::LogPlayerWithSync(MessageQueue& targetQueue, long* playtime)
: LogPlayer(targetQueue)
, m_syncTime(0)
, m_playTime(playtime)
, m_counterSend(0)
{
  _new();
}

unsigned long LogPlayerWithSync::getTimeStamp(int message)
{
  if (message < 0)
	  return 0;
  if (message >= getNumberOfMessages())
	  return -1;
  if (getMessageID(message) == idText)
	  return getTimeStamp(message+1);
  return queue.getTimeStamp(message);
}

void LogPlayerWithSync::setSelectedMessageForReading(int message)
{
	if (message < 0)
	{
		setSelectedMessageForReading(0);
		return;
	}
	if (message >= getNumberOfMessages())
	{
		setSelectedMessageForReading(getNumberOfMessages()-1);
		return;
	}
	queue.setSelectedMessageForReading(message);
}

MessageID LogPlayerWithSync::getMessageID(int message)
{
	int num = getCurrentMessageNumber();
	queue.setSelectedMessageForReading( message );
	MessageID mid = queue.getMessageID();
	setSelectedMessageForReading(num);
	return mid;
}

unsigned long LogPlayerWithSync::getSyncTime()
{
	return m_syncTime;
}

unsigned long LogPlayerWithSync::getGamestartTime()
{
	for(int i = 0; i < getNumberOfMessages(); ++i)
	{
		switch (getMessageID(i))
		{
			case idPercepts:
			{
				CameraMatrix cameraMatrix;
				CameraInfo cameraInfo;
				BallPercept	ballPercept;
				LandmarksPercept landmarksPercept;
				PlayersPercept			playersPercept;
				ObstaclesPercept		obstaclesPercept;
				PSDPercept				psdPercept;
				CollisionPercept		collisionPercept;
				LinesPercept			linesPercept;

				int num = getCurrentMessageNumber();
				queue.setSelectedMessageForReading( i );
				InMessage message(in);
				message.bin >> RECEIVE_PERCEPTS(cameraMatrix, cameraInfo, ballPercept,
				landmarksPercept,linesPercept,playersPercept, obstaclesPercept, psdPercept, collisionPercept);
				setSelectedMessageForReading(num);

				if (( ballPercept.ballWasSeen )
				||	( obstaclesPercept.numberOfPoints > 2 ))
					return getTimeStamp(i);
			}
			break;
			case idGlobalGameControlData:
			{
				int num = getCurrentMessageNumber();
				queue.setSelectedMessageForReading( i );
				InMessage message(in);

				GlobalGameControlData ggcd;
				message.bin >> ggcd;
				queue.setSelectedMessageForReading(num);

				if (ggcd.state == ggcd.playing)
				  return getTimeStamp(i);
			}
			break;
		}
	}
	return getTimeStamp(0);
}

long LogPlayerWithSync::getGameTime()
{
	return getTime()-this->getGamestartTime();
}

unsigned long LogPlayerWithSync::getTime()
{
	return getTimeStamp( getCurrentMessageNumber() );
}

unsigned long LogPlayerWithSync::getLogstartTime()
{
	return getTimeStamp(0);
}

unsigned long LogPlayerWithSync::getLogendTime()
{
	for(int i = getNumberOfMessages()-1; i > 0; --i)
	{
		if (getMessageID(i) == idText)
			continue;
		return queue.getTimeStamp(i);
	}
	return queue.getTimeStamp(0);
}

long LogPlayerWithSync::getNextStepForward()
{
	return getTimeStamp(getCurrentMessageNumber())-m_syncTime;
}

long LogPlayerWithSync::getNextStepBackward()
{
	return getTimeStamp(getCurrentMessageNumber()-1)-m_syncTime;
}

void LogPlayerWithSync::onIdle()
{
	if (m_playTime == 0)
		LogPlayer::onIdle();

	if (isEmpty())
		return;

	unsigned long atime;
	atime = m_syncTime + *m_playTime;
	
	if ( getTimeStamp(currentMessageNumber) < atime )
	{ //step forward
		LogPlayer::stepForward();
		m_counterSend++;
	}else if (getTimeStamp(currentMessageNumber-1) > atime )
	{ //setp backward
		LogPlayer::stepBackward();
		m_counterSend++;
	}else
	{
		m_counterSend = 0;
	}

}

bool LogPlayerWithSync::seek(unsigned long time)
{
	int messagenumber = getCurrentMessageNumber();
	int after(0);
	while( getTimeStamp(after) < time)
	{
		++after;
	}
	if (after <= 0)
	{
		currentMessageNumber = 0;
		return false;
	}
	if (after >= getNumberOfMessages() )
	{
		currentMessageNumber = 0;
		return false;
	}

	currentMessageNumber = after-1;
	return true;
}

InMessage LogPlayerWithSync::getMessage()
{
	in.config.reset();
	in.text.reset();
	return in;
}

bool LogPlayerWithSync::seek(long time)
{
	return seek(m_syncTime+ time);
}

void LogPlayerWithSync::setSync(unsigned long time)
{
	m_syncTime = time;
}

bool LogPlayerWithSync::hasTimeDelay()
{
	// damit nicht zuviel rckstand entsteht teile berspringen
	if (m_counterSend > 16)
		this->seek(*m_playTime);

	return m_counterSend > 15;
}

void LogPlayerWithSync::setPlayerForNewMessages(const Player&player)
{
	LogPlayer::setPlayerForNewMessages(player);
}

/*
void LogPlayerWithSync::syncToStartTime(unsigned long starttime)
{
	long tmp = getGamestartTime()-m_syncTime;
	m_syncTime = starttime-tmp;
}
*/


/*
* Change Log:
*
*/
