/**
* @file LogPlayer.h
*
* Definition of class LogPlayer
*
* @author Martin Ltzsch
*/

#ifndef __LogPlayer_h_
#define __LogPlayer_h_

#include "MessageQueue.h"

/**
* @class LogPlayer
*
* A message queue that can record and play logfiles.
* The messages are played in the same time sequence as they were recorded.
*
* @author Martin Ltzsch
*/
class LogPlayer : protected MessageQueue
{
public:
  /** 
  * Constructor 
  * @param targetQueue The queue into that messages from played logfiles shall be stored.
  */
  LogPlayer(MessageQueue& targetQueue);

  /** different states of the logplayer */
  enum LogPlayerState
  {
    initial, recording, paused, playing
  };

  /** Returns the state of the queue */
  LogPlayerState getState();

  /** Deletes all messages from the queue */
  void _new();

  /** 
  * Opens a log file and reads all messages into the queue.
  * @param fileName the name of the file to open
  * @return if the reading was successful
  */
  bool open(const char* fileName);

  /** 
  * Playes the queue. 
  * Note that you have to call onIdle() regularely if you want to use that function
  */
  void play();

  /** Pauses playing the queue. */
  void pause();

  /** Stops playing the queue, resets the position in the queue to the first message */
  void stop();

  /** Plays the next message in the queue */
  void stepForward();

  /** Plays the previous message in the queue */
  void stepBackward();

  /** repeats the current message in the queue */
  void stepRepeat();
  
  /** jumps to given message-number in the queue */
  void jumpFrame(int frame);
  
  /** 
  * Starts / Stops recording.
  * Note that you have to notify the queue on new messages with handleMessage().
  */
  void record();

  /** Activates/deactivates jpeg image smoothing */
  void smooth();
  
  /** 
  * Writes all messages in the log player queue to a log file.
  * @param fileName the name of the file to write
  * @return if the writing was successful
  */
  bool save(const char* fileName);

  /** 
  * Writes all messages in the log player queue to an AIBOVision compatibile movie file.
  * Images are encoded in a raw YUV format.
  * @param fileName the name of the file to write
  * @return if the writing was successful
  */
  bool saveAMV(const char* fileName);

  /** 
  * Writes an row of doubles to an comma separated value file.
  * @param file the open file to write to
  * @param row a row containing the doubles to be written
  * @param rowLen number of entries in row
  */
  void saveCSVrow(OutTextRawFile& file, double* row, unsigned int rowLen);
  
  /** 
  * Writes a couple of message types in the log player queue to an comma separated value file with a common time line.
  * @param fileName the name of the file to write
  * @return if the writing was successful
  */
  bool saveCSV(const char* fileName);

  /** 
  * Writes all images in the log player queue to a bunch of files (*.bmp or *.jpg).
  * @param fileName the name of one file to write, all files will be enumerated by appending a 3 digit number to the filename.
  * @return if the writing of all files was successful
  */
  bool saveImages(const char* fileName);

  /** 
  * Sets the playing speed of the log player.
  * "2" playes the messages double as fast as they were recorded, 
  * "0.5" plays the messages with the half speed.
  */
  void setPlaySpeed(double speed);

  /** 
  * Adds the message to the queue depending on isRecording. 
  * That function should be called for every message in the queue that the 
  * log player shall work on.
  */
  void handleMessage(InMessage& message);

  /** 
  * If playing a log file, that function checks if it is time to release the next
  * message dependend on the time stamp. Call that function whenever there is some
  * processing time left.
  */
  void onIdle();

  /** Returns the number of stored messages */
  int getNumberOfMessages() const;

  /** Returns the number of the current message */
  int getCurrentMessageNumber() const;

  /** Returns the type of the current message */
  MessageID getCurrentMessageID() const;
  
  /** Returns the status of the smoothing flag */
  inline bool isSmoothingEnabled() const
  {
    return smoothingEnabled;
  };

private:

  /** the state of the log player */
  LogPlayerState state;

  /* The queue into that messages from played logfiles shall be stored. */
  MessageQueue& targetQueue;

  /** little routine to convert an integer into C style string*/
  void convertIntString(char* str, int value);

  /** 
  * A factor, how fast the messages are played. 
  * "2" playes the messages double as fast as they were recorded, 
  * "0.5" plays the messages with the half speed.
  */
  double playSpeed;

  /**
  * Returns the time stamp of a given message.
  * @param message The number of the message 
  * @return The time stamp
  */
  unsigned long getTimeStamp(int message);

protected:
  /** The number of the current message */
  int currentMessageNumber;
private:

  /** The time when the first message was played */
  unsigned long timeWhenFirstMessageWasPlayed;

  /** The time stamp of the first played message */
  unsigned long timeOfFirstPlayedMessage;

  /** Flag which triggers smoothing of jpeg compressed images */
  bool smoothingEnabled;

};


#endif //__LogPlayer_h_

/*
* Change Log:
*
* $Log: LogPlayer.h,v $
* Revision 1.7  2004/05/19 07:58:15  dueffert
* saving to CSV implemented
*
* Revision 1.6  2004/03/26 16:33:57  thomas
* added field in logplayer to jump directly to a given frame-number
*
* Revision 1.5  2004/03/24 12:55:42  risler
* added logplayer repeat button
*
* Revision 1.4  2004/02/16 12:26:41  nistico
* Added noise reduction functionality for jpeg images in log file player
*
* Revision 1.3  2004/01/20 12:40:09  nistico
* - Added support for ColorTable32K (65K elements in packed format)
* - RobotControl can now convert GT *.log files into AIBOVision (external ColorTable32K calibration tool) *.amv file format
*
* Revision 1.2  2003/12/25 13:13:54  hamerla
* Logplayer extension to LogPlayerWithSync
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.4  2002/12/15 23:34:13  dueffert
* saving images from logfiles added
*
* Revision 1.3  2002/10/04 10:27:33  loetzsch
* Added functionality to adjust the speed of playing log files.
*
* Revision 1.2  2002/10/02 15:50:41  juengel
* Added getCurrentMessageID.
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1  2002/08/08 16:40:30  loetzsch
* added class LogPlayer and redesigned RobotControl's Logplayer GUI
*
*/
