/** 
 * @file Matrix.h
 * Contains template class Matrix3x3 of type V
 * and declares RotationMatrix
 *
 * @author <a href="mailto:martin.kallnik@gmx.de">Martin Kallnik</a>
 * @author Max Risler
 */

#ifndef __Matrix_h__
#define __Matrix_h__

#include "Vector3.h"

class Matrix_6x1;
class Matrix_6x3;
class Matrix_3x3;
class Matrix_3x6;

/** This class represents a 6x6-double-matrix */
class Matrix_6x6 {
public:
  /** The content of the matrix */
    double
      c00,c01,c02,c03,c04,c05,
      c10,c11,c12,c13,c14,c15,
      c20,c21,c22,c23,c24,c25,
      c30,c31,c32,c33,c34,c35,
      c40,c41,c42,c43,c44,c45,
      c50,c51,c52,c53,c54,c55;

  /** Default constructor. */
  Matrix_6x6();

  /** Constructor. */
  Matrix_6x6(
    double c00, double c01, double c02, double c03, double c04, double c05,
    double c10, double c11, double c12, double c13, double c14, double c15,
    double c20, double c21, double c22, double c23, double c24, double c25,
    double c30, double c31, double c32, double c33, double c34, double c35,
    double c40, double c41, double c42, double c43, double c44, double c45,
    double c50, double c51, double c52, double c53, double c54, double c55);
  
  //! Assignment operator
  /*!
  \param o The other matrix that is assigned to this one
  \return A reference to this object after the assignment.
  */
  Matrix_6x6& operator=(const Matrix_6x6& o);

  //! Subtraction of this matrix with another 6x6 matrix.
  /*!
  \param o The other matrix that is subtracted from this one
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_6x6 operator-(const Matrix_6x6& o) const;

  //! Multiplication of this matrix by another 6x6 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_6x6 operator*(const Matrix_6x6& o) const;
 
  //! Multiplication of this matrix by another 6x3 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_6x3 operator*(const Matrix_6x3& o) const;

  //! Multiplication of this matrix by another 6x1 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_6x1 operator*(const Matrix_6x1& o) const;

  /*! Transpose the matrix
  \return A new object containing transposed matrix
  */
  Matrix_6x6 transpose() const;
};


/** This class represents a 6x1-double-matrix */
class Matrix_6x1 {
public:
  /** The content of the matrix */
    double
      c00,
      c10,
      c20,
      c30,
      c40,
      c50;

  /** Default constructor. */
  Matrix_6x1();

  /** Constructor. */
  Matrix_6x1(
    double c00,
    double c10,
    double c20,
    double c30,
    double c40,
    double c50);
  
  //! Assignment operator
  /*!
  \param o The other matrix that is assigned to this one
  \return A reference to this object after the assignment.
  */
  Matrix_6x1& operator=(const Matrix_6x1& o);

  //! Addition of this matrix with another 6x1 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_6x1 operator+(const Matrix_6x1& o) const;
};

/** This class represents a 6x3-double-matrix */
class Matrix_6x3 {
public:
  /** The content of the matrix */
    double
      c00,c01,c02,
      c10,c11,c12,
      c20,c21,c22,
      c30,c31,c32,
      c40,c41,c42,
      c50,c51,c52;

  /** Default constructor. */
  Matrix_6x3();

  /** Constructor. */
  Matrix_6x3(
    double c00, double c01, double c02,
    double c10, double c11, double c12,
    double c20, double c21, double c22,
    double c30, double c31, double c32,
    double c40, double c41, double c42,
    double c50, double c51, double c52);
  
  //! Assignment operator
  /*!
  \param o The other matrix that is assigned to this one
  \return A reference to this object after the assignment.
  */
  Matrix_6x3& operator=(const Matrix_6x3& o);

  //! Multiplication of this matrix by a 3x6 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_6x6 operator*(const Matrix_3x6& o) const;

  //! Multiplication of this matrix by a 3x3 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_6x3 operator*(const Matrix_3x3& o) const;

  //! Multiplication of this matrix by a Vector3<double>.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_6x1 operator*(const Vector3<double>& o) const;

  /*! Transpose the matrix
  \return A new object containing transposed matrix
  */
  Matrix_3x6 transpose() const;
};


/** This class represents a 3x6-double-matrix */
class Matrix_3x6 {
public:
  /** The content of the matrix */
    double
      c00,c01,c02,c03,c04,c05,
      c10,c11,c12,c13,c14,c15,
      c20,c21,c22,c23,c24,c25;

  /** Default constructor. */
  Matrix_3x6(
    double c00, double c01, double c02, double c03, double c04, double c05,
    double c10, double c11, double c12, double c13, double c14, double c15,
    double c20, double c21, double c22, double c23, double c24, double c25);
  
  //! Assignment operator
  /*!
  \param o The other matrix that is assigned to this one
  \return A reference to this object after the assignment.
  */
  Matrix_3x6& operator=(const Matrix_3x6& o);

  //! Multiplication of this matrix by a 6x1 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Vector3<double> operator*(const Matrix_6x1& o) const;

  //! Multiplication of this matrix by a 6x3 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_3x3 operator*(const Matrix_6x3& o) const;

  //! Multiplication of this matrix by a 6x6 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_3x6 operator*(const Matrix_6x6& o) const;

  /*! Transpose the matrix
  \return A new object containing transposed matrix
  */
  Matrix_6x3 transpose() const;
};


/** This class represents a 3x3-double-matrix */
class Matrix_3x3 {
public:
  /** The content of the matrix */
  double
      c00,c01,c02,
      c10,c11,c12,
      c20,c21,c22;

  /** Default constructor. */
  Matrix_3x3();

  /** Constructor. */
  Matrix_3x3(
    double c00, double c01, double c02,
    double c10, double c11, double c12,
    double c20, double c21, double c22);
  
  //! Assignment operator
  /*!
  \param o The other matrix that is assigned to this one
  \return A reference to this object after the assignment.
  */
  Matrix_3x3& operator=(const Matrix_3x3& o);

  //! Multiplication of this matrix by a 3x3 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_3x3 operator*(const Matrix_3x3& o) const;

  //! Multiplication of this matrix by a 3x6 matrix.
  /*!
  \param o The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix_3x6 operator*(const Matrix_3x6& o) const;

  /*! Transpose the matrix
  \return A new object containing transposed matrix
  */
  Matrix_3x3 transpose() const;

  //! Calculation of the determinant of this matrix.
  /*! 
  \return The determinant.
  */
  double det() const;

  //! Calculation of the determinant of this matrix.
  /*! 
  \return The determinant.
  */
  Matrix_3x3 inverse() const;
};




/** This class represents a 3x3-matrix */
template <class V> class Matrix3x3 {
public:
  /** The columns of the matrix */
  Vector3<V> c[3];

  /** Default constructor. */
  Matrix3x3<V>() {
    c[0]=Vector3<V>(1,0,0);
    c[1]=Vector3<V>(0,1,0);
    c[2]=Vector3<V>(0,0,1);
  }

  //! Constructor.
  /*!
  \param c0 the first column of the matrix.
  \param c1 the second column of the matrix.
  \param c2 the third column of the matrix.
  */
  Matrix3x3<V>(const Vector3<V>& c0,const Vector3<V>& c1,const Vector3<V>& c2)
    {c[0]=c0;c[1]=c1;c[2]=c2;}

  //! Assignment operator
  /*!
  \param other The other matrix that is assigned to this one
  \return A reference to this object after the assignment.
  */
  Matrix3x3<V>& operator=(const Matrix3x3<V>& other)
    {c[0] = other.c[0]; c[1] = other.c[1]; c[2] = other.c[2]; return *this;}

  //! Copy constructor
  /*!
  \param other The other matrix that is copied to this one
   */
  Matrix3x3<V>(const Matrix3x3<V>& other) {*this = other;}

   //! Multiplication of this matrix by vector.
  /*!
  \param vector The vector this one is multiplied by 
  \return A reference to a new vector containing the result
    of the calculation.
  */
  Vector3<V> operator*(const Vector3<V>& vector) const
  {
    return (c[0]*vector.x + c[1]*vector.y + c[2]*vector.z);
  }


  //! Multiplication of this matrix by another matrix.
  /*!
  \param other The other matrix this one is multiplied by 
  \return A reference to a new object containing the result
    of the calculation.
  */
  Matrix3x3<V> operator*(const Matrix3x3<V>& other) const
  {
    return Matrix3x3<V>((*this)*other.c[0], (*this)*other.c[1], (*this)*other.c[2]);
  }

  //! Multiplication of this matrix by another matrix.
  /*!
  \param other The other
  matrix this one is multiplied by 
  \return A reference this object after the calculation.
  */
  Matrix3x3<V> operator*=(const Matrix3x3<V>& other)
    {return *this = *this * other;}

  //! Multiplication of this matrix by a factor.
  /*!
  \param factor The factor this matrix is multiplied by 
  \return A reference to this object after the calculation.
  */
  Matrix3x3<V>& operator*=(const V& factor)
  {
    c[0] *= factor;
    c[1] *= factor;
    c[2] *= factor;
    return *this;
  }

  //! Division of this matrix by a factor.
  /*!
  \param factor The factor this matrix is divided by 
  \return A reference to this object after the calculation.
   */
  Matrix3x3<V>& operator/=(const V& factor)
    {return *this *= 1 / factor;}

  //! Multiplication of this matrix by a factor.
  /*!
  \param factor The factor this matrix is multiplied by 
  \return A new object that contains the result of the calculation.
  */
  Matrix3x3<V> operator*(const V& factor) const
    {return Matrix3x3<V>(*this) *= factor;}

  //! Division of this matrix by a factor.
  /*!
  \param factor The factor this matrix is divided by 
  \return A new object that contains the result of the calculation.
  */
  Matrix3x3<V> operator/(const V& factor) const
    {return Matrix3x3<V>(*this) /= factor;}

  //! Comparison of another matrix with this one.
  /*!
  \param other The other matrix that will be compared to this one
  \return Whether the two matrices are equal.
  */
  bool operator==(const Matrix3x3<V>& other) const
  {
    return (c[0]==other.c[0] && c[1]==other.c[1] && c[2]==other.c[2]);
  }


  //! Comparison of another matrix with this one.
  /*!
  \param other The other matrix that will be compared to this one
  \return Whether the two matrixs are unequal.
  */
  bool operator!=(const Matrix3x3<V>& other) const
    {return !(*this == other);}

  /*! Transpose the matrix
  \return A new object containing transposed matrix
  */
  Matrix3x3<V> transpose() const
  {
    return Matrix3x3<V>(Vector3<V>(c[0].x,c[1].x,c[2].x),
                     Vector3<V>(c[0].y,c[1].y,c[2].y),
                     Vector3<V>(c[0].z,c[1].z,c[2].z));
  }

  //! Calculation of the determinant of this matrix.
  /*! 
  \return The determinant.
  */
  V det() const 
    {return c[0].v[0] * (c[1].v[1] * c[2].v[2] - c[1].v[2] * c[2].v[1]) +
            c[0].v[1] * (c[1].v[2] * c[2].v[0] - c[1].v[0] * c[2].v[2]) +
            c[0].v[2] * (c[1].v[0] * c[2].v[1] - c[1].v[1] * c[2].v[0]);}
};

/**
* Streaming operator that reads a Matrix3x3<V> from a stream.
* @param stream The stream from which is read.
* @param matrix3x3 The Matrix3x3<V> object.
* @return The stream.
*/ 
template <class V> In& operator>>(In& stream, Matrix3x3<V>& matrix3x3);

/**
* Streaming operator that writes a Matrix3x3<V> to a stream.
* @param stream The stream to write on.
* @param matrix3x3 The Matrix3x3<V> object.
* @return The stream.
*/ 
template <class V> Out& operator<<(Out& stream, const Matrix3x3<V>& matrix3x3);


/** representation for 3x3 RotationMatrices*/
class RotationMatrix : public Matrix3x3<double> {
public:
  /** Default constructor. */
  RotationMatrix() {}

  //! Constructor.
  /*!
  \param c0 the first column of the matrix.
  \param c1 the second column of the matrix.
  \param c2 the third column of the matrix.
  */
  RotationMatrix(const Vector3<double>& c0,const Vector3<double>& c1,const Vector3<double>& c2) :
    Matrix3x3<double>(c0,c1,c2) {}

  //! Assignment operator
  /*!
  \param other The other matrix that is assigned to this one
  \return A reference to this object after the assignment.
  */
  RotationMatrix& operator=(const Matrix3x3<double>& other)
    {c[0] = other.c[0]; c[1] = other.c[1]; c[2] = other.c[2]; return *this;}

  //! Copy constructor
  /*!
  \param other The other matrix that is copied to this one
   */
  RotationMatrix(const Matrix3x3<double>& other) {*this = other;}

  /**
   * Rotationsmatrix from RPY-angles
   * roll dreht um z, pitch um y und yaw um x
   * R(roll,pitch,yaw)= R(z,roll)*R(y,pitch)*R(x,yaw)
   * analog to skript: "Robotik 1 Ausgabe Sommersemester 2001" by Prof. Dr. O. von Stryk
   * @attention: RPY-angles are not clearly defined!
   */
  RotationMatrix& fromKardanRPY(const double yaw, const double pitch, const double roll);

  /**
   * Invert the matrix
   * Inverted rotation matrix is transposed matrix
   */
  RotationMatrix invert() {return transpose();}

  /** Rotation around the x-axis.
  * \param angle The angle this pose will be rotated by
  * \return A reference to this object after the calculation.
  */
  RotationMatrix& rotateX(const double angle);

  /** Rotation around the y-axis.
  * \param angle The angle this pose will be rotated by
  * \return A reference to this object after the calculation.
  */
  RotationMatrix& rotateY(const double angle);

  /** Rotation around the z-axis.
  * \param angle The angle this pose will be rotated by
  * \return A reference to this object after the calculation.
  */
  RotationMatrix& rotateZ(const double angle);

  /**Get the x-angle of a RotationMatrix
  * \return The angle around the x-axis between the original and the rotated z-axis projected on the y-z-plane
  */
  double getXAngle() const;

  /**Get the y-angle of a RotationMatrix
  * \return The angle around the y-axis between the original and the rotated x-axis projected on the x-z-plane
  */
  double getYAngle() const;

  /**Get the z-angle of a RotationMatrix
  * \return The angle around the z-axis between the original and the rotated x-axis projected on the x-y-plane
  */
  double getZAngle() const;

  /** create and return a Rotationmatrix, rotatate around x-axis
  *\param angle 
  *\return rotated Rotationmatrix
  */
  static RotationMatrix getRotationX(const double angle){
    return RotationMatrix().rotateX(angle);
  }


  /** create and return a Rotationmatrix, rotatate around y-axis
  *\param angle 
  *\return rotated Rotationmatrix
  */
  static RotationMatrix getRotationY(const double angle){
    return RotationMatrix().rotateY(angle);
  }

  /** create and return a Rotationmatrix, rotatate around z-axis
  *\param angle 
  *\return rotated Rotationmatrix
  */
  static RotationMatrix getRotationZ(const double angle){
    return RotationMatrix().rotateZ(angle);
  }
};

/**
* Streaming operator that reads a RotationMatrix from a stream.
* @param stream The stream from which is read.
* @param rotationMatrix The RotationMatrix object.
* @return The stream.
*/ 
In& operator>>(In& stream, RotationMatrix& rotationMatrix);

/**
* Streaming operator that writes a RotationMatrix to a stream.
* @param stream The stream to write on.
* @param rotationMatrix The RotationMatrix object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const RotationMatrix& rotationMatrix);


#endif // __Matrix_h__

/*
* Change log :
* 
* $Log: Matrix.h,v $
* Revision 1.2  2003/12/02 13:44:55  cesarz
* added streaming operators
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.4  2002/11/19 17:38:31  dueffert
* doxygen bugs corrected
*
* Revision 1.3  2002/11/19 15:43:04  dueffert
* doxygen comments corrected
*
* Revision 1.2  2002/11/12 23:00:47  dueffert
* started restore greenhills compatibility
*
* Revision 1.1  2002/09/22 13:10:50  risler
* new Math headers added
*
*
*/
