/**
 * @file Histogram.h
 *
 * The file defines a struct to represent histograms.
 * 
 * @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef __Histogram_h__
#define __Histogram_h__

#include "Tools/Streams/InOut.h"
#include <string.h>
#include "Tools/Debugging/Debugging.h"
#include "Platform/GTAssert.h"

/**
 * A struct to represent histograms
 */
class Histogram
{
public:
  enum HistogramID
  {
    noID,
    imageIntensityY,
    imageIntensityU,
    imageIntensityV,
    scanLineIntensityY,
    scanLineIntensityU,
    scanLineIntensityV,
    colorFrequency,
    lengthOfSegments,
    numberOfHistogramIDs
  };

  static const char* getName(HistogramID histogramID)
  {
    switch(histogramID)
    {
    case noID: return "no histogram";
    case imageIntensityY: return "image intensity y";
    case imageIntensityU: return "image intensity u";
    case imageIntensityV: return "image intensity v";
    case scanLineIntensityY: return "scan line intensity y";
    case scanLineIntensityU: return "scan line intensity u";
    case scanLineIntensityV: return "scan line intensity v";
    case colorFrequency: return "color frequency";
    case lengthOfSegments: return "length of segments";
    default: return "Please edit Histogram::getName() !";
    }
  }
  
  static DebugKeyTable::debugKeyID getDebugKeyID(HistogramID histogramID)
  {
    switch(histogramID)
    {
    case imageIntensityY: return DebugKeyTable::sendHistogram_imageIntensityY;
    case imageIntensityU: return DebugKeyTable::sendHistogram_imageIntensityU;
    case imageIntensityV: return DebugKeyTable::sendHistogram_imageIntensityV;
    case scanLineIntensityY: return DebugKeyTable::sendHistogram_scanLineIntensityY;
    case scanLineIntensityU: return DebugKeyTable::sendHistogram_scanLineIntensityU;
    case scanLineIntensityV: return DebugKeyTable::sendHistogram_scanLineIntensityV;
    case colorFrequency: return DebugKeyTable::sendHistogram_colorFrequency;
    case lengthOfSegments: return DebugKeyTable::sendHistogram_lengthOfSegments;
    default: 
      {
        ASSERT(false);
        return (DebugKeyTable::debugKeyID)0;
      }
    }
  }


  Histogram() { histogramID = noID; init(); }
  Histogram(HistogramID id) { histogramID = id; init(); }

  void init();
  void init(int numberOfEntries);
  void add(int index);
  void setID(HistogramID id) { histogramID = id; }

  int getValue(int index);
  int getNumberOfEntries();

  HistogramID getHistogramID();

  Histogram& operator=(const Histogram& other)
  {
    memcpy(value, other.value, sizeof(value));
    numberOfEntries = other.numberOfEntries;
    histogramID = other.histogramID;
    sum = other.sum;
    numberOfAddedEntries = other.numberOfAddedEntries;
    return *this;
  }

  double getAverage();
  double getAverageFrequencyOverAllEntries();
  double getAverageFrequencyOverUsedEntries();

  void analyseClusters();
  int getNumberOfClusters();
  int getBeginOfCluster(int index);
  int getEndOfCluster(int index);

private:
  enum{maxNumberOfEntries = 256};
  int value[maxNumberOfEntries];
  int numberOfEntries;
  HistogramID histogramID;

  int numberOfAddedEntries;
  int sum;

  int numberOfClusters;
  int beginOfCluster[maxNumberOfEntries];
  int endOfCluster[maxNumberOfEntries];
};

/**
 * Streaming operator that reads a Histogram from a stream.
 * @param stream The stream from which is read.
 * @param histogram The Histogram object.
 * @return The stream.
 */ 
In& operator>>(In& stream, Histogram& histogram);
 
/**
 * Streaming operator that writes a Histogram to a stream.
 * @param stream The stream to write on.
 * @param histogram The Histogram object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, Histogram& histogram);


#endif // __Histogram_h__

/*
 * Change log:
 * 
 * $Log: Histogram.h,v $
 * Revision 1.2  2003/10/29 13:15:02  juengel
 * "colorFrequency" added.
 *
 * Revision 1.1  2003/10/07 10:13:24  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/28 09:28:48  juengel
 * Comments corrected.
 *
 * Revision 1.2  2003/09/26 15:28:10  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1  2003/09/26 11:40:40  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.2  2003/09/02 11:15:35  dueffert
 * doxygen docu corrected
 *
 * Revision 1.1  2003/08/25 17:19:39  juengel
 * Added Histogram
 *
*/
