/**
 * @file Tools/Math/GaussBell.h
 * Declaration of class GaussBell.
 *
 * @author <a href=mailto:Kai_Engel@gmx.de>Kai Engel</a>
 */

#ifndef __GaussBell_h_
#define __GaussBell_h_

#include "Tools/Math/Matrix2x2.h"
#include "Tools/Math/Vector2.h"
#include "Tools/Debugging/DebugDrawings.h"

/**
  * @class GaussBell 
  * This class contains attributes and methods which are necessary for the fusion 
  * of Gauss-Bells. The percepts from differnt robots will be interpreted as 
  * values which defines special Gauss-Bells. 
  *
  * @author <A href=mailto:kai_engel@gmx.de>Kai Engel</A>
  */
class GaussBell 
{
  public:
    
    /** 
      * Constructor.
      */
    GaussBell(); 
    ~GaussBell();

    //###############################################################################################
    //# --- Get-Methods:                                                                            #
    //###############################################################################################
    Matrix2x2<double> getCovarianceMatrix();/**< Returns the covariance matrix of the 
                                                 Gaussbell. */
    Vector2<double> getPositionOfMaximum();/**< Returns the position of the GaussBell-Maximum */
    double getValidity(Vector2<double> _position);/**< Returns the validity-value at the position given in the argument*/
    double getValidityAtPositionOfMaximum();
    unsigned int getTimeStamp();
    void getSigmas(double& sigmaMaj, double& sigmaMin); /**< Returns the sigmas */
    double getAngle(); /**< Returns the angle */

    //###############################################################################################
    //# --- Set-Methods:                                                                            #
    //###############################################################################################
    void setCovarianceMatrix(double _validity, double _angle);/**< Sets the covarianceMatrix by the percepts of a player */
    void setCovarianceMatrix(double _validity, double _deltaX, double _deltaY); /**< like the previous method aber faster! */
    void setCovarianceMatrix(Matrix2x2<double> _covarianceMatrix);/**< Sets the covarianceMatrix by another Matrix */
    void setCovarianceMatrix(double _sigmaMaj, double _sigmaMin, double _deltaX, double _deltaY); /** when the sigmas are given but no validity
                                                                   This method will be used for the SensorFusionPlayersLocator */ 
    void setPositionOfMaximum(Vector2<double> _positionOfMaximum);/**< Sets the position of one object one robot has percepted*/
    void setRotationAngle(double _rotationAngle);
    void setValidity(double _validity);
    void setSigmas(double _sigmaMaj, double _sigmaMin);

    void setTimeStamp(unsigned long _timeStampGaussBell);
    void setTimeStamp(); /**< set the time stamp to the actual system time */
    
    void updateValidity(double _validity); /**< if a older percept should be merged the validity has to be decreased.
                                                This will be used in SensorFusionBallLocator. */

    //###############################################################################################
    //# --- other Methods:                                                                          #
    //###############################################################################################
    void mergeBells(GaussBell _gaussBell1, GaussBell _gaussBell2); /**< merges the bells _gaussBell1 and _gaussBell2 */

  private:
	  Matrix2x2<double> covarianceMatrix;
    Vector2<double> position; // double because of calculating the merged position
                              
    double  sigmaMaj, /**< variance in maj. direction */
            sigmaMin; /**< variance in min. direction */
    double rotationAngle;
    double validity;  /**< the validity of the percept */

    static const int phi[330]; /**< phi-table of the normalized gauss distribution (index: factor 100, array: factor 10000) */
    static const int invProb[1000]; /**< inverted probability table of the normalized gauss distribution (index: factor 1000, array: factor 100) */
    static const double radiusMaj; /**< radius in maj-direction of the validity-ellipse.  the validity is the probability that the real position lies within this ellipe*/
    static const double radiusMin; /**< radius in min-direction of the validity-ellipse */
    static const double factor;

	  void transformSigmaToValidity();
    void transformValidityToSigma(); /**< calculates sigmaMaj and sigmaMin when only the validity is given */
    void updateSigmasAndAngle();     /**< calculates sigmaMaj, sigamMin and rotationAngle form the covariance matrix */
	  void updateCovarianceMatrix();   /**< calculates the CovarianceMatrix from sigmaMaj, sigmaMin and the rotationAngle */
       
    unsigned long timeStampGaussBell; /**< The time when this GaussBell was active. 
                                           This attribut will be needed for merging history-Data into the fusion. */
};


#endif// __GaussBell_h_

/*
 * Changelog:
 * 
 * $Log: GaussBell.h,v $
 * Revision 1.1  2003/10/07 10:13:24  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/28 09:28:48  juengel
 * Comments corrected.
 *
 * Revision 1.1  2003/09/26 11:40:40  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.3  2003/05/22 06:48:44  dueffert
 * doxygen docu corrected
 *
 * Revision 1.2  2003/05/19 15:35:01  wachter
 * Moved genration of debug-drawing
 *
 * Revision 1.1  2003/05/19 09:17:30  wachter
 * Renamed SensorFusionGaussBell to GaussBell as requested.
 *
 * Revision 1.8  2003/04/25 11:32:03  engel
 * Coments inserted
 *
 * Revision 1.7  2003/04/06 15:57:52  engel
 * Little changes (cosmetic and some improvements).
 *
 * Revision 1.6  2003/04/01 15:57:52  wachter
 * Fix of transformValidityToSigma
 *
 * Revision 1.5  2003/03/31 14:03:06  wachter
 * Cleanup of SensorFusionGaussBell,
 * added "real" Validity to SensorFusionGaussBell
 *
 * Revision 1.4  2003/03/21 15:11:52  engel
 * fixed crash for dx=dy=0
 * inserted getValitityAtPositionOfMaximum()
 *
 * Revision 1.3  2003/02/25 16:51:16  engel
 * some changes to calculate the validity (mean-Value)
 *
 * Revision 1.2  2003/02/25 11:43:44  dueffert
 * line feed and doxygen bugs fixed
 *
 * Revision 1.1  2003/02/14 14:34:02  wachter
 * Added SensorFusionBallLocator
 *
 *
 */
