/**
* @file FourierCoefficient.h
* 
* Declaration of class FourierCoefficient
*
* @author <a href="mailto:jhoffman@informatik.hu-berlin.de">Jan Hoffmann</a>, Matthias Jngel
*/ 

#ifndef __FourierCoefficient_h_
#define __FourierCoefficient_h_

#include "Tools/Streams/InOut.h"
#include "Representations/Motion/JointData.h"


/**
* This class contains a set of fourier coefficients as well as
* methods transform them from and to a period of function values
*/
class FourierCoefficient
{
public:
  enum {numOfCoeffs = 80};

  /** Constructor */ 
  FourierCoefficient();

  /** Destructor */ 
  ~FourierCoefficient();

  /**
  * Synthesizes the value of the original function for a given joint
  * using the fourier coefficients. A scaling factor is added. This allows
  * the amplitude of the curve to be changed while keeping the origin/offset constant.
  * This factor is also used for inverting the direction of the function (in essence
  * playing the function backward if the factor is smaller than 0)
  * @param joint The joint for which the synthesis is calculated.
  * @param time Time (in the format returned by "getSystemTime"). This time 
  *   is used to generate the curves as a function of that time. 
  * @param cMax Number of coefficients used. Small numbers result in a low 
  *   pass behavior, numbers larger than numOfCoeffs/2 result in 
  *   oversampling effects. (This is due to the fact that a continous Fourier 
  *   (back) transformation is calculated from descrete FT coefficients.)
  * @param scalingFactor is used to vary the amplitude of the wave without
  *   touching the offset. 1.0 = 100%,  values greater than
  *   1 possible. Negative numbers will be interpreted as running the function 
  *   backwards (negative time line)
  * @return The synthesized function value.
  */ 
  long fourierSynth(
    JointData::JointID joint, 
    unsigned long time, 
    int cMax, 
    double scalingFactor=1.0);

  /**
  * Like the other fourierSynth, a change to the joint parameter
  * and no return value. In contrast to the above, a reference to the
  * entire joint data array is passed in order to minimize function 
  * calls. All joint values are computed in one go.
  * 
  * Also scalingfactor is no longer used. don't need it, really.
  * 
  * @param jointData Array of joint datas to be calculated.
  * @param time This time is used to generate the curves as a function of that time.
  * @param cMax Number of coefficients used.
  * @return nothing
  */ 
  long fourierSynth(
    JointData* jointData, 
    unsigned long time, 
    int cMax); 

  /**
  * Calculate LUT for the function values so they don't have
  * to be computed every time the synth. is used
  * @param joint The joint that is described by the function
  * @param cMax Number of coefficients used. 
  * @return true if successful, false if failed
  */ 
  bool calcLUT(JointData::JointID joint, int cMax);

  /**
  * Calculate all LUTs
  * @param cMax Number of coefficients used. 
  * @return true if successful, false if failed
  */ 
  bool calcAllLUTs(int cMax);


  /**
  * Calculate coefficients from the function values over
  * one period.
  * @param functionValues Pointer to an array containing the 
  *   function values
  * @param joint The joint that is described by the function
  * @return true if successful, false if failed
  */ 
  bool calculate(long* functionValues, JointData::JointID joint);

  /**
  * Calculates phi and r for the first numOfC coefficients only
  * Calculates the coefficients in the exponential form, i.e. 
  * z = R * exp (i*phi), from the "vector" representation 
  * z = x + iy
  *
  * @param numOfC number of coefficients
  * @return true if successful, false if failed
  */ 
  bool calcPhiAndR(int numOfC=numOfCoeffs);

  /**
  * Calculates the coefficients in the vector form, i.e. 
  * z = x + iy, from the exponential representation 
  * z = R * exp (i*phi)
  * Calculates the real and imaginary parts for the first 
  * numOfC coefficients
  *
  * @param numOfC
  * @return true if successful, false if failed
  */ 
  bool calcReAndIm(int numOfC=numOfCoeffs);

  /**
  * Takes 1+3=4 sets of Fourier coefficients and averages them
  * according to the weights. negative weights are interpreted
  * relativly to fc0, a weight=1 means: go from fc0 100% into
  * direction fcn, so fc=fc0+w1*(fc1-fc0)+w2*(fc2-fc0)+...
  * Takes an parameter that determines the accuracy of
  * the merge, i.e. the number of coefficients that are merged
  * @param fc0 Fourier Coefficient set 0 ("stand" would be useful)
  * @param fc1 Fourier Coefficient set 1
  * @param w1 the weigth of set 1
  * @param fc2 Fourier Coefficient set 2
  * @param w2 the weigth of set 2
  * @param fc3 Fourier Coefficient set 3
  * @param w3 the weigth of set 3
  * @param numOfC the number of coefficients (starting from the first)
  *   that should be merged
  * @return true if successful, false if failed
  */ 
  bool merge(FourierCoefficient *fc0, FourierCoefficient *fc1, double w1, FourierCoefficient *fc2, double w2, FourierCoefficient *fc3, double w3, int numOfC);

  /**
  * Similar to merge(many parameters) but only merges two sets!
  * @param fc1 Fourier Coefficient set 1
  * @param w1 the weigth of set 1
  * @param fc2 Fourier Coefficient set 2
  * @param w2 the weigth of set 2
  * @param numOfC the number of coefficients (starting from the first)
  *   that should be merged
  * @return true if successful, false if failed
  */ 
  bool merge(FourierCoefficient *fc1, double w1, FourierCoefficient *fc2, double w2, int numOfC);

  /**
  * Function shifts all phases so that the 1 Hz frequency
  * of the joint is a sine function starting at t=0 (i.e.
  * there is no cosine part).This is necessary when using 
  * different function since they usually are not
  * coherent. Coherence may be desirable in order to merge
  * or superimpose two given functions.
  *
  * @param joint reference joint (it is examined 
  * @return phaseShift how much the current function is 
  * out of phase with respect to a 1Hz sine
  */ 
  double phaseAlign(JointData::JointID joint);

  /** length of period these coefficients are designed for */
  int lengthOfPeriod;

  /** real and imaginary parts of the coefficients*/
  double real[JointData::numOfJoint][numOfCoeffs];
  double imaginary[JointData::numOfJoint][numOfCoeffs];

  /** alternative representation: 
  * length r and phase phi of the coefficients*/
  double phi[JointData::numOfJoint][numOfCoeffs];
  double r[JointData::numOfJoint][numOfCoeffs];

  /** LUT that stores the function calculated from the coeffs */ 
  double functionLUT[JointData::numOfJoint][numOfCoeffs];

  bool useLookUp[numOfCoeffs];
};

/**
 * Streaming operator that reads FourierCoefficients from a stream.
 * @param stream The stream from which is read.
 * @param fc The GT2001WalkingParameters object.
 * @return The stream.
 */ 
In& operator>>(In& stream,FourierCoefficient& fc);
 
/**
 * Streaming operator that writes FourierCoefficients to a stream.
 * @param stream The stream to write on.
 * @param fc The FourierCoefficients object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, FourierCoefficient& fc);

#endif //__FourierCoefficient_h_

/*
* Change log :
* 
* $Log: FourierCoefficient.h,v $
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:28:10  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1  2003/09/26 11:40:40  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.13  2003/02/02 23:30:30  jhoffman
* added merge for only two types of motion and more debugging
* and control stuff (mostly needed for performance testing)
*
* Revision 1.12  2003/01/22 12:16:21  dueffert
* doxygen docu corrected
*
* Revision 1.11  2003/01/21 14:58:39  jhoffman
* increased performance by precalculating sin/cos
*
* Revision 1.10  2003/01/18 08:26:35  jhoffman
* changes in order to improve performance
*
* Revision 1.9  2002/12/10 16:07:05  dueffert
* Fourier working now
*
* Revision 1.8  2002/12/09 15:24:06  dueffert
* some optimisation
*
* Revision 1.7  2002/11/26 12:25:57  dueffert
* doxygen docu added
*
* Revision 1.6  2002/11/22 13:42:08  dueffert
* cleanup
*
* Revision 1.5  2002/11/22 13:41:21  loetzsch
* - removed the FourierCoefficient::loadLegs and ::saveLegs functions
*   (streaming operators are now used)
* - .fcb files have text format now
* . moved the .fcb files from /Config to /Config/Fourier
*
* Revision 1.4  2002/11/19 17:14:14  risler
* coding conventions: renamed JointData::joint to JointID, GetName to getName
*
* Revision 1.3  2002/10/14 13:14:24  dueffert
* doxygen comments corrected
*
* Revision 1.2  2002/09/22 18:40:54  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:26:40  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/07/23 13:32:57  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.8  2002/04/25 14:50:37  kallnik
* changed double/float to double
* added several #include GTMath
*
* PLEASE use double
*
* Revision 1.7  2002/04/23 15:00:06  jhoffman
* changes and additions
*
* Revision 1.6  2002/04/03 16:44:31  jhoffman
* added "stabilizeRobot" to motionControl (which is turned off as a default)
*
* Revision 1.5  2002/03/19 16:45:43  jhoffman
* no message
*
* Revision 1.4  2002/03/19 13:23:50  jhoffman
* no message
*
* Revision 1.3  2002/03/19 12:11:14  jhoffman
* extended functionality (synth, load, save, ...)
*
* Revision 1.2  2002/03/12 14:10:48  jhoffman
* added fourier synthesis functionality which calculates the FT from the coefficients
*
* Revision 1.1  2002/01/26 20:22:52  juengel
* no message
*
*
*/
