/**
* @file TacticChooser.h
* 
* This file contains a generic class for Tactic Choosing.
* @author <A href=mailto:jochen@kerdels.de>Jochen Kerdels</A>
*/

#ifndef __TacticChooser_h_
#define __TacticChooser_h_

#include "Modules/BehaviorControl/BehaviorControl.h"
#include "CollectedBeliefs.h"
#include "RateableOptions.h"
#include "GlobalAnalyser.h"
#include "ChooserRaterInterfaces.h"

#include "Modules/BehaviorControl/MSH2004BehaviorControl/MSH2004DTT/OffDefAnalyser.h"
#include "Modules/BehaviorControl/MSH2004BehaviorControl/MSH2004DTT/GoalieCoach.h"
#include "Modules/BehaviorControl/MSH2004BehaviorControl/MSH2004DTT/SpecialEventAnalyser.h"

/**
* @class TacticChooser 
*
* The Base class for tactic choosing
*
* It is the task of tactic choosing to choose one option from all avaible options.
*/
class TacticChooser :  public BehaviorControlInterfaces
{
protected:

	CollectedBeliefs& collectedBeliefs;

	// TacticChooser engine(s)
  GlobalAnalyser* globalAnalyser[RateableOptions::numOfGlobalAnalysers];

  /**
  * Structure to communicate between TacticChooser und OptionRater 
  */
  ChooserRaterInterfaces& chooserRaterInterfaces;
  
public:

  /**
   * Constructor.
   * @param interfaces The parameters of the OptionRating module.
   * @param collectedBeliefs The collected-Beliefs to work on.
   * @param chooserRaterInterfaces We want to talk to the Option-Rating.

   */
  TacticChooser(const BehaviorControlInterfaces& interfaces,
                CollectedBeliefs& collectedBeliefs,
                ChooserRaterInterfaces& chooserRaterInterfaces)
    : BehaviorControlInterfaces(interfaces),
		  collectedBeliefs(collectedBeliefs),
      chooserRaterInterfaces(chooserRaterInterfaces)
  {
		// Global Analysers registrieren ..

	  // Hier die OffDefAnalyser Engine..
	  globalAnalyser[RateableOptions::offDefAnalyser] =
		  new OffDefAnalyser(interfaces);
	  
	  // Hier den GoalieCoach..
	  globalAnalyser[RateableOptions::goalieCoach] =
		  new GoalieCoach(interfaces);

	  // Hier den SpecialEventAnalyser..
	  globalAnalyser[RateableOptions::specialEventAnalyser] =
		  new SpecialEventAnalyser(interfaces);

	}
  
  /** Destructor */
  virtual ~TacticChooser()
	{
		// Global Analysers freigeben ..
	  delete(globalAnalyser[RateableOptions::offDefAnalyser]);
	  delete(globalAnalyser[RateableOptions::goalieCoach]);
	  delete(globalAnalyser[RateableOptions::specialEventAnalyser]);
	}


	/** This method should be implemented by descendants */
	virtual RateableOptions::OptionID chooseOption() = 0;

  /** 
  * Is called for every incoming debug message.
  * @param message An interface to read the message from the queue
  * @return if the messag was read
  */
  virtual bool handleMessage(InMessage& message) {return false;}

};

#endif // __TacticChooser_h_

/*
 * Change log :
 * $Log: TacticChooser.h,v $
 * Revision 1.9  2004/03/08 01:07:20  roefer
 * Interfaces should be const
 *
 * Revision 1.8  2004/03/02 22:28:47  huelsbusch
 * Added missing doxygen comments
 *
 * Revision 1.7  2004/02/19 23:34:59  kerdels
 * added SpecialEventAnalyser
 *
 * Revision 1.6  2004/02/03 23:47:33  kerdels
 * added GoalieCoach and adjusted tactics.dat.
 *
 * Revision 1.5  2004/01/29 21:33:28  kerdels
 * added turn-for-ball Option to DTT,
 * added link between TacticChooser and OptionRating
 *
 * Revision 1.4  2004/01/07 14:56:09  kerdels
 * added send functionality to TacticDesigner
 *
 * Revision 1.3  2004/01/04 01:00:47  kerdels
 * TacticDesigner hinzugefgt
 *
 * Revision 1.2  2003/12/29 12:26:27  kerdels
 * global analysis hinzugefgt
 *
 * Revision 1.1  2003/12/11 18:28:21  kerdels
 * DTT Framework fertiggestellt.
 *
 *
 */
