/**
 * @file RateableOptions.h
 *
 * Declaration of class RateableOptions
 * @author <A href=mailto:jochen@kerdels.de>Jochen Kerdels</A>
 */

#ifndef __RateableOptions_h_
#define __RateableOptions_h_

#include "../../Modules/BehaviorControl/MSH2004BehaviorControl/OpenChallengeDefines.h"

#include <string.h>

 /**
 *  The class contains an enumeration of rateable options
 */
class RateableOptions
{
public:

  #ifndef OPENCHALLENGE_DEFINES
  /**
   * Enumerator for available options
   * Add new options here.
   */
  enum OptionID
  {
    noOption = 0,                  //!<  to disable DynamicTactics and use std. Xabsl 
    doNothing,                     //!<  doesn't change the motion request 
    stand,                         //!<  standing around... 
   	keepOption,                    //!<  keeps previous choosen option ...
    goToFrontLeft,                 //!<  robot goes to FrontLeft    (ca.  1400, 900) position 
    goToFrontPenLeft,              //!<  robot goes to FrontPenLeft (ca.  1500, 600) position 
    goToFrontMiddle,               //!<  robot goes to FrontMiddle  (ca.  1400,   0) position 
    goToFrontPenRight,             //!<  robot goes to FrontPenright (ca.  1500, -600) position 
    goToFrontRight,                //!< robot goes to FrontRight   (ca.  1400,-900) position 
    goToMiddleLeft,                //!< robot goes to MiddleLeft   (ca.     0, 900) position 
    goToMiddle,                    //!< robot goes to Middle       (ca.     0,   0) position 
    goToMiddleRight,               //!< robot goes to MiddleRight  (ca.     0,-900) position 
    goToBehindLeft,                //!< robot goes to BehindLeft   (ca.-1400, 900) position 
    goToBehindMiddle,              //!< robot goes to BehindMiddle (ca.-1000,   0) position 
    goToBehindRight,               //!< robot goes to BehindRight  (ca.-1400,-900) position 
    goToCatenaccioLeft,            //!<  robot goes to CatenaccioLeft    (ca. -1500, 600) position 
    goToCatenaccioMiddle,          //!<  robot goes to CatenaccioMiddle  (ca. -1400,   0) position 
    goToCatenaccioRight,           //!<  robot goes to CatenaccioRight   (ca. -1500,-600) position 
    approachCloseBall,             //!< approaches close ball 
    goToBall,                      //!< goes to the ball 
    goToBallFrontLeft,             //!< goes to Ball if FrontLeft    (ca.  1400, 900) position 
    goToBallFrontMiddle,           //!< goes to Ball if FrontMiddle  (ca.  1400,   0) position 
    goToBallFrontRight,            //!< goes to Ball if FrontRight   (ca.  1400,-900) position 
    goToBallMiddleLeft,            //!< goes to Ball if MiddleLeft   (ca.     0, 900) position 
    goToBallMiddle,                //!< goes to Ball if Middle       (ca.     0,   0) position 
    goToBallMiddleRight,           //!< goes to Ball if MiddleRight  (ca.     0,-900) position 
    goToBallBehindLeft,            //!< goes to Ball if BehindLeft   (ca.-1400, 900) position 
    goToBallBehindMiddle,          //!< goes to Ball if BehindMiddle (ca.-1000,   0) position 
    goToBallBehindRight,           //!< goes to Ball if BehindRight  (ca.-1400,-900) position 
    goToBallGoalie,                //!< goes to the ball (goaliemodified)
    goIntoOwnGoal,                 //!< robot goes into own goal (ca.-2100,   0) position 
    doJumpDefend,                  //!< robot jumps with both arms stretched 
    doJumpLeft,                    //!< robot jumps to left side 
    doJumpRight,                   //!< robot jumps to right side 
    kickToGoal,                    //!< robot kicks ball towards opponent goal 
    kickToGoalFromFrontLeft,       //!< robot kicks ball towards opponent goal from FrontLeft Position
    kickToGoalFromFrontMiddle,     //!< robot kicks ball towards opponent goal from FrontMiddle Position
    kickToGoalFromFrontRight,      //!< robot kicks ball towards opponent goal from FrontRight Position
    kickToGoalFromMiddleLeft,      //!< robot kicks ball towards opponent goal from MiddleLeft Position
    kickToGoalFromMiddle,          //!< robot kicks ball towards opponent goal from Middle Position
    kickToGoalFromMiddleRight,     //!< robot kicks ball towards opponent goal from MiddleRight Position
    kickToGoalFromBehindLeft,      //!< robot kicks ball towards opponent goal from BehindLeft Position
    kickToGoalFromBehindMiddle,    //!< robot kicks ball towards opponent goal from BehindMiddle Position
    kickToGoalFromBehindRight,     //!< robot kicks ball towards opponent goal from BehindRight Position
    kickToTeammate1,               //!< robot kicks ball towards teammate1 
    kickToTeammate2,               //!< robot kicks ball towards teammate2 
    kickToTeammate3,               //!< robot kicks ball towards teammate3 
    kickToClear,                   //!< robot kicks ball to front 
    kickToClearFromFrontLeft,      //!< robot kicks ball towards front FrontLeft Position
    kickToClearFromFrontMiddle,    //!< robot kicks ball towards front from FrontMiddle Position
    kickToClearFromFrontRight,     //!< robot kicks ball towards front from FrontRight Position
    kickToClearFromMiddleLeft,     //!< robot kicks ball towards front from MiddleLeft Position
    kickToClearFromMiddle,         //!< robot kicks ball towards front from Middle Position
    kickToClearFromMiddleRight,    //!< robot kicks ball towards front from MiddleRight Position
    kickToClearFromBehindLeft,     //!< robot kicks ball towards front from BehindLeft Position
    kickToClearFromBehindMiddle,   //!< robot kicks ball towards front from BehindMiddle Position
    kickToClearFromBehindRight,    //!< robot kicks ball towards front from BehindRight Position
    kickToClearGoalie,             //!< robot kicks ball towards front from own penalty area
    intercept1,                    //!< positions the robot between enemy1 and own goal 
    intercept2,                    //!< positions the robot between enemy2 and own goal 
    intercept3,                    //!< positions the robot between enemy3 and own goal 
    intercept4,                    //!< positions the robot between enemy4 and own goal 
    handleBallLeftBorder,          //!< handles the ball when it's at the left border 
    handleBallRightBorder,         //!< handles the ball when it's at the right border 
    dribbleToFront,                //!< dribbles the ball 70cm to front 
    goToInterceptBall,             //!< robot goes to position halfway between ball and own goal 
   	bananaCrossFromLeft,           //!< robot kicks ball from left towards the penalty point
	  bananaCrossFromRight,          //!< robot kicks ball from right towards the penalty point
    doRollLeft,                    //!< robot rolls to Left
    doRollRight,                   //!< robot rolls to Right
    turnForBall,                   //!< robot turns until ball is ahead
    goToGoaliePos,                 //!< positions the robot in front of the own goal, between the goal and the ball
    
    backUp,                        //!< robot goes some steps back to prevent colliding with teammate 

    waveLeft,                      //!< robot waves with left forepaw 
    waveRight,                     //!< robot waves with right forepaw 

    newKickToGoal,                 //!< robot kicks ball towards opponent goal using kickengine
    newKickToGoalFromFrontLeft,    //!< robot kicks ball towards opponent goal from FrontLeft Position
    newKickToGoalFromFrontMiddle,  //!< robot kicks ball towards opponent goal from FrontMiddle Position
    newKickToGoalFromFrontRight,   //!< robot kicks ball towards opponent goal from FrontRight Position
    newKickToGoalFromMiddleLeft,   //!< robot kicks ball towards opponent goal from MiddleLeft Position
    newKickToGoalFromMiddle,       //!< robot kicks ball towards opponent goal from Middle Position
    newKickToGoalFromMiddleRight,  //!< robot kicks ball towards opponent goal from MiddleRight Position
    newKickToGoalFromBehindLeft,   //!< robot kicks ball towards opponent goal from BehindLeft Position
    newKickToGoalFromBehindMiddle, //!< robot kicks ball towards opponent goal from BehindMiddle Position
    newKickToGoalFromBehindRight,  //!< robot kicks ball towards opponent goal from BehindRight Position
    newKickToTeammate1,            //!< robot kicks ball towards teammate1 using kickengine 
    newKickToTeammate2,            //!< robot kicks ball towards teammate2 using kickengine 
    newKickToTeammate3,            //!< robot kicks ball towards teammate3 using kickengine 
    newKickToClear,                //!< robot kicks ball to front using kickengine 
    newKickToClearFromFrontLeft,   //!< robot kicks ball towards opponent goal from FrontLeft Position
    newKickToClearFromFrontMiddle, //!< robot kicks ball towards opponent goal from FrontMiddle Position
    newKickToClearFromFrontRight,  //!< robot kicks ball towards opponent goal from FrontRight Position
    newKickToClearFromMiddleLeft,  //!< robot kicks ball towards opponent goal from MiddleLeft Position
    newKickToClearFromMiddle,      //!< robot kicks ball towards opponent goal from Middle Position
    newKickToClearFromMiddleRight, //!< robot kicks ball towards opponent goal from MiddleRight Position
    newKickToClearFromBehindLeft,  //!< robot kicks ball towards opponent goal from BehindLeft Position
    newKickToClearFromBehindMiddle,//!< robot kicks ball towards opponent goal from BehindMiddle Position
    newKickToClearFromBehindRight, //!< robot kicks ball towards opponent goal from BehindRight Position
    newKickToClearGoalie,          //!< robot kicks ball towards opponent goal from own penalty area

    goToFrontAtBall,               //!< robot goes somewhere in the front in parallel to the ball 
    goToMiddleAtBall,              //!< robot goes somewhere in the middle + 50cm in parallel to the ball 
    goToBehindAtBall,              //!< robot goes somewhere in the behind in parallel to the ball 
    goToRightAtBall,               //!< robot goes somewhere in the right in parallel to the ball 
    goToLeftAtBall,                //!< robot goes somewhere in the left in parallel to the ball 

    numOfOptions                   //!< Number of options 
  };
#else 
  enum OptionID
  {
    noOption = 0,                  //!<  to disable DynamicTactics and use std. Xabsl 
    doNothing,                     //!<  doesn't change the motion request 
    stand,                         //!<  standing around... 
   	keepOption,                    //!<  keeps previous choosen option ...
    intro,                         
    extro,                         
    finished,                         
    gotoBitePos1,
    gotoBitePos2,
    gotoBitePos3,
    gotoBitePos4,
    bitePos1,
    bitePos2,
    bitePos3,
    bitePos4,
    movePos1,
    movePos2,
    movePos3,
    movePos4,
    gotoBridge,
    climbBridge,
    waitForBiteDogs,
    moveBridge,
    numOfOptions                   //!< Number of options 
  };
#endif

  /**
   * Enumerator for available options types ... NOT classes
   * This is used for the xabsl-interface
   */
  enum OptionType {
		special,
		gotoPos,
		gotoBallPos,
		kickAction,
    goalieAction,
    notype
  };


#ifndef OPENCHALLENGE_DEFINES
  /**
   * Enumerator for available options classes
   * Add new option classes here.
   */
  enum OptionClassID
  {
    goToAnyFront = numOfOptions, //!< robot goes front 
    goToAnyMiddle,               //!< robot goes middle 
    goToAnyBehind,               //!< robot goes behind 
    kickToTeammate,              //!< robot kicks the ball to teammates 
    intercept,                   //!< positions the robot between an enemy and the own goal 
    goToAnyLeft,                 //!< robot goes left 
    goToAnyRight,                //!< robot goes right 
    goToBallAnyLeft,             //!< robot goes to ball left 
    goToBallAnyRight,            //!< robot goes to ball right
    newKickToTeammate,           //!< robot kicks the ball to teammates with kickengine
    maxOptionClassID,
    numOfOptionClasses = maxOptionClassID - numOfOptions //!< Number of optionclasses 
  };
#else 
  enum OptionClassID
  {
    doSomething = numOfOptions,
    maxOptionClassID,
    numOfOptionClasses = maxOptionClassID - numOfOptions //!< Number of optionclasses 
  };
#endif

  typedef struct {
    int        id;
    char      *name;
    int        numOfSubOptions;
    int        changeThreshold;
    OptionType optionType;
    int        slidingAverageRange;
    int        ratingBoost;
  } OptionInfo;
  
  static OptionInfo optionInfos[];
 
  /**
   * Enumerator for available TacticEntryTypes used by GlobalAnalysers
   * Add new engines here.
   */
  enum TacticEntryTypeID {

    FirstTETID = 0,

    // Types for OffDefAnalyser
    offDef_Neutral,               // every analyser should offer a neutral type
    offDef_Offensive,
    offDef_Defensive,
    offDef_Max,

    // Types for GoalieCoach
    gc_Neutral,
    gc_ClearBall,
    gc_StayInGoal,
    gc_Max,

    // Types for SpecialEventAnalyser
    spEv_Neutral,
    spEv_NothingSpecial,
    spEv_NoBall,
    spEv_FourPlayersLeft,
    spEv_ThreePlayersLeft,
    spEv_TwoPlayersLeft,
    spEv_OnePlayerLeft,
    spEv_Konter,
    spEv_Max,

    numOfTETIDs
  };

  
  /**
   * returns names for option and option class ids
   */
  static const char* getOptionName(int id);
  static int         getOptionID(const char*  name);
  static OptionType  getOptionType(int id);
  static int         getSlidingAverageRange(int id);
  static int         getRatingBoost(int id);

  /**
   * The following two functions represent the members
   * of a OptionClass.. kind of a hack, i know..
   */
  static int         getNumberOfClassMembers(OptionClassID id);
  static int         getOptionClassChangeThreshold(OptionClassID id);
  static OptionID    getClassMember(OptionClassID id, int idx);
  
  static bool        isOptionClass(int id);
  static bool        isClassMember(OptionClassID OCid, OptionID Oid);

  /**
   * Enumerator for available OptionRating engines
   * Add new engines here.
   */
  enum OptionRatingID
  {
    defaultOptionRating = 0, //!< default engine
    numOfOptionRatings       //!< Number of engines
  };

  /**
   * Enumerator for available TacticChooser engines
   * Add new engines here.
   */
  enum TacticChooserID
  {
    defaultTacticChooser = 0, //!< default engine 
    numOfTacticChoosers       //!< Number of engines 
  };



  /**
   * Enumerator for available GlobalAnalyser engines
   * Add new engines here.
   */
  enum GlobalAnalyserID
  {
    offDefAnalyser = 0,      //!< offensive / defensive analysis engine 
    goalieCoach,             //!< goalie coach analysis engine
    specialEventAnalyser,    //!< special situations analysis engine
    numOfGlobalAnalysers     //!< Number of engines 
  };


  /**
   * returns names of a TETID
   * Add new IDs here
   */
  static const char* getTETIDName(TacticEntryTypeID id)
  {
    switch (id) {
      case offDef_Neutral        : return "Neutral";
      case offDef_Offensive      : return "Offensive";
      case offDef_Defensive      : return "Defensive";
      case gc_Neutral            : return "Neutral";
      case gc_ClearBall          : return "Clear ball";
      case gc_StayInGoal         : return "Stay in goal";
      case spEv_Neutral          : return "Neutral";
      case spEv_NothingSpecial   : return "Nothing special";
      case spEv_NoBall           : return "No ball";
      case spEv_FourPlayersLeft  : return "Four Players Left";
      case spEv_ThreePlayersLeft : return "Three Players Left";
      case spEv_TwoPlayersLeft   : return "Two Players Left";
      case spEv_OnePlayerLeft    : return "One Player Left";
      case spEv_Konter           : return "Konter";
      default: return "please edit RateableOptions::getTETIDName";
    }
  }

  /** 
   * returns names of GlobalAnalysers engines
   * Add new engines here
   */
  static const char* getGlobalAnalyserName(GlobalAnalyserID id)
  {
    switch (id)
    {
      case offDefAnalyser       : return "OffDefAnalyser";
      case goalieCoach          : return "GoalieCoach";
      case specialEventAnalyser : return "SpecialEventAnalyser";

      default: return "please edit RateableOptions::getGlobalAnalyserName";
    }
  }

  /** returns the base index of TETIDs of a GlobalAnalysers engine
   * Add new engines here
   */
  static int getGlobalAnalyserTETBase(GlobalAnalyserID id)
  {
    switch (id)  {
      case offDefAnalyser       : return (int)offDef_Neutral;
      case goalieCoach          : return (int)gc_Neutral;
      case specialEventAnalyser : return (int)spEv_Neutral;

      default: return -1;
    }
  }

  /** returns the TETID Count of a GlobalAnalysers engine
   * Add new engines here
   */
  static int getGlobalAnalyserTETCount(GlobalAnalyserID id)
  {
    switch (id)  {
      case offDefAnalyser       :  return ((int)offDef_Max - (int)offDef_Neutral);
      case goalieCoach          :  return ((int)gc_Max     - (int)gc_Neutral);
      case specialEventAnalyser :  return ((int)spEv_Max   - (int)spEv_Neutral);

      default: return -1;
    }
  }

  /** returns the neutral TETID of a GlobalAnalysers engine
   * Add new engines here
   */
  static TacticEntryTypeID getGlobalAnalyserNeutralTETID(GlobalAnalyserID id)
  {
    switch (id) {
      case offDefAnalyser       : return offDef_Neutral;
      case goalieCoach          : return gc_Neutral;
      case specialEventAnalyser : return spEv_Neutral;

      default: return FirstTETID;
    }
  }

  /**
   * returns ids for TETID names
   * Add new IDs here
   */
  static TacticEntryTypeID getTETID(GlobalAnalyserID id, const char*  name)
  {
    switch (id) {
      case offDefAnalyser: {
        if (strcmp(name, "Neutral") == 0)
          return offDef_Neutral;
        else if (strcmp(name, "Offensive") == 0)
          return offDef_Offensive;
        else if (strcmp(name, "Defensive") == 0)
          return offDef_Defensive;
        else
          return FirstTETID;
                           } break;

      case goalieCoach: {
        if (strcmp(name, "Neutral") == 0)
          return gc_Neutral;
        else if (strcmp(name, "Clear ball") == 0)
          return gc_ClearBall;
        else if (strcmp(name, "Stay in goal") == 0)
          return gc_StayInGoal;
        else
          return FirstTETID;
                           } break;

      case specialEventAnalyser: {
        if (strcmp(name, "Neutral") == 0)
          return spEv_Neutral;
        else if (strcmp(name, "Nothing special") == 0)
          return spEv_NothingSpecial;
        else if (strcmp(name, "No ball") == 0)
          return spEv_NoBall;
        else if (strcmp(name, "Four Players Left") == 0)
          return spEv_FourPlayersLeft;
        else if (strcmp(name, "Three Players Left") == 0)
          return spEv_ThreePlayersLeft;
        else if (strcmp(name, "Two Players Left") == 0)
          return spEv_TwoPlayersLeft;
        else if (strcmp(name, "One Player Left") == 0)
          return spEv_OnePlayerLeft;
        else if (strcmp(name, "Konter") == 0)
          return spEv_Konter;
        else
          return FirstTETID;
                           } break;

      default: return FirstTETID;
    }
  }

};

#endif //__RateableOptions_h_

/*
 * Change log :
 * $Log: RateableOptions.h,v $
 * Revision 1.67  2004/06/21 11:55:04  hamerla
 * OpenChallenge
 *
 * Revision 1.66  2004/06/07 22:43:27  kerdels
 * added a dtt-switch for open-challenge,
 * merged a bugfix for optionratingdlgbar from local-cvs to tamara,
 * created tactic-entries for openchallenge
 *
 * Revision 1.65  2004/06/07 16:00:58  hamerla
 * OptionRation fr OpenChallenge
 *
 * Revision 1.64  2004/05/12 19:33:14  kerdels
 * merged the behavior changes during australian, american and japan open
 *
 * Revision 1.63  2004/04/08 17:14:27  wachter
 * GT04 checkin of Microsoft-Hellounds
 *
 * Revision 1.68  2004/04/04 01:27:52  kerdels
 * adjusted defence behavior
 *
 * Revision 1.67  2004/03/30 01:32:23  pg_joke
 * added new states to the SpecialEventAnalyser to determine how many teammates are connected via wlan. Used this information in a new, slightly bigger default.dtt. The old default.dtt is saved to tbaseV5.dtt and a copy of the new default.dtt is tbaseV6.dtt. Changed some OptionRatings too. Hopefully this stuff will improve the behavior when the network is broken.
 *
 * Revision 1.66  2004/03/24 15:32:11  pg_joke
 * added a kickToClearGoalie and a newKickToClearGoalie Option
 *
 * Revision 1.65  2004/03/24 15:11:36  pg_joke
 * added location dependend kickratings to improve team-behavior
 *
 * Revision 1.64  2004/03/24 12:54:24  pg_joke
 * added GoToBallGoalie
 *
 * Revision 1.63  2004/03/21 04:36:06  pg_joke
 * added slidingAverage to OptionRatings,
 * added ratingBoost (Hysteresis) to OptionRatings
 *
 * Revision 1.62  2004/03/10 21:44:30  kerdels
 * added goToMiddleAtBall Option
 *
 * Revision 1.61  2004/03/01 17:14:07  kerdels
 * added robot-pose.distance-to-opponent-goal,
 * moved robot-pose.angle-to-teammate1-3 to MSH2004StrategySymbols as fieldpos.angle-to-teammate1-3,
 * added fieldpos.distance-to-teammate1-3,
 * added DTT-Options newKickToGoal, newKickToClear, newKickToTeammate1-3,
 * added DTT-OptionClass newKickToTeammate,
 * added kickToPosRating function in DefaultOptionRating
 *
 * Revision 1.60  2004/02/28 11:20:17  kerdels
 * removed single kicks from behavior...
 *
 * Revision 1.59  2004/02/19 23:34:59  kerdels
 * added SpecialEventAnalyser
 *
 * Revision 1.58  2004/02/19 17:01:11  kerdels
 * added goalie-switch with two demo-options waveLeft and waveRight
 *
 * Revision 1.57  2004/02/18 23:32:59  kerdels
 * added goToFrontPenLeft, goToFrontPenRight
 * added backUp, not rated yet
 * splittet goto-switch.xml in goto-switch.xml and gotoBall-switch.xml
 *
 * Revision 1.56  2004/02/04 23:26:41  kerdels
 * added kicksToClear, clearkick-switch and option-type, specialKickToClear optionClass and apropriate ratings,
 * modified GoToBallPos ratings
 *
 * Revision 1.55  2004/02/04 17:48:06  kerdels
 * heelKicks removed
 *
 * Revision 1.54  2004/02/04 17:19:00  kerdels
 * added Threshold for OptionClasses
 *
 * Revision 1.53  2004/02/04 16:45:10  kerdels
 * added keepOption
 *
 * Revision 1.52  2004/02/04 15:34:51  rossdeutscher
 * improved overall behaviour
 *
 * Revision 1.51  2004/02/03 23:47:33  kerdels
 * added GoalieCoach and adjusted tactics.dat.
 *
 * Revision 1.50  2004/02/03 12:51:53  lohmann
 * BananaCross splitted into Left and Right.
 *
 * Revision 1.49  2004/02/02 16:33:58  kerdels
 * added DTT-options goToCatenaccioLeft, goToCatenaccioMiddle and goToCatenaccioRight.
 * Also added catfun.dat to test this new options
 *
 * Revision 1.48  2004/02/02 12:23:45  lohmann
 * unsw-bash-optimized-kick added
 *
 * Revision 1.47  2004/01/29 23:56:35  kerdels
 * added special goalie position to DTT
 *
 * Revision 1.46  2004/01/29 21:33:28  kerdels
 * added turn-for-ball Option to DTT,
 * added link between TacticChooser and OptionRating
 *
 * Revision 1.45  2004/01/28 23:27:32  kerdels
 * added rollLeft and rollRight Option to DTT,
 * rating has to be adjusted
 *
 * Revision 1.44  2004/01/28 12:43:25  kerdels
 * added kickFun.dat
 * With this TacticEntries 2 Dogs do some kicking exercise :)
 *
 * Revision 1.43  2004/01/28 00:53:02  kerdels
 * added OptionClasses specialKickToGoal and specialKickToTeammate
 *
 * Revision 1.42  2004/01/28 00:01:46  kerdels
 * added legKick, leftKick, rightKick, chestKick, leftHeadKick, rightHeadKick and fastDiveKick to DTT
 *
 * Revision 1.41  2004/01/27 22:26:55  kerdels
 * added fastHeadKicks to DTT
 *
 * Revision 1.40  2004/01/27 20:20:07  kerdels
 * some minor changes
 *
 * Revision 1.39  2004/01/27 14:26:08  lohmann
 * Banana Cross added.
 *
 * Revision 1.38  2004/01/27 02:32:19  kerdels
 * restructuring the restructuring of our xabsl-stuff... +
 * found and removed a bad bug in collectedBeliefs.cpp
 *
 * Revision 1.37  2004/01/26 21:33:31  kerdels
 * added swimKicks
 *
 * Revision 1.36  2004/01/26 20:40:50  kerdels
 * added pawKickRight
 *
 * Revision 1.35  2004/01/26 18:45:06  kerdels
 * added pawKickLeft
 *
 * Revision 1.34  2004/01/26 18:02:17  kerdels
 * added heel-kicks to dtt
 *
 * Revision 1.33  2004/01/25 21:22:56  kerdels
 * added mantisKickToTeammates
 *
 * Revision 1.32  2004/01/25 17:28:24  kindler
 * Added .cpp file for RateableOptions
 *
 * Revision 1.31  2004/01/25 12:17:24  kerdels
 * added first single-kick-option "mantisKick" and modified aging of ratedOptions
 *
 * Revision 1.30  2004/01/23 12:54:15  udo
 * InterceptBall Position and Rating included
 *
 * Revision 1.29  2004/01/22 23:16:38  kerdels
 * dribbel-option hinzugefgt
 *
 * Revision 1.28  2004/01/22 22:28:46  kerdels
 * Optionen fr Bandenverhalten hinzugefuegt
 *
 */
