/** 
* @file Stopwatch.h
* Makros and for the Stopwatch
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#ifndef __Stopwatch_h_
#define __Stopwatch_h_

#include "Tools/Debugging/Debugging.h"
#include "Platform/SystemCall.h"
#include "Platform/GTAssert.h"

/** 
* @class Stopwatch
* Provides IDs and names for different stopwatch tasks; makros
* are used to provide real stopwatch functionality with that.
*/
class Stopwatch
{
public:
  enum StopwatchEventID
  {
    imageProcessor,
      sensorDataProcessor,
      ballLocator,
      teamBallLocator,
      selfLocator,
      playersLocator,
      obstaclesLocator,
      behaviorControl,
      motionControl,
      soundControl,
      specialVision,
      perceptBehaviorControl,
      sensorBehaviorControl,
      headControl,
      collisionDetector,
      robotStateDetector,

      numberOfStopwatchEventIDs
  };
  
  static DebugKeyTable::debugKeyID getDebugKeyID(enum StopwatchEventID stopwatchEventID)
  {
    switch (stopwatchEventID) 
    {
   case imageProcessor: return DebugKeyTable::sendImageProcessorTime;
   case sensorDataProcessor: return DebugKeyTable::sendSensorDataProcessorTime;
   case ballLocator: return DebugKeyTable::sendBallLocatorTime;
   case teamBallLocator: return DebugKeyTable::sendTeamBallLocatorTime;
   case selfLocator: return DebugKeyTable::sendSelfLocatorTime;
   case playersLocator: return DebugKeyTable::sendPlayersLocatorTime;
   case obstaclesLocator: return DebugKeyTable::sendObstaclesLocatorTime;
   case behaviorControl: return DebugKeyTable::sendBehaviorControlTime;
    case motionControl: return DebugKeyTable::sendMotionControlTime;
    case soundControl: return DebugKeyTable::sendSoundControlTime;
    case specialVision: return DebugKeyTable::sendSpecialVisionTime;
    case perceptBehaviorControl: return DebugKeyTable::sendPerceptBehaviorControlTime;
    case sensorBehaviorControl: return DebugKeyTable::sendSensorBehaviorControlTime;
    case headControl: return DebugKeyTable::sendHeadControlTime;
    case collisionDetector: return DebugKeyTable::sendCollisionDetectorTime;
    case robotStateDetector: return DebugKeyTable::sendRobotStateDetectorTime;
    default: 
      {
        ASSERT(false);
        return (DebugKeyTable::debugKeyID)0;
      }
    }
  }
  
  static const char* getStopwatchEventIDName(enum StopwatchEventID stopwatchEventID)
  {
    switch (stopwatchEventID) 
    {
   case imageProcessor: return "imageProcessor";
   case sensorDataProcessor: return "sensorDataProcessor";
   case ballLocator: return "ballLocator";
   case teamBallLocator: return "teamBallLocator";
   case selfLocator: return "selfLocator";
   case playersLocator: return "playersLocator";
   case obstaclesLocator: return "obstaclesLocator";
   case behaviorControl: return "behaviorControl";
   case perceptBehaviorControl: return "perceptBehaviorControl";
   case sensorBehaviorControl: return "sensorBehaviorControl";
    case motionControl: return "motionControl";
    case soundControl: return "soundControl";
    case specialVision: return "specialVision";
    case headControl: return "headControl";
    case collisionDetector: return "collisionDetector";
    case robotStateDetector: return "robotStateDetector";
    default: return "check available stopwatchEventIDs";
    }
  }
};

#ifdef NDEBUG

#define STOP_TIME_ON_REQUEST(eventID, expression) expression
#define STOP_TIME(expression) expression/**/

#else //NDEBUG

/**
* Stops the execution time of "expression"  if a DebugKey is set for the eventID.
* The start and stop times are sent as a idStopwatch message to RobotControl
*/

#define STOP_TIME_ON_REQUEST(eventID, expression) \
  if (getDebugKeyTable().isActive(Stopwatch::getDebugKeyID(Stopwatch::eventID))) \
{\
  unsigned long eventID##StartTime; \
  static unsigned long evendID##Counter = 0; \
  eventID##StartTime = SystemCall::getCurrentSystemTime();\
  expression \
  OUTPUT(idStopwatch, bin, (char) Stopwatch::eventID << eventID##StartTime << SystemCall::getCurrentSystemTime() << evendID##Counter++)\
}\
  else {expression}\


/** 
* Starts a StopWatch without an id.
*/
#define STOP_TIME(expression) \
{\
  unsigned long stopwatchStartTime = SystemCall::getCurrentSystemTime();\
  expression\
  OUTPUT(idText, text, "Stopwatch: " << SystemCall::getTimeSince(stopwatchStartTime) << "ms")\
}\

#endif //NDEBUG

#endif //Stopwatch_h

/*
 * Change log :
 * 
 * $Log: Stopwatch.h,v $
 * Revision 1.1  2003/10/07 10:13:22  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1  2003/09/26 11:40:40  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.3  2003/09/01 15:09:37  juengel
 * Removed playersPerceptor
 *
 * Revision 1.2  2003/09/01 10:23:14  juengel
 * DebugDrawings clean-up 2
 * DebugImages clean-up
 * MessageIDs clean-up
 * Stopwatch clean-up
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.15  2003/06/24 08:10:53  dueffert
 * stopping times works in WIN32 again, no special (buggy) solution any more
 *
 * Revision 1.14  2003/06/22 12:55:24  roefer
 * Stopwatch RT compatible (2nd try)
 *
 * Revision 1.13  2003/06/22 09:55:12  roefer
 * Stopwatch now RT compatible
 *
 * Revision 1.12  2003/06/21 12:49:25  juengel
 * Added robotStateDetector.
 *
 * Revision 1.11  2003/05/26 20:35:44  goehring
 * collisionDetectorTime added
 *
 * Revision 1.10  2003/05/08 13:20:22  loetzsch
 * added the execution of the team ball locator to the Cognition process
 *
 * Revision 1.9  2003/04/16 07:00:17  roefer
 * Bremen GO checkin
 *
 * Revision 1.9  2003/04/08 18:24:42  roefer
 * HeadControl time measurement added
 *
 * Revision 1.8  2003/03/24 09:56:25  schumann
 * added profiling for strategical database
 *
 * Revision 1.7  2003/03/12 13:46:04  dasmuli
 * PlayersPerceptor added to cognition, modules.cfg, DebugKey-table etc.
 *
 * Revision 1.6  2003/03/10 14:00:52  juengel
 * Added ObstaclesLocator
 *
 * Revision 1.5  2002/11/27 13:47:21  dueffert
 * doxygen docu added
 *
 * Revision 1.4  2002/11/26 19:22:18  juengel
 * added stopwatchID for SensorBehaviorControl
 *
 * Revision 1.3  2002/10/02 15:50:54  loetzsch
 * added the debug key sendPerceptBehaviorControlTime
 *
 * Revision 1.2  2002/10/01 11:14:34  loetzsch
 * Redesigned DebugKey and DebugKeyTable
 *
 * Revision 1.1  2002/09/10 15:53:58  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.8  2002/09/07 13:36:55  loetzsch
 * unified the vision modules into one module "ImageProcessor"
 * - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
 *   and PlayersPerceptor were are combined to the new solution
 *   "BlobImageProcessor"
 * - The GridImageProcessor and the SubPixelGradientCalculator became
 *   a solution of "ImageProcessor"
 *
 * Revision 1.7  2002/07/23 13:48:41  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - exchanged StaticQueue by MessageQueue
 * - new debug message handling
 * - general clean up
 *
 * Revision 1.6  2002/06/22 03:05:20  dueffert
 * bug fixed: never return numOfDebugKeys
 *
 * Revision 1.5  2002/06/12 11:34:29  roefer
 * SimpleLinesPerceptor removed, PerceptBehaviorControl added
 *
 * Revision 1.4  2002/05/26 15:33:43  loetzsch
 * removed the looped execution of modules in _WIN32
 *
 * Revision 1.3  2002/05/14 20:58:05  hebbel
 * added id for SoundInProcessor
 *
 * Revision 1.2  2002/05/10 17:31:35  juengel
 * Added SpecialVision and SpecialPercept.
 *
 * Revision 1.16  2002/05/03 17:14:35  giese
 * StopTimeOnRequest for SoundOutControl added
 *
 * Revision 1.15  2002/03/16 13:44:49  juengel
 * GridImageProcessor created.
 *
 * Revision 1.14  2002/02/28 10:39:12  juengel
 * change log added.
 *
 *
 */












