/** 
* @file DebugKeyTable.h
* Declaration of class DebugKey and DebugKeyTable
*
* @author <A href=mailto:brunn@sim.informatik.tu-darmstadt.de>Ronnie Brunn</A>
* @author <A href=mailto:risler@sim.informatik.tu-darmstadt.de>Max Risler</A>
* @author Martin Ltzsch
*/

#ifndef __DEBUGKEYTABLE_H__
#define __DEBUGKEYTABLE_H__

#include "Tools/Streams/InOut.h"

/** 
* @class DebugKey
* 
* Represents a request for debug output.
*/
class DebugKey
{
public:
  enum modes { 
    disabled, /*!< dont send anything */
      always, /*!< always send the debug message */
      n_times, /*!< send the debug message the next n times (frames) */
      every_n_times, /*!< send the message every n times (frames) */
      every_n_ms /*!< send the message at most every n milliseconds */
  };
  
  /** the mode of this DebugKey */
  modes mode; 
  
  /** the parameter for timed or counted debug messages */
  unsigned int n; 
  
  /** last time (ever_n_ms) or number of times (every_n_times, n_times) the message was sent */
  unsigned long last; 
  
  /** 
  * Determines if the key is active in the current frame 
  * This is set by activateDebugKeys()
  */
  bool active;
  
  /**
  * construct a new DebugKey
  */
  DebugKey(modes mode=disabled, unsigned int n=0);
  
  /** sets the mode and the parameter of a DebugKey */
  void set(modes mode=always, unsigned int n=0);
};


/**
* Contains a table of debug keys and all methods for using debug keys.
*
* @author <A href=mailto:brunn@sim.informatik.tu-darmstadt.de>Ronnie Brunn</A>
* @author <A href=mailto:risler@sim.informatik.tu-darmstadt.de>Max Risler</A>
* @author Martin Ltzsch
*/
class DebugKeyTable
{
public:
/**
* IDs for Debug Keys.
* enter new Debug Keys here and add the corresponding string in getDebugKeyName
  */
  enum debugKeyID {
      sendImage,
      sendJPEGImage,
      sendLowResImage,
      sendSensorData,
      sendJointData,
      sendOdometryData,
      sendExecutedMotionRequest,
      sendPercepts,
      sendSpecialPercept,
      sendWorldState,
      sendGameControlData,
      sendPlayerConfig,
      sendSoundRequest,
      sendOptionRatings,
      sendShootRecords,
      
      sendTimeOffsets,
      printRobotStatus,
      printCBBRules,
      printPixelUsage,
      learnWalking,
      sendDDPInfo,
      sendWalkingEngineInfo,
      sendKickEngineInfo,

      send_ballLocatorField_drawing,
      numOfDebugKeysForDebugKeyToolbar,

      sendXabsl2DebugMessagesForBehaviorControl,
      sendXabsl2DebugMessagesForHeadControl,
      sendColorTable64,
      sendColorTableReferenceColor,
      sendColorTableCuboids,
      
      // histograms
      sendHistogram_imageIntensityY,
      sendHistogram_imageIntensityU,
      sendHistogram_imageIntensityV,
      sendHistogram_scanLineIntensityY,
      sendHistogram_scanLineIntensityU,
      sendHistogram_scanLineIntensityV,
      sendHistogram_colorFrequency,
      sendHistogram_lengthOfSegments,
      
      //image drawings
      send_selfLocator_drawing,
      send_imageProcessor_general_drawing,
      send_imageProcessor_horizon_drawing,
      send_imageProcessor_scanLines_drawing,
      send_imageProcessor_coloredSegments1_drawing,
      send_imageProcessor_coloredSegments2_drawing,
      send_imageProcessor_coloredSegments3_drawing,
      send_imageProcessor_calibration1_drawing,
      send_imageProcessor_calibration2_drawing,
      send_imageProcessor_ball1_drawing,
      send_imageProcessor_ball2_drawing,
      send_imageProcessor_ball3_drawing,
      send_imageProcessor_ball4_drawing,
      send_imageProcessor_ground_drawing,
      send_imageProcessor_flagsAndGoals_drawing,
      send_imageProcessor_obstacles_drawing,
      send_imageProcessor_gradients_drawing,
      send_sketch_drawing,

      //field drawings
      send_selfLocatorField_drawing,
      send_teamBallLocatorField_drawing,
      // send_obstaclesLocatorField_drawing,
      send_behavior_CBBVector_drawing,
      send_behavior_CBBWay_drawing,
      send_behavior_kickAngles_drawing,
      send_behavior_aStarSearch_drawing,
      send_dribbleBall_drawing,
      
      //radar drawings
      send_behavior_kickAnglesRadar_drawing,
      send_models_corridorsRadar_drawing,

      //debug images
      send_segmentedImage1_image,
      send_segmentedImage2_image,
      send_segmentedImage3_image,
      send_classificationY_image,
      send_classificationU_image,
      send_classificationV_image,
      send_colorFrequency_image,
      send_imageProcessorGeneral_image,
      send_imageProcessorScanLines_image,
      send_imageProcessorBall_image,
      send_imageProcessorGoals_image,
      send_imageProcessorFlags_image,
      send_imageProcessorPlayers_image,
      send_imageProcessorGradients_image,
	  send_imageMotionRecognition_image,
      
      //times
      sendImageProcessorTime,
      sendSensorDataProcessorTime,
      sendBallLocatorTime,
      sendTeamBallLocatorTime,
      sendSelfLocatorTime,
      sendPlayersLocatorTime,
      sendObstaclesLocatorTime,
      sendBehaviorControlTime,
      sendMotionControlTime,
      sendSoundControlTime,
      sendSpecialVisionTime,
      sendPerceptBehaviorControlTime,
      sendSensorBehaviorControlTime,
      sendHeadControlTime,
      sendCollisionDetectorTime,
      sendRobotStateDetectorTime,

      
      numOfDebugKeys
  } ;
  
  /**
  * returns the DebugKey's name
  */
  static char* getDebugKeyName (debugKeyID aID) 
  {
    switch (aID) {
    case sendImage: return "sendImage";
    case sendJPEGImage: return "sendJPEGImage";
    case sendLowResImage: return "sendLowResImage";
    case sendSensorData: return "sendSensorData";
    case sendJointData: return "sendJointData";
    case sendOdometryData: return "sendOdometryData";
    case sendExecutedMotionRequest: return "sendExecutedMotionRequest";
    case sendPercepts: return "sendPercepts";
    case sendSpecialPercept: return "sendSpecialPercept";
    case sendWorldState: return "sendWorldState";
    case sendGameControlData: return "sendGameControlData";
    case sendPlayerConfig: return "sendPlayerConfig";
    case sendSoundRequest: return "sendSoundRequest";
      
    case sendXabsl2DebugMessagesForBehaviorControl: return "sendXabsl2DebugMessagesForBehaviorControl";
    case sendXabsl2DebugMessagesForHeadControl: return "sendXabsl2DebugMessagesForHeadControl";
    case sendTimeOffsets: return "sendTimeOffsets";
    case sendColorTable64: return "sendColorTable64";
    case sendColorTableCuboids: return "sendColorTableCuboids";
    case sendColorTableReferenceColor: return "sendColorTableReferenceColor";
    case printRobotStatus: return "printRobotStatus";
    case printCBBRules: return "printCBBRules";
    case printPixelUsage: return "printPixelUsage";
    case learnWalking: return "learnWalking";
    case sendDDPInfo: return "printDogDiscoveryProtocolInformation";
    case sendShootRecords: return "sendShootRecords";
    case sendWalkingEngineInfo: return "sendWalkingEngineInfo";
    case sendKickEngineInfo: return "sendKickEngineInfo";
      
      // histograms
    case sendHistogram_imageIntensityY: return "sendHistogram_imageIntensityY";
    case sendHistogram_imageIntensityU: return "sendHistogram_imageIntensityU";
    case sendHistogram_imageIntensityV: return "sendHistogram_imageIntensityV";
    case sendHistogram_scanLineIntensityY: return "sendHistogram_scanLineIntensityY";
    case sendHistogram_scanLineIntensityU: return "sendHistogram_scanLineIntensityU";
    case sendHistogram_scanLineIntensityV: return "sendHistogram_scanLineIntensityV";
    case sendHistogram_colorFrequency: return "sendHistogram_colorFrequency";
    case sendHistogram_lengthOfSegments: return "sendHistogram_lengthOfSegments";
      
      //image drawings
    case send_selfLocator_drawing: return "send_selfLocator_drawing";
    case send_imageProcessor_general_drawing: return "send_imageProcessor_general_drawing";
    case send_imageProcessor_horizon_drawing: return "send_imageProcessor_horizon_drawing";
    case send_imageProcessor_scanLines_drawing: return "send_imageProcessor_scanLines_drawing";
    case send_imageProcessor_coloredSegments1_drawing: return "send_imageProcessor_coloredSegments1_drawing";
    case send_imageProcessor_coloredSegments2_drawing: return "send_imageProcessor_coloredSegments2_drawing";
    case send_imageProcessor_coloredSegments3_drawing: return "send_imageProcessor_coloredSegments3_drawing";
    case send_imageProcessor_calibration1_drawing: return "send_imageProcessor_calibration1_drawing";
    case send_imageProcessor_calibration2_drawing: return "send_imageProcessor_calibration2_drawing";
    case send_imageProcessor_ball1_drawing: return "send_imageProcessor_ball1_drawing";
    case send_imageProcessor_ball2_drawing: return "send_imageProcessor_ball2_drawing";
    case send_imageProcessor_ball3_drawing: return "send_imageProcessor_ball3_drawing";
    case send_imageProcessor_ball4_drawing: return "send_imageProcessor_ball4_drawing";
    case send_imageProcessor_ground_drawing: return "send_imageProcessor_ground_drawing";
    case send_imageProcessor_flagsAndGoals_drawing: return "send_imageProcessor_flagsAndGoals_drawing";
    case send_imageProcessor_obstacles_drawing: return "send_imageProcessor_obstacles_drawing";
    case send_imageProcessor_gradients_drawing: return "send_imageProcessor_gradients_drawing";
    case send_sketch_drawing: return "send_sketch_drawing";
      
      // field drawings
    case send_ballLocatorField_drawing: return "send_ballLocatorField_drawing";
    case send_selfLocatorField_drawing: return "send_selfLocatorField_drawing";
    case send_teamBallLocatorField_drawing: return "send_teamBallLocatorField_drawing";
      //    case send_obstaclesLocatorField_drawing: return "send_obstaclesLocatorField_drawing";
    case send_behavior_CBBVector_drawing: return "send_behavior_CBBVector_drawing";
    case send_behavior_CBBWay_drawing: return "send_behavior_CBBWay_drawing";
    case send_behavior_kickAngles_drawing: return "send_behavior_kickAngles_drawing";
    case send_behavior_aStarSearch_drawing: return "send_behavior_aStarSearch_drawing";
    case send_dribbleBall_drawing: return "send_dribbleBall_drawing";
      
      // radar drawings 
    case send_models_corridorsRadar_drawing: return "send_models_corridorsRadar";
    case send_behavior_kickAnglesRadar_drawing: return "send_behavior_kickAnglesRadar_drawing";
      
      // debug images
    case send_segmentedImage1_image: return "send_segmentedImage1_image";
    case send_segmentedImage2_image: return "send_segmentedImage2_image";
    case send_segmentedImage3_image: return "send_segmentedImage3_image";
    case send_classificationY_image: return "send_classificationY_image";
    case send_classificationU_image: return "send_classificationU_image";
    case send_classificationV_image: return "send_classificationV_image";
    case send_colorFrequency_image: return "send_colorFrequency_image";
    case send_imageProcessorGeneral_image: return "send_imageProcessorGeneral_image";
    case send_imageProcessorScanLines_image: return "send_imageProcessorScanLines_image";
    case send_imageProcessorBall_image: return "send_imageProcessorBall_image";
    case send_imageProcessorGoals_image: return "send_imageProcessorGoals_image";
    case send_imageProcessorFlags_image: return "send_imageProcessorFlags_image";
    case send_imageProcessorPlayers_image: return "send_imageProcessorPlayers_image";
    case send_imageProcessorGradients_image: return "send_imageProcessorGradients_image";
	case send_imageMotionRecognition_image: return "send_imageMotionRecognition_image";
        
      // times
    case sendImageProcessorTime: return "sendImageProcessorTime";
    case sendSensorDataProcessorTime: return "sendSensorDataProcessorTime";
    case sendBallLocatorTime: return "sendBallLocatorTime";
    case sendTeamBallLocatorTime: return "sendTeamBallLocatorTime";
    case sendSelfLocatorTime: return "sendSelfLocatorTime";
    case sendPlayersLocatorTime: return "sendPlayersLocatorTime";
    case sendObstaclesLocatorTime: return "sendObstaclesLocatorTime";
    case sendBehaviorControlTime: return "sendBehaviorControlTime";
    case sendMotionControlTime: return "sendMotionControlTime";
    case sendSoundControlTime: return "sendSoundControlTime";
    case sendSpecialVisionTime: return "sendSpecialVisionTime";
    case sendPerceptBehaviorControlTime: return "sendPerceptBehaviorControlTime";
    case sendSensorBehaviorControlTime: return "sendSensorBehaviorControlTime";
    case sendHeadControlTime: return "sendHeadControlTime";
    case sendCollisionDetectorTime: return "sendCollisionDetectorTime";
    case sendRobotStateDetectorTime: return "sendRobotStateDetectorTime";

			//option ratings
		case sendOptionRatings: return "sendOptionRatings";
      
    default: return "please edit DebugKeyTable::getDebugKeyName!";
    }
  }
  
protected:  
/**
* The Debug Key Table
  */
  DebugKey debugKeys[numOfDebugKeys];
  
public:
  /** Constructor, resets the table */
  DebugKeyTable();
  
  /** sets a debug key */
  void set(debugKeyID id, DebugKey::modes mode=DebugKey::always, int n=0);
  
  /** sets only the mode of a debug key */
  void setMode(debugKeyID id, DebugKey::modes mode);
  
  /** sets only the parameter n of a debug key */
  void setN(debugKeyID id, unsigned int n);
  
  /** returns the mode of a debug key */
  DebugKey::modes getMode(debugKeyID id) const;
  
  /** returns the parameter n of a debug key */
  unsigned int getN(debugKeyID id) const;
  
  /** returns whether an debug Key is active in the current frame */
  bool isActive (debugKeyID id) const;
  
  /** resets all debug keys to disabled */
  void reset();
  
  /** Determines for all keys if they are active in the current frame */
  void activate();
  
  /** the == operator */
  bool operator == (const DebugKeyTable& other);
  
  /** the != operator */
  bool operator != (const DebugKeyTable& other) 
  { return ! (*this == other); }
};

/**
* Streaming operator that reads a DebugKeyTable from a stream.
* @param stream The stream from which is read.
* @param debugKeyTable The DebugKeyTable object.
* @return The stream.
*/ 
In& operator>>(In& stream,DebugKeyTable& debugKeyTable);

/**
* Streaming operator that writes a DebugKeyTable to a stream.
* @param stream The stream to write on.
* @param debugKeyTable The DebugKeyTable object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, DebugKeyTable& debugKeyTable);


#endif //__DEBUGKEYTABLE_H__

/*
* Change log :
* 
* $Log: DebugKeyTable.h,v $
* Revision 1.19  2004/05/17 18:35:23  loetzsch
* continued support for multiple Xabsl engines in different modules
*
* Revision 1.18  2004/04/08 15:33:08  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.18  2004/04/01 14:45:29  kindler
* - added debug keys for walking- and kick-engine
*
* Revision 1.17  2004/03/26 11:42:21  pg_arce
* improved catch and turn with ball
*
* Revision 1.16  2004/02/12 14:29:44  juengel
* Added drawing ball4.
*
* Revision 1.15  2004/02/03 23:16:46  loetzsch
* added debug key sendShootRecords
*
* Revision 1.14  2004/01/28 15:40:48  schumann
* added transmission of ball locator debugging
*
* Revision 1.13  2004/01/24 18:35:03  wachter
* Reactivated debug-drawings in SensorFusionTeamBallLocator
*
* Revision 1.12  2004/01/21 17:33:09  wachter
* UDP Team-communication now working with packets<1400 bytes.
* Not activated at the moment.
*
* Revision 1.11  2004/01/20 15:46:17  tim
* Added drawing for potential field A* search
*
* Revision 1.10  2004/01/03 23:07:49  ordyniak
* new debugimage imageMotionRecognition
*
* Revision 1.9  2003/12/29 15:48:54  roefer
* Bremen Byters evo walking added
*
* Revision 1.8  2003/12/28 19:03:29  kerdels
* option rating dialog added
*
* Revision 1.7  2003/12/01 16:24:18  juengel
* Added drawing ball3.
*
* Revision 1.6  2003/11/05 16:43:59  juengel
* Added drawing imageProcessor_ball2
*
* Revision 1.5  2003/10/29 13:34:22  juengel
* added drawing imageProcessor_coloredSegments
* an image colorFrequencyImage
*
* Revision 1.4  2003/10/23 07:21:20  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.3  2003/10/12 20:25:29  juengel
* Added keys for debug images.
*
* Revision 1.2  2003/10/12 11:50:30  juengel
* Added DebugKey sendColorTable64.
*
* Revision 1.1  2003/10/07 10:13:22  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.10  2003/09/26 15:28:10  juengel
* Renamed DataTypes to representations.
*
* Revision 1.9  2003/09/25 11:21:59  juengel
* Removed BlobCollection.
*
* Revision 1.8  2003/09/01 16:18:58  juengel
* Not all DebugKeys are displayed in DebugKeysToolBar.
*
* Revision 1.7  2003/09/01 15:09:37  juengel
* Removed playersPerceptor
*
* Revision 1.6  2003/09/01 10:23:14  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.5  2003/08/30 10:16:41  juengel
* DebugDrawings clean-up 1
*
* Revision 1.4  2003/08/25 17:23:01  juengel
* Added Histograms
*
* Revision 1.3  2003/08/18 11:51:34  juengel
* Added sendColorTableAuto
*
* Revision 1.2  2003/07/03 10:50:17  juengel
* Added Drawing kickAnglesRadar.
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.61  2003/06/21 12:46:56  juengel
* Added sendRobotStateDetectorTime.
*
* Revision 1.60  2003/06/19 18:35:30  juengel
* Added drawing kickAngles.
*
* Revision 1.59  2003/06/05 08:15:10  juengel
* Added "corridors" drawing.
*
* Revision 1.58  2003/05/26 20:35:24  goehring
* collisionDetectorTime added
*
* Revision 1.57  2003/05/09 16:21:58  engel
* DebugKey changed for the GaussBellTeamBallLocator
*
* Revision 1.56  2003/05/08 13:20:22  loetzsch
* added the execution of the team ball locator to the Cognition process
*
* Revision 1.55  2003/05/05 12:26:35  juengel
* Removed sendGridImageProcessor2ColorCubes
*
* Revision 1.54  2003/04/30 09:17:08  kallnik
* renamed some DebugKeys and defines
*
* Revision 1.53  2003/04/28 11:01:02  kallnik
* new ContinuousBasicBehavior version (NORules is taken from number of lines in Rules-section
* and some bugfixes
*
* Revision 1.52  2003/04/25 12:57:14  kallnik
* new version of ContinuousBasicBehavior released
*
* Revision 1.51  2003/04/16 14:14:38  loetzsch
* removed Xabsl 1 from GT2003
*
* Revision 1.50  2003/04/16 07:00:17  roefer
* Bremen GO checkin
*
* Revision 1.49  2003/04/15 15:52:07  risler
* DDD GO 2003 code integrated
*
* Revision 1.48  2003/04/10 13:53:43  max
* send_playerSpecialistDrawing added
*
* Revision 1.47  2003/04/09 15:51:56  kallnik
* ContinuousBasicBehavior Debug Drawing   for   ContinuousBasicBehaviorTester
*
* Revision 1.48  2003/03/29 20:01:05  loetzsch
* added missing key "printAdviseOutput"
*
* Revision 1.47  2003/03/29 13:10:57  thiel
* added HexAreaPotentialfieldOutput messages
* added DroneOutput messages
* added VisionaryOutput messages
*
* Revision 1.46  2003/03/28 14:13:58  loetzsch
* added debug drawing areaOfResponsibility
*
* Revision 1.45  2003/03/27 13:12:09  schumann
* added strategicalDatabaseOutput messages
*
* Revision 1.44  2003/03/25 10:54:56  juengel
* Added some descriptions.
*
* Revision 1.43  2003/03/25 08:47:11  dasmuli
* Added simple segmentation and the 180/360 memory. Dialog was changed in order to display segemented data.
*
* Revision 1.42  2003/03/24 09:56:50  schumann
* added profiling for strategical database
*
* Revision 1.41  2003/03/21 12:22:21  thiel
* added hexAreaPotentialfieldDrawing
*
* Revision 1.40  2003/03/19 20:16:19  schumann
* added strategical database drawing
*
* Revision 1.39  2003/03/19 15:39:08  jhoffman
* changed drawing name
*
* Revision 1.38  2003/03/19 09:37:46  dueffert
* merge warning removed
*
* Revision 1.37  2003/03/19 09:34:00  dueffert
* merge bug fixed
*
* Revision 1.36  2003/03/19 09:09:55  dueffert
* no message
*
* Revision 1.35  2003/03/15 13:25:49  juengel
* Added descriptions for several debug drawings and debug keys.
*
* Revision 1.34  2003/03/12 13:46:04  dasmuli
* PlayersPerceptor added to cognition, modules.cfg, DebugKey-table etc.
*
* Revision 1.33  2003/03/10 14:06:35  juengel
* Added ObstaclesLocator
*
* Revision 1.32  2003/03/07 19:05:13  juengel
* Added sendExecutedMotionRequest
*
* Revision 1.31  2003/03/07 15:35:59  wachter
* Added debug-drawings for SensorFusionBall and PlayerLocator.
*
* Revision 1.30  2003/03/05 14:07:10  roefer
* LinesPerceptor2 is now LinesImageProcessor
*
* Revision 1.29  2003/03/05 10:35:35  jhoffman
* added debugdrawing
*
* Revision 1.28  2003/03/04 14:52:33  juengel
* Added sendParametersFromGridImageProcessor2
*
* Revision 1.27  2003/02/27 12:56:35  juengel
* Added some drawings.
*
* Revision 1.26  2003/02/19 15:01:23  roefer
* LinesPerceptor2 added
*
* Revision 1.25  2003/02/18 13:24:37  wachter
* added new TeamMessageCollection and TeamMessage
*
* Revision 1.24  2003/02/16 08:29:40  roefer
* sendOdometryData added
*
* Revision 1.23  2003/02/11 23:12:50  roefer
* useConstantBodyPose debug key added
*
* Revision 1.22  2003/02/08 18:40:06  juengel
* added imageProcessorDrawing
*
* Revision 1.21  2003/02/06 17:04:43  goehring
* Added BarcodeReaderTest Key
*
* Revision 1.20  2003/01/15 13:47:29  roefer
* Progress in LinesSelfLocator, new debug drawing
*
* Revision 1.19  2003/01/13 18:26:04  juengel
* Added LineFollowerField drawing
*
* Revision 1.18  2003/01/08 18:46:52  loetzsch
* added key sendXabsl2Messages
*
* Revision 1.17  2002/11/28 20:42:47  risler
* added walking drawing
*
* Revision 1.16  2002/11/26 19:22:18  juengel
* added stopwatchID for SensorBehaviorControl
*
* Revision 1.15  2002/11/20 15:46:27  juengel
* PathPlanner added.
*
* Revision 1.14  2002/11/19 17:12:39  juengel
* Line Follower added.
*
* Revision 1.13  2002/11/19 15:43:03  dueffert
* doxygen comments corrected
*
* Revision 1.12  2002/11/18 19:24:16  juengel
* Line Follower added.
*
* Revision 1.11  2002/11/18 17:30:31  loetzsch
* Sound requests generated from BehaviorControl are now automatically sent
* to RobotControl and played there.
*
* Revision 1.10  2002/11/18 10:33:30  juengel
* renamed drawing keys
*
* Revision 1.9  2002/10/11 13:54:44  roefer
* JPEGImage added
*
* Revision 1.8  2002/10/10 13:30:14  jhoffman
* added ISL related stuff
*
* Revision 1.7  2002/10/10 13:09:50  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
* Revision 1.6  2002/10/02 15:50:55  loetzsch
* added the debug key sendPerceptBehaviorControlTime
*
* Revision 1.5  2002/10/01 11:14:34  loetzsch
* Redesigned DebugKey and DebugKeyTable
*
* Revision 1.4  2002/09/29 18:03:29  loetzsch
* renamed calculate... to activateDebugKeys
*
* Revision 1.3  2002/09/29 12:32:37  juengel
* Changed semantics of "debug key is active".
* If a debug key is active changes only before the execution of a process.
*
* Revision 1.2  2002/09/18 16:35:13  loetzsch
* made GT2003 compilable,
* rechanged message ids
*
* Revision 1.1  2002/09/10 15:53:58  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.16  2002/09/07 13:36:57  loetzsch
* unified the vision modules into one module "ImageProcessor"
* - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
*   and PlayersPerceptor were are combined to the new solution
*   "BlobImageProcessor"
* - The GridImageProcessor and the SubPixelGradientCalculator became
*   a solution of "ImageProcessor"
*
* Revision 1.15  2002/08/21 11:50:44  cesarz
* GT2001PlayersPerceptor, BerlinSelfLocator, GT2001WalkingEngine,
* DoWalkingEngine, FuzzyBehaviorControl, BremenGOBehaviorControl,
* FuzzyBehaviortesterdlgbar and Behaviortesterdlgbar removed.
*
* Revision 1.14  2002/08/04 19:42:10  roefer
* SimGT2002 receives player config
*
* Revision 1.13  2002/07/23 13:48:28  loetzsch
* - new streaming classes
* - removed many #include statements
* - exchanged StaticQueue by MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.12  2002/06/12 17:57:20  juengel
* sendImageFromPatternPerceptor added.
*
* Revision 1.11  2002/06/08 11:44:00  mkunz
* Special Percept on Field
*
* Revision 1.10  2002/06/04 00:11:52  loetzsch
* added == and != operator
*
* Revision 1.9  2002/06/01 10:12:20  juengel
* DebugKey sendCrosshairsDrawing added.
*
* Revision 1.8  2002/05/24 23:01:04  mkunz
* "sendBarDrawing"-Key corected
*
* Revision 1.7  2002/05/17 11:59:03  brunn
* fixed merge error
*
* Revision 1.6  2002/05/17 11:51:48  brunn
* first experimental version of the barPerceptor
*
* Revision 1.5  2002/05/17 11:44:50  dueffert
* missing enum added
*
* Revision 1.4  2002/05/16 08:33:59  juengel
* Added DebugDrawing "pattern"
*
* Revision 1.3  2002/05/14 21:01:08  hebbel
* added id for SoundInProcessor
*
* Revision 1.2  2002/05/10 17:30:03  juengel
* Added SpecialVision and SpecialPercept.
*
* Revision 1.27  2002/05/05 22:17:06  loetzsch
* GameControlData can now be sent from the Game toolbar to BehaviorControl
*
* Revision 1.26  2002/05/05 22:12:42  loetzsch
* GameControlData can now be sent from the Game toolbar to BehaviorControl
*
* Revision 1.25  2002/05/03 17:15:17  giese
* StopTimeOnRequest for SoundOutControl added
*
* Revision 1.24  2002/04/23 14:16:47  jhoffman
* added gradient image processor
*
* Revision 1.1.1.1  2002/04/10 11:03:22  juengel
* no message
*
* Revision 1.23  2002/04/08 19:53:14  juengel
* Drawing of percept collections in images added.
*
* Revision 1.22  2002/04/08 15:58:51  fischer
* Added functionality to Fuzzy Behavior Control Tester
*
* Revision 1.21  2002/04/04 18:43:59  juengel
* FlagSpecialist improved.
*
* Revision 1.20  2002/04/02 15:05:39  loetzsch
* XabslDlgBar can now test skills
*
* Revision 1.19  2002/04/02 10:30:34  juengel
* GridImageProcessor enhanced.
*
* Revision 1.18  2002/03/24 18:15:01  loetzsch
* continued change to blocking sensor data receivers
*
* Revision 1.17  2002/03/24 17:47:02  juengel
* LinesPercept defined and LinesPerceptor added.
*
* Revision 1.16  2002/02/25 15:07:45  loetzsch
* neue keys
*
* Revision 1.15  2002/02/24 10:02:21  juengel
* TimeDiagramDlgBar completed.
*
* Revision 1.14  2002/02/22 17:17:33  loetzsch
* DebugKey "SendAllStopWatches" removed
*
* Revision 1.13  2002/02/21 14:21:50  loetzsch
* Debug Keys for stop watch added
*
* Revision 1.12  2002/02/18 15:36:16  loetzsch
* added a debug key for monte carlo drawings
*
* Revision 1.11  2002/02/05 03:39:23  loetzsch
* added ids for sending world states and percept collections
*
* Revision 1.10  2002/01/28 14:01:56  loetzsch
* useless DebugKeys removed
*
* Revision 1.9  2002/01/23 12:36:16  juengel
* DebugDrawing Makros eingefgt.
*
* Revision 1.8  2002/01/23 07:50:03  loetzsch
* removed workaround in the >> streaming operator
*
* Revision 1.7  2002/01/22 14:55:15  juengel
* ImageToPerceptCollection eingefhrt
*
* Revision 1.6  2002/01/20 23:34:27  loetzsch
* Sending images and sensor data to processes running in RobotControl now possible
*
* Revision 1.5  2002/01/19 08:07:29  juengel
* stopwatch weiterentwickelt
*
* Revision 1.4  2001/12/20 17:13:42  loetzsch
* streaming operators implemented,  simplified
*
* Revision 1.3  2001/12/13 15:03:02  loetzsch
* DebugKeyToolBar fertig
*
* Revision 1.2  2001/12/12 18:55:23  loetzsch
* Representations/DebugKeyTable.h entfernt
*
* Revision 1.1  2001/12/12 15:21:09  juengel
* CDebugKeyToolBar in eigenes .cpp und .h File verlagert.
*
* Revision 1.5  2001/12/10 17:47:05  risler
* change log added
*
*/
