/**
* @file Tools/Debugging/DebugImages.h
*
* Macros to manipulate and send debug images
* 
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/ 
#ifndef __DebugImages_h_
#define __DebugImages_h_

#include "Tools/Debugging/Debugging.h"

#ifdef NDEBUG
#define NODEBUGDRAWINGS
#endif
#ifdef NO_DEBUG_DRAWING
#define NODEBUGDRAWINGS
#endif

#ifdef NODEBUGDRAWINGS
#define DECLARE_DEBUG_COLOR_CLASS_IMAGE(id) /**/
#define SEND_DEBUG_COLOR_CLASS_IMAGE(id) /**/
#else //NODEBUGDRAWINGS
/**
* Declares a debug image
* @param id An image id (Images::ImageID)
*/
#define DECLARE_DEBUG_COLOR_CLASS_IMAGE(id) ColorClassImage id##ColorClassImage

/**Sends the debug image with the specified id (Images::ImageID) */
#define SEND_DEBUG_COLOR_CLASS_IMAGE(id) INFO(send_##id##_image, idDebugColorClassImage, bin, Images::id << id##ColorClassImage)

#endif //NODEBUGDRAWINGS


#if defined(_WIN32)

/**
* Declares a debug image
* @param id An image id (Images::ImageID)
*/
#define DECLARE_DEBUG_IMAGE(id) Image id##Image

/**
* Initializes a debug image with an image
* @param id An image id (Images::ImageID)
* @param image The Image.
*/
#define INIT_DEBUG_IMAGE(id, image) id##Image = image

/** 
* If an debug image is not created with macros the source code for
* image generation should be encapsuled by this macro.
*/
#define GENERATE_DEBUG_IMAGE(id,expression) \
  if (getDebugKeyTable().isActive(DebugKeyTable::send_##id##_image)) {\
  expression;\
  }

/**Sends the debug image with the specified id (Images::ImageID) */
#define SEND_DEBUG_IMAGE(id) INFO(send_##id##_image, idDebugImage, bin, Images::id << id##Image)

/**Changes the y channel of the specified pixel in the specified debug image */
#define DEBUG_IMAGE_SET_PIXEL_Y(id, xx, yy, y) \
  id##Image.image[yy][0][xx] = y;

/**Sets the y, u and v values of the specified pixel in the specified debug image */
#define DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, y, u, v) \
  id##Image.image[yy][0][xx] = y; \
  id##Image.image[yy][1][xx] = u; \
  id##Image.image[yy][2][xx] = v; 

#define DEBUG_IMAGE_GET_PIXEL_Y(id, xx, yy) id##Image.image[yy][0][xx]
#define DEBUG_IMAGE_GET_PIXEL_U(id, xx, yy) id##Image.image[yy][1][xx]
#define DEBUG_IMAGE_GET_PIXEL_V(id, xx, yy) id##Image.image[yy][2][xx]


#define DEBUG_IMAGE_SET_PIXEL_BLACK(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 0, 127, 127)

#define DEBUG_IMAGE_SET_PIXEL_WHITE(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 255, 127, 127)

#define DEBUG_IMAGE_SET_PIXEL_GREEN(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 180, 0, 0)

#define DEBUG_IMAGE_SET_PIXEL_LIGHT_GRAY(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 192, 127, 127)

#define DEBUG_IMAGE_SET_PIXEL_GRAY(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 127, 127, 127)

#define DEBUG_IMAGE_SET_PIXEL_DARK_GRAY(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 64, 127, 127)

#define DEBUG_IMAGE_SET_PIXEL_DARK_GREEN(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 0, 0, 0)

#define DEBUG_IMAGE_SET_PIXEL_ORANGE(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 100, 255, 0)

#define DEBUG_IMAGE_SET_PIXEL_YELLOW(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 180, 255, 0)

#define DEBUG_IMAGE_SET_PIXEL_RED(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 0, 255, 0)

#define DEBUG_IMAGE_SET_PIXEL_MAUVE(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 0, 180, 255)

#define DEBUG_IMAGE_SET_PIXEL_BLUE(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 180, 0, 255)

#define DEBUG_IMAGE_SET_PIXEL_PINK(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 255, 255, 255)

#define DEBUG_IMAGE_SET_PIXEL_DARK_BLUE(id, xx, yy)\
  DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, 100, 0, 255)

#else //_WIN32

#ifdef __GNUC__
#define DECLARE_DEBUG_IMAGE(id) /**/
#else
#define DECLARE_DEBUG_IMAGE(id) static bool id##Dummy
#endif
#define INIT_DEBUG_IMAGE(id, image) /**/
#define GENERATE_DEBUG_IMAGE(id,expression) /**/
#define SEND_DEBUG_IMAGE(id) /**/
#define DEBUG_IMAGE_SET_PIXEL_Y(id, xx, yy, y) /**/
#define DEBUG_IMAGE_SET_PIXEL_YUV(id, xx, yy, y, u, v) /**/
#define DEBUG_IMAGE_SET_PIXEL_BLACK(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_WHITE(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_GREEN(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_GRAY(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_LIGHT_GRAY(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_DARK_GRAY(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_DARK_GREEN(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_ORANGE(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_YELLOW(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_RED(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_MAUVE(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_BLUE(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_PINK(id, xx, yy) /**/
#define DEBUG_IMAGE_SET_PIXEL_DARK_BLUE(id, xx, yy) /**/
#define DEBUG_IMAGE_GET_PIXEL_Y(id, xx, yy) 0
#define DEBUG_IMAGE_GET_PIXEL_U(id, xx, yy) 0
#define DEBUG_IMAGE_GET_PIXEL_V(id, xx, yy) 0

#endif //_WIN32

/**
* Contains IDs for different (debug) image types
* as well as a method to output their names.
*/
class Images
{
public:
/**
* IDs for images.
* enter new image IDs here and add the corresponding string in getImageIDName.
  */
  enum ImageID
  {
		noImage,
    rawImage,
    segmentedImage1,
    segmentedImage2,
    segmentedImage3,
    classificationY,
    classificationU,
    classificationV,
    colorFrequency,
    imageProcessorGeneral,
    imageProcessorScanLines,
    imageProcessorBall,
    imageProcessorGoals,
    imageProcessorFlags,
    imageProcessorPlayers,
    imageProcessorGradients,
	  imageMotionRecognition, 
    numberOfImageIDs,
    image = numberOfImageIDs, // dummy entry, never use it when constructing a DebugDrawing
    segmentedImage, // dummy entry, never use it when constructing a DebugDrawing
    correctedImage, // dummy entry, never use it when constructing a DebugDrawing
    correctedSegmentedImage // dummy entry, never use it when constructing a DebugDrawing
  };

  /**
  * Returns a description for an indexed image id.
  * Add descriptions for new image ids here.
  */
  static char* getImageIDName(ImageID imageID)
  {
    switch (imageID) 
    {
    case noImage: return "no image";
    case rawImage: return "raw image";
    case segmentedImage1: return "segmented image 1";
    case segmentedImage2: return "segmented image 2";
    case segmentedImage3: return "segmented image 3";
    case classificationY: return "classificationY";
    case classificationU: return "classificationU";
    case classificationV: return "classificationV";
    case colorFrequency: return "colorFrequency";
    case imageProcessorGeneral: return "imageProcessorGeneral";
    case imageProcessorScanLines: return "imageProcessorScanLines";
    case imageProcessorBall: return "imageProcessorBall";
    case imageProcessorFlags: return "imageProcessorFlags";
    case imageProcessorGoals: return "imageProcessorGoals";
    case imageProcessorPlayers: return "imageProcessorPlayers";
    case imageProcessorGradients: return "imageProcessorGradients";
		case imageMotionRecognition: return "imageMotionRecognition";	
    default: return "check available drawings";
    }
  }

  static bool getDebugKeyID(enum ImageID imageID, DebugKeyTable::debugKeyID& debugKeyID)
  {
    switch (imageID) 
    {
    case segmentedImage1: debugKeyID = DebugKeyTable::send_segmentedImage1_image; return true;
    case segmentedImage2: debugKeyID = DebugKeyTable::send_segmentedImage2_image; return true;
    case segmentedImage3: debugKeyID = DebugKeyTable::send_segmentedImage3_image; return true;
    case classificationY: debugKeyID = DebugKeyTable::send_classificationY_image; return true;
    case classificationU: debugKeyID = DebugKeyTable::send_classificationU_image; return true;
    case classificationV: debugKeyID = DebugKeyTable::send_classificationV_image; return true;
    case colorFrequency: debugKeyID = DebugKeyTable::send_colorFrequency_image; return true;
    case imageProcessorGeneral: debugKeyID = DebugKeyTable::send_imageProcessorGeneral_image; return true;
    case imageProcessorScanLines: debugKeyID = DebugKeyTable::send_imageProcessorScanLines_image; return true;
    case imageProcessorBall: debugKeyID = DebugKeyTable::send_imageProcessorBall_image; return true;
    case imageProcessorFlags: debugKeyID = DebugKeyTable::send_imageProcessorFlags_image; return true;
    case imageProcessorGoals: debugKeyID = DebugKeyTable::send_imageProcessorGoals_image; return true;
    case imageProcessorPlayers: debugKeyID = DebugKeyTable::send_imageProcessorPlayers_image; return true;
    case imageProcessorGradients: debugKeyID = DebugKeyTable::send_imageProcessorGradients_image; return true;
		case imageMotionRecognition: debugKeyID = DebugKeyTable::send_imageMotionRecognition_image; return true;
		
    default: return false;
    }
  }
};

#endif //__DebugImages_h_

/*
 * Change log :
 * 
 * $Log: DebugImages.h,v $
 * Revision 1.8  2004/05/01 17:10:51  roefer
 * Debug images are only defined if also drawing functions are available
 *
 * Revision 1.7  2004/04/27 19:39:37  roefer
 * Support for debug images added to simulator
 *
 * Revision 1.6  2004/04/07 13:00:46  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.3  2004/04/06 13:19:37  risler
 * cleaned up and improved high resolution image support
 *
 * Revision 1.2  2004/03/29 15:19:02  Marc
 * Intruduced the Black and White Image
 * Normal Images (not Jpeg) images were now send as Color Image with BW
 *
 * Revision 1.5  2004/01/03 23:07:49  ordyniak
 * new debugimage imageMotionRecognition
 *
 * Revision 1.4  2003/11/03 20:16:09  juengel
 * color class images can be sent from robot now
 *
 * Revision 1.3  2003/10/29 13:32:57  juengel
 * Added colorFrequencyImage
 *
 * Revision 1.2  2003/10/12 20:29:11  juengel
 * DebugImages are only sent if requested.
 *
 * Revision 1.1  2003/10/07 10:13:22  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/01 10:23:14  juengel
 * DebugDrawings clean-up 2
 * DebugImages clean-up
 * MessageIDs clean-up
 * Stopwatch clean-up
 *
 * Revision 1.2  2003/08/25 17:23:19  juengel
 * Added some debug images.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.15  2003/06/18 09:01:40  wachter
 * Added macros to get pixel data from debug images
 *
 * Revision 1.14  2003/06/05 12:53:24  dueffert
 * useless Greenhills warnings removed
 *
 * Revision 1.13  2003/05/04 17:34:33  roefer
 * Support for debug images only in RobotControl
 *
 * Revision 1.12  2003/03/31 03:37:36  osterhues
 * added DEBUG_IMAGE_SET_PIXEL_DARK_BLUE(id, xx, yy)
 *
 * Revision 1.11  2003/02/25 13:59:40  juengel
 * added
 * DEBUG_IMAGE_SET_PIXEL_GRAY(id, xx, yy) ,
 * DEBUG_IMAGE_SET_PIXEL_LIGHT_GRAY(id, xx, yy), and
 * DEBUG_IMAGE_SET_PIXEL_DARK_GRAY(id, xx, yy)
 *
 * Revision 1.10  2003/02/19 15:58:04  dueffert
 * some checkerboard detection work
 *
 * Revision 1.9  2002/11/26 12:19:26  dueffert
 * doxygen docu added
 *
 * Revision 1.8  2002/11/25 14:50:24  jhoffman
 * added motion detector
 *
 * Revision 1.7  2002/11/19 17:13:05  juengel
 * Some "SET_PIXEL_..." macros added.
 *
 * Revision 1.6  2002/11/18 10:33:01  juengel
 * Comments added.
 *
 * Revision 1.5  2002/11/12 23:00:47  dueffert
 * started restore greenhills compatibility
 *
 * Revision 1.4  2002/10/10 13:30:14  jhoffman
 * added ISL related stuff
 *
 * Revision 1.3  2002/09/21 23:17:05  juengel
 * Removed different debug image types.
 *
 * Revision 1.2  2002/09/20 23:28:34  juengel
 * Moved instance of DebugDrawingManager to RobotControlMainFrame.
 *
 * Revision 1.1  2002/09/19 23:49:08  juengel
 * Changed debug image mechanisms.
 *
 */
