/** 
* @file RobotVertices.h
*
* Definition of RobotVertices struct.
*
* @author Uwe Dffert
*/
#ifndef __ROBOTVERTICES_H__
#define __ROBOTVERTICES_H__

#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector3.h"
#include "Tools/RobotConfiguration.h"

/**
* The RobotVertices struct contains the 3D position of important
* points of the robot like the shoulder, knee and foot positions.
* All positions are relative to the neck or a point below the neck.
* So the coordinate of the neck will always be (0,0,z)
* @author Uwe Dffert
*/
struct RobotVertices
{
public:
  Vector3<double> neck;
  //order is the same as in Kinematics::LegIndex{fr,fl,hr,hl}:
  Vector3<double> shoulderPosition[4];
  Vector3<double> kneePosition[4];
  Vector3<double> footPosition[4];
  double bodyTilt;
  double bodyRoll;
  double neckHeight;
  unsigned frameNumber;

  /** Default constructor */
  RobotVertices() : bodyTilt(0), bodyRoll(0), neckHeight(0), frameNumber(0) {}

  /** Constructor.
  * Constructs vertices for a robot with all joints set to 0.
  * @param r The dimensions of the robot.
  */
  RobotVertices(const RobotDimensions& r);
};

/**
 * Streaming operator that reads a RobotVertices from a stream.
 * @param stream The stream from which is read.
 * @param robotVertices The RobotVertices object.
 * @return The stream.
 */ 
In& operator>>(In& stream,RobotVertices& robotVertices);
 
/**
 * Streaming operator that writes a RobotVertices to a stream.
 * @param stream The stream to write on.
 * @param robotVertices The RobotVertices object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const RobotVertices& robotVertices);

#endif //__ROBOTVERTICES_H__

/*
 * Change log :
 * 
 * $Log: RobotVertices.h,v $
 * Revision 1.3  2004/03/20 09:55:28  roefer
 * Preparation for improved odometry
 *
 * Revision 1.2  2004/01/28 21:55:50  roefer
 * RobotDimensions revised
 *
 * Revision 1.1  2003/10/07 10:13:21  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1  2003/09/26 11:40:40  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.2  2003/09/16 11:08:52  dueffert
 * doxygen comments improved
 *
 * Revision 1.1  2003/09/15 20:37:07  dueffert
 * new DataType added
 *
 */
