/** 
* @file JointDataSequencer.cpp
*
* Implementation of JointDataSequencer class.
*
* @author Max Risler
*/

#include "JointDataSequencer.h"

JointDataSequencer::JointDataSequencer()
{
  running = false;
}

JointDataSequencer& JointDataSequencer::operator=(const JointDataSequence& other)
{
  numOfEntries = other.numOfEntries;
  loop = other.loop;
  for (int i = 0; i < numOfEntries; i++)
  {
    entry[i] = other.entry[i];
    interpolate[i] = other.interpolate[i];
    repeats[i] = other.repeats[i];
  }

  running = false;

  return *this;
}

bool JointDataSequencer::isRunning()
{
  return running;
}

void JointDataSequencer::start()
{
  running = (numOfEntries>0);
  sequencePos = 0;
  repetitionCounter = 0;
}

void JointDataSequencer::stop()
{
  running = false;
}

void JointDataSequencer::nextSequenceData(JointData& jointData)
{
  if (!running) return;

  //set next data
  if (interpolate[sequencePos] && sequencePos > 0)
    interpolateJointData(jointData,entry[sequencePos-1],entry[sequencePos],repetitionCounter,repeats[sequencePos]);
  else if (interpolate[sequencePos] && sequencePos == 0 && loop)
	interpolateJointData(jointData,entry[numOfEntries - 1],entry[0],repetitionCounter,repeats[sequencePos]);
  else
    jointData = entry[sequencePos];

  //increase position in sequence
  repetitionCounter++;
  if (repetitionCounter >= repeats[sequencePos])
  {
    //finished repeating current line
    repetitionCounter = 0;
    sequencePos++;
    if (sequencePos >= numOfEntries)
    {
      //finished sequence
      if (loop) 
        //repeat
        sequencePos = 0;
      else
        //finished
        running = false;
    }
  } 
}

void JointDataSequencer::interpolateJointData(
    JointData& jointData,
    const JointData& from, const JointData& to, int step, int len
    )
{
  for (int i = 0; i < JointData::numOfJoint; i++)
    if (from.data[i] == jointDataInvalidValue || to.data[i] == jointDataInvalidValue)
      jointData.data[i] = jointDataInvalidValue;
    else
      jointData.data[i] = 
        from.data[i] + 
        (to.data[i] - from.data[i]) * step / len;
}

/*
 * Change log :
 * 
 * $Log: JointDataSequencer.cpp,v $
 * Revision 1.1  2003/10/07 10:13:21  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1  2003/09/26 11:40:40  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.1  2002/09/10 15:53:58  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/06/10 20:38:16  risler
 * removed bug in loop with interpolating sequences
 *
 * Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.3  2002/02/23 14:50:42  risler
 * start on empty sequence sets running flag correctly
 *
 * Revision 1.2  2002/02/21 18:02:51  risler
 * added assignment operator
 *
 * Revision 1.1  2002/02/21 16:29:21  risler
 * added JointDataSequencer
 *
 *
 */
