/**
 * @file Whisker.h
 * 
 * Definition of class Whisker
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef WHISKER_H_
#define WHISKER_H_

#include "Sensor.h"

/**
* @class Whisker
*
* A class representing a whisker for sensing distances
*/
class Whisker : public Sensor
{
protected:
  /** The current value*/
  double value;
  /** The maximum value (longest distance to be measured)*/
  double maxValue;
  /** The maximum value (shortest distance to be measured)*/
  double minValue;
  
public:
  /** Constructor*/
  Whisker(): value(0.0) {}

  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "whisker";}
  
  /** Computes the value
  * @param The returned value
  * @param portId The id of the sensorport
  */
  virtual void computeValue(double& value, int portId);

  /** Sets the minimum value
  * @param value The value
  */
  void setMinValue(double value) {minValue = value;}

  /** Sets the maximum value
  * @param value The value
  */
  void setMaxValue(double value) {maxValue = value;}

  /** Adds the object to some internal lists
  * @param sensorportList A list of all sensor ports in the scene
  * @param actuatorportList A list of all actuator ports in the scene
  * @param actuatorList A list of all actuators in the scene
  */
  void addToLists(std::vector<Sensorport*>& sensorportList,
    std::vector<Actuatorport*>& actuatorportList,
    std::vector<Actuator*>& actuatorList);

  /** Adds descriptions of the object and its children to a list
  * @param objectDescriptionTree The list of descriptions
  * @param The depth of the object in the object tree
  */
  void addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
                         int depth);

  /** Clones the object and its child nodes
  * @return A pointer to a copy of the object
  */
  SimObject* clone() const;

  /** Draws the whisker ray and all child nodes
  * @param pointOfView The point of view from which the scene is drawn
  * @param visParams Parameters for Visualization
  */
  virtual void draw(const Vector3d& pointOfView,
                    const VisualizationParameterSet& visParams);
};

#endif //WHISKER_H_

/*
 * $Log: Whisker.h,v $
 * Revision 1.2  2003/12/09 13:40:54  roefer
 * href attribute corrected
 *
 * Revision 1.9  2003/12/09 12:38:29  roefer
 * href attribute corrected
 *
 * Revision 1.8  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.7  2003/10/05 16:56:41  tim
 * - changed comments
 *
 * Revision 1.6  2003/09/18 01:52:09  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.5  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.4  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.3  2003/09/08 11:46:54  tim
 * - added Simulation::getSensorportMinValue
 * - added Simulation::getSensorportMaxValue
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */