/**
 * @file Whisker.cpp
 * 
 * Implementation of class Whisker
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include "Platform/OpenGL.h"
#include "Whisker.h"
#include "Sensorport.h"
#include "SimGeometry.h"


void Whisker::computeValue(double& value, int portId)
{
  if(simulationStep>lastComputationStep)
  {
    Vector3d forwardVec(1.0,0.0,0.0);
    forwardVec.rotate(rotation);
    this->value = maxValue;
    std::vector<Intersection*> intersections;
    Vector3d currentPos;
    if(rootNode->intersect(intersections, position, forwardVec))
    {
      double dist;
      std::vector<Intersection*>::iterator pos;
      for(pos = intersections.begin(); pos != intersections.end(); ++pos)
      {
        currentPos = (*pos)->position;
        dist = (currentPos - position).getLength();
        if(dist < this->value && dist >= 0.0)
        {
          this->value = dist;
        }
        delete (*pos);
      }
    }
    intersections.clear();
    lastComputationStep=simulationStep;
  }
  value = this->value;
}

void Whisker::addToLists(std::vector<Sensorport*>& sensorportList,
                         std::vector<Actuatorport*>& actuatorportList,
                         std::vector<Actuator*>& actuatorList) 
{
  Sensorport* whiskerPort = new Sensorport("distance", 0, doubleSensor, this, minValue, maxValue);
  sensorportList.push_back(whiskerPort);
}

void Whisker::addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
                                int depth) 
{
  Sensor::addToDescriptions(objectDescriptionTree, depth);  

  //Add Sensorport:
  ObjectDescription sensorportDesc;
  sensorportDesc.name = "distance";
  sensorportDesc.fullName = fullName + ".distance";
  sensorportDesc.depth = depth + 1;
  sensorportDesc.type = OBJECT_TYPE_SENSORPORT;
  objectDescriptionTree.push_back(sensorportDesc);
}

SimObject* Whisker::clone() const
{
  Whisker* newWhisker = new Whisker();
  newWhisker->setName(name);
  newWhisker->setFullName(fullName);
  newWhisker->setPosition(position);
  newWhisker->rotation = rotation;
  newWhisker->parentNode = parentNode;
  newWhisker->deactivatedChildNode = deactivatedChildNode;
  newWhisker->deactivatedChildNodeSuccessor = deactivatedChildNodeSuccessor;
  newWhisker->intersectionSphereRadius = intersectionSphereRadius;
  newWhisker->minValue = minValue;
  newWhisker->maxValue = maxValue;
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    SimObject* childNode = (*pos)->clone();
    newWhisker->addChildNode(childNode, false);
  }
  SimObject* newObject = newWhisker;
  return newObject;
}

void Whisker::draw(const Vector3d& pointOfView,
                   const VisualizationParameterSet& visParams)
{
  if(visualizeSensors && !drawForCamera)
  {
    Vector3d p1 = position;
    Vector3d p2(1.0,0.0,0.0);
    double drawValue;
    computeValue(drawValue,0);
    p2*=drawValue;
    p2.rotate(rotation);
    p2+=p1;
    glColor3f(1.0,0.0,0.0);
    glBegin(GL_LINES);
    glVertex3d (p1.v[0], p1.v[1], p1.v[2]);
    glVertex3d (p2.v[0], p2.v[1], p2.v[2]);
    glEnd();
  }
  SimObject::draw(pointOfView, visParams);
}

/*
 * $Log: Whisker.cpp,v $
 * Revision 1.2  2003/12/09 13:40:54  roefer
 * href attribute corrected
 *
 * Revision 1.8  2003/12/09 12:38:29  roefer
 * href attribute corrected
 *
 * Revision 1.7  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.6  2003/10/05 15:24:30  tim
 * - changed drag & drop visualization
 *
 * Revision 1.5  2003/09/18 01:52:09  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.4  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */
