/**
 * @file Sphere.h
 * 
 * Definition of class Sphere
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef SPHERE_H_
#define SPHERE_H_

#include "DrawableObject.h"


/**
* @class Sphere
*
* A class representing a sphere in 3D space
*/
class Sphere : public DrawableObject
{
protected:
  /** The radius of the sphere*/
  double radius;

  /** Intersects the sphere and its children with a ray
  * @param pos The starting position of the ray
  * @param ray The ray
  * @param intersectPos The intersection point nearest to pos (if found)
  * @return true, if an intersection point has been found
  */
  virtual bool intersectWithRay(const Vector3d& pos, const Vector3d& ray, Vector3d& intersectPos);
  
public:
  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "sphere";}

  /** Draws the sphere and its child nodes
  * @param pointOfView The point of view from which the scene is drawn
  * @param visParams Parameters for Visualization
  */
  virtual void draw(const Vector3d& pointOfView,
                    const VisualizationParameterSet& visParams);

  /** Sets the radius of the sphere
  * @param radius The radius
  */
  void setRadius(double radius) {this->radius = radius;}

  /** Clones the object and its child nodes
  * @return A pointer to a copy of the object
  */
  virtual SimObject* clone() const;

  /** Returns the maximum distance of any point in the subtree to a given point
  * @param base The point to compute the distance to
  * @return The distance
  */
  virtual double getMaxDistanceTo(Vector3d& base) const;
};

#endif //SPHERE_H_

/*
 * $Log: Sphere.h,v $
 * Revision 1.2  2003/12/09 13:40:54  roefer
 * href attribute corrected
 *
 * Revision 1.7  2003/12/09 12:38:28  roefer
 * href attribute corrected
 *
 * Revision 1.6  2003/10/23 21:30:31  tim
 * - normal computation only after rotations
 * - first steps to faster geometry rendering
 *
 * Revision 1.5  2003/09/18 01:52:09  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.4  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */
