/**
 * @file Sphere.cpp
 * 
 * Implementation of class Sphere
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include "Platform/OpenGL.h"
#include "Sphere.h"
#include "Surface.h"


void Sphere::draw(const Vector3d& pointOfView,
                  const VisualizationParameterSet& visParams) 
{
  if(!invisible || drawForCamera)
  {
    Vector3d color(surface->getColor(inverted && !drawForCamera));
    glColor3f(float(color.v[0]),float(color.v[1]),float(color.v[2]));
    glTranslated(position.v[0],position.v[1],position.v[2]);
    GLUquadricObj* q = gluNewQuadric();
    gluSphere(q, radius, 15, 15);
    gluDeleteQuadric(q);
    glTranslated(-position.v[0],-position.v[1],-position.v[2]);
    SimObject::draw(pointOfView, visParams);
  }
}

SimObject* Sphere::clone() const
{
  Sphere* newSphere = new Sphere();
  newSphere->setName(name);
  newSphere->setFullName(fullName);
  newSphere->setPosition(position);
  newSphere->rotation = rotation;
  newSphere->surface = surface;
  newSphere->radius = radius;
  newSphere->invisible = invisible;
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    SimObject* childNode = (*pos)->clone();
    newSphere->addChildNode(childNode, false);
  }
  SimObject* newObject = newSphere;
  return newObject;
}

double Sphere::getMaxDistanceTo(Vector3d& base) const
{
  double dist(0.0);
  Vector3d vec(base - position);
  double vecLen(vec.getLength());
  if(vecLen < radius)
  {
    dist = vecLen + radius;
  }
  else
  {
    dist = vecLen + 2.0*radius;
  }
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    double childDist = (*pos)->getMaxDistanceTo(base);
    if(dist < childDist)
    {
      dist = childDist;
    }
  }
  return dist;
}

bool Sphere::intersectWithRay(const Vector3d& pos, const Vector3d& ray, Vector3d& intersectPos) 
{
  //Compute distance ray <-> position
  Vector3d distVec = position - pos;
  distVec = ray * distVec;
  double distance = distVec.getLength() / ray.getLength();
  //Does the ray intersect the sphere
  if(distance <= radius)
  {
    //formula: (pos + t*ray - position) = radius     | m = pos - position
    //     <=> (m + t*ray) = radius
    //     <=> m + 2*t*m*ray + t*ray = radius
    //     <=> ray*t + 2*m*ray*t + m-radius = 0    | a=ray, b=2*m*ray, c=m-radius
    //     <=> a*t + b*t +c = 0;
    Vector3d m = pos - position;
    double a = ray.vecmul(ray);
    double b = 2 * m.vecmul(ray);
    double c = m.vecmul(m) - radius * radius;
    double p = b / a;
    double q = c / a;
    double det = p / 2 * p / 2 - q;
    if(det >= 0)
    {
      double t1 = - p / 2 + sqrt(det);
      double t2 = - p / 2 - sqrt(det);
      Vector3d intVec1 = ray;
      intVec1 *= t1;
      Vector3d intVec2 = ray;
      intVec2 *= t2;
      if(intVec1.getLength() >= intVec2.getLength())
      {
        intersectPos = pos + intVec1;
      }
      else
      {
        intersectPos = pos + intVec2;
      }
      return true;
    }
    else
    {
      return false;
    }
  }
  else
  {
    return false;
  }
}

/*
 * $Log: Sphere.cpp,v $
 * Revision 1.4  2003/12/09 13:40:53  roefer
 * href attribute corrected
 *
 * Revision 1.11  2003/12/04 19:57:37  roefer
 * Memory leak (gluNewQuadric) closed
 *
 * Revision 1.10  2003/12/03 18:12:37  roefer
 * Compatibility with VC2003.NET, GUI does still not work completely there
 *
 * Revision 1.9  2003/10/23 21:30:31  tim
 * - normal computation only after rotations
 * - first steps to faster geometry rendering
 *
 * Revision 1.8  2003/10/11 14:53:57  tim
 * - added standard length
 * - added invisibility for objects
 * - changed parser implementation
 *
 * Revision 1.7  2003/10/05 15:24:30  tim
 * - changed drag & drop visualization
 *
 * Revision 1.6  2003/09/18 01:52:09  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.5  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.4  2003/09/11 22:10:57  tim
 * - added new zoom functions
 * - added changing of opening angle
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */
