/**
 * @file SimObject.h
 * 
 * Definition of class SimObject
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef SIMOBJECT_H_
#define SIMOBJECT_H_

#include "Platform/Globals.h"
#include <string>
#include <vector>
#include <list>
#include "SimMath.h"
#include "APIDatatypes.h"

class Sensorport;
class Actuatorport;
class Actuator;
class SimObject;
class Intersection;

/** A type for a list of objects*/
typedef std::list<SimObject*> ObjectList;


/**
* @class SimObject
* A class representing an object.
* All other objects are derivated from SimObject. The class itself
* will only be used from grouping other objects.
*/
class SimObject
{
protected:
  /** The position of the object in 3D space*/
  Vector3d position;
  /** The rotation of the object in 3D space*/
  Matrix3d rotation;
  /** The name of the object*/
  std::string name;
  /** The name of the object including the path in the object tree*/
  std::string fullName;
  /** The child nodes*/
  ObjectList childNodes;
  /** The time simulated by one simulation step*/
  static double stepLength;
  /** The current simulation step*/
  static int simulationStep;
  /** A pointer to the root node of the object tree*/
  static SimObject* rootNode;
  /** A pointer to the parent node*/
  SimObject* parentNode;
  /** A pointer to one deactivated child node
  * (deactivation is used for intersection tests)
  */
  SimObject* deactivatedChildNode;
  /** A pointer to the successor of the deactivated child node
  * (deactivation is used for intersection tests)
  */
  SimObject* deactivatedChildNodeSuccessor;
  /** The radius of a sphere containing the object and all its children
  * (is used for intersection tests and zoom functions)
  */
  double intersectionSphereRadius;
  /** A standard length for drawing*/
  static double standardLength;
  /** Flag for turning sensor visualization on and off*/
  static bool visualizeSensors;
  /** The background color of the scene view*/
  static Vector3d backgroundColor;
  /** The color of ambient light*/
  static Vector3d ambientColor;
  /** Flag: turs off some visualizations, if true*/
  static bool drawForCamera;
  /** Flag: object is invisible in scene, if true*/
  bool invisible;

  /** Intersects the object (and its child nodes) with a ray
  * @param pos The starting position of the ray
  * @param ray The ray
  * @param intersectPos The intersection point nearest to pos (if found)
  * @return true, if an intersection point has been found
  */
  virtual bool intersectWithRay(const Vector3d& pos, const Vector3d& ray, Vector3d& intersectPos) 
  {return false;}
   
public:
  /** Constructor*/
  SimObject();
  
  /** Destructor*/
  virtual ~SimObject();
  
  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "group";}
  
  /** Draws the object and its child nodes
  * @param pointOfView The point of view from which the scene is drawn
  * @param visParams Parameters for Visualization
  */
  virtual void draw(const Vector3d& pointOfView,
                    const VisualizationParameterSet& visParams);
  
  /** Sets the name of the object
  * @param newName The name
  */
  void setName(const std::string& newName) {name = newName;}

  /** Sets the full name of the object (name+path)
  * (The member name will not be set)
  * @param newFullName The full name
  */
  void setFullName(const std::string& newFullName) {fullName = newFullName;}

  /** Returns the name
  * @return The name
  */
  const std::string& getName() const {return name;}

  /** Returns the full name
  * @return The full name
  */
  const std::string& getFullName() const {return fullName;}
  
  /** Sets the current simulation step
  * @param simulationStep The step
  */
  void setSimulationStep(int simulationStep);

  /** Sets the time (in milliseconds) one simulation step simulates
  * @param stepLength The time one simulation step simulates
  */
  void setStepLength(double stepLength) {this->stepLength = stepLength;}
  
  /** Translates the object and its children in 3D space
  * @param translation A vector describing the translation
  */
  virtual void translate(const Vector3d& translation);

  /** Sets the new position of the object (children will be translated, too)
  * @param pos The new position
  */
  virtual void setPosition(const Vector3d& pos);

  /** Returns the position of the object
  * @return The position
  */
  Vector3d getPosition() const {return position;}

  /** Rotates the object and its children in 3D space
  * @param rotation The rotation in matrix form
  * @param origin The point to rotate around
  */
  virtual void rotate(const Matrix3d& rotation, 
                      const Vector3d& origin);

  /** Rotates the object and its children in 3D space
  * @param angles The rotation in angles
  * @param origin The point to rotate around
  */
  void rotate(const Vector3d& angles, const Vector3d& origin);

  /** Rotates the object around a certain axis
  * @param angles The rotation angle
  * @param origin The axis to rotate around
  */
  void rotateAroundAxis(double angle, Vector3d axis);

  /** Returns the rotation matrix
  * @return The matrix
  */
  Matrix3d getRotation() const {return rotation;}
  
  /** Adds a child node
  * @param child A pointer to a new child node
  * @param move Flag: translate and rotate child, if true. Keep child coordinates, otherwise
  */
  void addChildNode(SimObject* child, bool move);

  /** Returns a pointer to the child nodes (hacking interface ;-)
  * @return A pointer to a list of child nodes
  */
  ObjectList* getPointerToChildNodes() {return &childNodes;}

  /** Returns the number of child nodes
  * @return The number of child nodes
  */
  unsigned int getNumberOfChildNodes() const {return childNodes.size();}

  /** Returns a pointer to a child node (hacking interface ;-)
  * @param num The number of the child node
  * @return A pointer to the child node
  */
  SimObject* getChildNode(int num) const;

  /** Sets the root node of the object tree
  * @param node A pointer to the root node
  */
  static void setRootNode(SimObject* node) {SimObject::rootNode = node;}

  /** Sets the parent node of an object
  * @param node A pointer to the parent node
  */
  void setParentNode(SimObject* node) {parentNode = node;}

  /** Returns a pointer to the parent node
  * @return A pointer to the parent node
  */
  SimObject* getParentNode() const { return parentNode;}

  /** Deactivates one child node
  * @node A pointer to the child node to be deactivated
  */
  void deactivateNode(SimObject* node);

  /** Reactivates the deactivated child node*/
  void reactivateNode();

  /** Searches for another object in the object's subtree
  * @param object A pointer to the searched object, if found
  * @param name The name of the object to search for
  */
  void findObject(SimObject*& object, const std::string& name);
  
  /** Clones the object and its child nodes
  * @return A pointer to a copy of the object
  */
  virtual SimObject* clone() const;

  /** Adds the object to some internal lists
  * @param sensorportList A list of all sensor ports in the scene
  * @param actuatorportList A list of all actuator ports in the scene
  * @param actuatorList A list of all actuators in the scene
  */
  virtual void addToLists(std::vector<Sensorport*>& sensorportList,
    std::vector<Actuatorport*>& actuatorportList,
    std::vector<Actuator*>& actuatorList) {};

  /** Adds descriptions of the object and its children to a list
  * @param objectDescriptionTree The list of descriptions
  * @param The depth of the object in the object tree
  */
  virtual void addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
    int depth);
  
  /** Checks the collision of the object and its children with any other object in the scene
  * @return true, if any collision occured
  */
  bool checkCollisionWithScene();

  /** Checks the collision of the object and its children with another object
  * @param The other object
  * @return true, if any collision occured
  */
  virtual bool checkCollisionWithObject(SimObject* object);

  /** Intersects the object and its child nodes with a ray
  * @param intersections A list to insert intersection points into
  * @param pos The starting position of the ray
  * @param ray The ray
  * @return true, if any intersection occured
  */
  bool intersect(std::vector<Intersection*>& intersections, 
    const Vector3d& pos, const Vector3d& ray);

  /** Intersects the object and its child nodes with any other object in the scene
  * @param intersections A list to insert intersection points into
  */
  virtual void intersectWithScene(std::vector<Intersection>& intersections);

  /** Intersects the object and its child nodes with another object
  * @param intersections A list to insert intersection points into
  * @param object The other object
  */
  virtual void intersectWithObject(std::vector<Intersection>& intersections, SimObject* object);

  /** Computes the radius of a sphere containing the object and its child nides*/
  void computeIntersectionSphereRadius();

  /** Returns the radius of the intersection sphere
  * @return The radius
  */
  double getIntersectionSphereRadius() const {return intersectionSphereRadius;}

  /** Returns the maximum distance of any point in the subtree to a given point
  * @param base The point to compute the distance to
  * @return The distance
  */
  virtual double getMaxDistanceTo(Vector3d& base) const;
  
  /** Checks if the object is movable or part of a movable object or
  *   part of an interactive button
  *   (needed for interaction)
  * @param nameOfObject The name of the object (if one has been found)
  * @param testClickable Flag: Also test for interactive buttons, if true
  * @return true, if the object is part of a movable object or an interactive button
  */
  virtual bool isMovableOrClickable(std::string& nameOfObject, 
                                    bool testClickable) const;

  /** Inverts the colors of the object and all subobjects
  *   Used for displaying selections.
  */
  virtual void invertColors();

  /** Set the value of visualizeSensors
  * @param visualizeSensors The new value
  */
  void setVisualizeSensors(bool visualizeSensors) {this->visualizeSensors = visualizeSensors;}

  /** Sets the color of the background of the scene
  * (The color has to be a triple of (R,G,B) with values [0.0,..,1.0]
  * @param color The color
  */
  void setBackgroundColor(const Vector3d& color) {backgroundColor = color;}

  /** Sets the color of the ambient light
  * (The color has to be a triple of (R,G,B) with values [0.0,..,1.0]
  * @param color The color
  */
  void setAmbientColor(const Vector3d& color) {ambientColor = color;}

  /** Sets the drawForCamera flag
  * @param drawForCamera The new value for drawForCamera
  */
  void setDrawForCamera(bool drawForCamera) {this->drawForCamera = drawForCamera;}

  /** Sets the invisibility flag
  * @param invisible The new value of the invisibility flag
  */
  void setInvisibility(bool invisible) {this->invisible = invisible;}

  /** Sets the standard length
  * @param standardLength
  */
  void setStandardLength(double standardLength) {this->standardLength = standardLength;}

  /** Informs the object about being clicked by the user*/
  virtual void hasBeenClicked() {};

  /** Informs the object about being released by the user*/
  virtual void hasBeenReleased() {};
};

#endif //SIMOBJECT_H_


/*
 * $Log: SimObject.h,v $
 * Revision 1.4  2004/01/15 22:43:44  roefer
 * Return references, not strings
 *
 * Revision 1.3  2003/12/09 13:40:53  roefer
 * href attribute corrected
 *
 * Revision 1.16  2003/12/09 12:38:28  roefer
 * href attribute corrected
 *
 * Revision 1.15  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.14  2003/10/23 21:30:31  tim
 * - normal computation only after rotations
 * - first steps to faster geometry rendering
 *
 * Revision 1.13  2003/10/20 17:11:18  roefer
 * Ambient light added
 *
 * Revision 1.12  2003/10/12 13:19:13  tim
 * - added interactive buttons
 *
 * Revision 1.11  2003/10/11 14:53:57  tim
 * - added standard length
 * - added invisibility for objects
 * - changed parser implementation
 *
 * Revision 1.10  2003/10/05 15:24:30  tim
 * - changed drag & drop visualization
 *
 * Revision 1.9  2003/09/25 07:54:57  roefer
 * Drag and drop/rotate in all three axes
 *
 * Revision 1.8  2003/09/18 05:27:37  roefer
 * simulationStep is an int now, so -1 is a possible value
 *
 * Revision 1.7  2003/09/18 01:52:09  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.6  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.5  2003/09/11 22:54:05  tim
 * - background color now in camera images, too
 *
 * Revision 1.4  2003/09/10 10:09:14  tim
 * - added comments
 * - small interface changes
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */