/**
 * @file SimObject.h
 * 
 * Implementation of class SimObject
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 


#include "SimObject.h"
#include "SimGeometry.h"


SimObject* SimObject::rootNode = 0;
double SimObject::stepLength = 0.0;
int SimObject::simulationStep = 0;
double SimObject::standardLength = 1.0;
bool SimObject::visualizeSensors = false;
bool SimObject::drawForCamera = false;
Vector3d SimObject::backgroundColor(0.0,0.0,0.0);
Vector3d SimObject::ambientColor(0.2,0.2,0.2);


SimObject::SimObject()
{
  name = "(no name)";
  fullName = "(no name)";
  deactivatedChildNode = 0;
  deactivatedChildNodeSuccessor = 0;
  parentNode = 0;
  invisible = false;
}

SimObject::~SimObject()
{
  ObjectList::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    delete *pos;
  }
  childNodes.clear();
}

void SimObject::draw(const Vector3d& pointOfView,
                     const VisualizationParameterSet& visParams)  
{
  if(drawForCamera || !invisible)
  {
    ObjectList::const_iterator pos;
    for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
    {
      (*pos)->draw(pointOfView, visParams);
    }
  }  
}

void SimObject::setSimulationStep(int simulationStep)
{
  SimObject::simulationStep = simulationStep;
}

void SimObject::setPosition(const Vector3d& pos)
{
  Vector3d translation = pos;
  translation -= position;
  position = pos;
  ObjectList::const_iterator listPos;
  for(listPos = childNodes.begin(); listPos != childNodes.end(); ++listPos)
  {
    (*listPos)->translate(translation);
  }
}

void SimObject::translate(const Vector3d& translation)
{
  position+=translation;
  ObjectList::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    (*pos)->translate(translation);
  }
}

void SimObject::rotate(const Vector3d& angles, const Vector3d& origin)
{
  Matrix3d matrix(angles);
  rotate(matrix, origin);
}

void SimObject::rotate(const Matrix3d& rotation, 
                       const Vector3d& origin)
{
  Vector3d v = position - origin;
  v.rotate(rotation);
  position = origin + v;
  Matrix3d dummy = rotation;
  this->rotation = dummy * (this->rotation);
  ObjectList::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    (*pos)->rotate(rotation, origin);
  }
}

void SimObject::rotateAroundAxis(double angle, Vector3d axis)
{
  Matrix3d m;
  axis.normalize();
  m.setRotationAroundAxis(axis, angle);
  rotate(m, position);
}

void SimObject::addChildNode(SimObject* child, bool move)
{
  childNodes.push_back(child);
  child->setParentNode(this);
  if(move)
  {
    child->translate(position);
    child->rotate(rotation, position);
  }
}

SimObject* SimObject::getChildNode(int num) const
{
  unsigned int num2 = num;
  if(num2 >= childNodes.size() || num2 < 0)
  {
    return 0;
  }
  else
  {
    ObjectList::const_iterator pos;
    int counter = 0;
    for(pos = childNodes.begin(); counter<num; ++pos)
    {
      ++counter;
    }
    return (*pos);
  }
}

void SimObject::deactivateNode(SimObject* node)
{
  ObjectList::iterator pos = childNodes.begin();
  while((*pos) != node)
  {
    ++pos;
  }
  deactivatedChildNode = (*pos);
  ++pos;
  if(pos==childNodes.end())
  {
    deactivatedChildNodeSuccessor = 0;
  }
  else 
  {
    deactivatedChildNodeSuccessor = (*pos);
  }
  --pos;
  childNodes.erase(pos);
}

void SimObject::findObject(SimObject*& object, const std::string& name) 
{
  object = 0;
  //The name of this object has to be the first part of the given name
  if(name.find(fullName) == 0)
  {
    if(this->fullName == name)
    {
      object = this;
    }
    else
    {
    ObjectList::const_iterator pos;
    for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
    {
      (*pos)->findObject(object, name);
      if(object)
      {
        break;
      }
    }
    }
  }
}

void SimObject::reactivateNode()
{
  if(deactivatedChildNode != 0)
  {
    if(deactivatedChildNodeSuccessor != 0)
    {
      ObjectList::iterator pos = childNodes.begin();
      while((*pos) != deactivatedChildNodeSuccessor)
      {
        ++pos;
      }
      childNodes.insert(pos,deactivatedChildNode);
      deactivatedChildNodeSuccessor = 0;
    }
    else
    {
      childNodes.push_back(deactivatedChildNode);
    }
    deactivatedChildNode = 0;
  }
}

SimObject* SimObject::clone() const
{
  SimObject* newObject = new SimObject();
  newObject->setName(name);
  newObject->setFullName(fullName);
  newObject->setPosition(position);
  newObject->rotation = rotation;
  newObject->parentNode = parentNode;
  newObject->deactivatedChildNode = deactivatedChildNode;
  newObject->deactivatedChildNodeSuccessor = deactivatedChildNodeSuccessor;
  newObject->intersectionSphereRadius = intersectionSphereRadius;
  newObject->invisible = invisible;

  newObject->childNodes.clear();
  ObjectList::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    SimObject* childNode = (*pos)->clone();
    newObject->addChildNode(childNode, false);
  }
  return newObject;
}

void SimObject::addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
                                  int depth) 
{
  ObjectDescription myDesc;
  myDesc.kind = !depth ? "scene" : getKind();
  myDesc.name = name;
  myDesc.fullName = fullName;
  myDesc.depth = depth;
  objectDescriptionTree.push_back(myDesc);
  ObjectList::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    (*pos)->addToDescriptions(objectDescriptionTree, depth + 1);
  }
}

bool SimObject::checkCollisionWithScene()
{
  //Remove object from scene:
  parentNode->deactivateNode(this);
  //Check against resulting scene:
  bool result = rootNode->checkCollisionWithObject(this);
  //Add object to scene again:
  parentNode->reactivateNode();
  return result;
}

bool SimObject::checkCollisionWithObject(SimObject* object)
{
  double dist = (position - object->position).getLength();
  if(dist < intersectionSphereRadius + object->intersectionSphereRadius)
  {
    ObjectList::const_iterator pos;
    for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
    {
      if((*pos)->checkCollisionWithObject(object))
      {
        return true;
      }
    }
    return false;
  }
  else
  {
    return false;
  }
}

void SimObject::intersectWithScene(std::vector<Intersection>& intersections)
{
}

void SimObject::intersectWithObject(std::vector<Intersection>& intersections, SimObject* object)
{
}

bool SimObject::intersect(std::vector<Intersection*>& intersections, 
                          const Vector3d& pos, const Vector3d& ray) 
{
  bool result = false;
  //Compute distance ray <-> position
  Vector3d distVec = position - pos;
  distVec = ray * distVec;
  double distance = distVec.getLength() / ray.getLength();
  //Does the ray intersect the intersectionSphere?
  if(distance <= intersectionSphereRadius)
  {
    Vector3d intersectionPos;
    if(intersectWithRay(pos, ray, intersectionPos))
    {
      result = true;
      Intersection* intersection = new Intersection();
      intersection->objectName = fullName;
      intersection->hasPosition = true;
      intersection->position = intersectionPos;
      intersections.push_back(intersection);
    }
    ObjectList::const_iterator child;
    for(child = childNodes.begin(); child != childNodes.end(); ++child)
    {
      if((*child)->intersect(intersections, pos, ray))
      {
        result = true;
      };
    }
  }
  return result;
}

void SimObject::computeIntersectionSphereRadius()
{
  intersectionSphereRadius = getMaxDistanceTo(position);
}

double SimObject::getMaxDistanceTo(Vector3d& base) const
{
  double dist = 0;
  ObjectList::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    double childDist = (*pos)->getMaxDistanceTo(base);
    if(dist < childDist)
    {
      dist = childDist;
    }
  }
  return dist;
}

bool SimObject::isMovableOrClickable(std::string& nameOfObject,
                                     bool testClickable) const
{
  if(parentNode)
  {
    return parentNode->isMovableOrClickable(nameOfObject,testClickable);
  }
  else
  {
    return false;
  }
}

void SimObject::invertColors()
{
  ObjectList::const_iterator child;
  for(child = childNodes.begin(); child != childNodes.end(); ++child)
    (*child)->invertColors();
}


/*
 * $Log: SimObject.cpp,v $
 * Revision 1.3  2003/12/09 13:40:52  roefer
 * href attribute corrected
 *
 * Revision 1.16  2003/12/09 12:38:28  roefer
 * href attribute corrected
 *
 * Revision 1.15  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.14  2003/10/23 21:30:31  tim
 * - normal computation only after rotations
 * - first steps to faster geometry rendering
 *
 * Revision 1.13  2003/10/20 17:11:18  roefer
 * Ambient light added
 *
 * Revision 1.12  2003/10/12 13:19:13  tim
 * - added interactive buttons
 *
 * Revision 1.11  2003/10/11 14:53:57  tim
 * - added standard length
 * - added invisibility for objects
 * - changed parser implementation
 *
 * Revision 1.10  2003/10/05 15:24:30  tim
 * - changed drag & drop visualization
 *
 * Revision 1.9  2003/09/25 07:54:57  roefer
 * Drag and drop/rotate in all three axes
 *
 * Revision 1.8  2003/09/18 05:27:37  roefer
 * simulationStep is an int now, so -1 is a possible value
 *
 * Revision 1.7  2003/09/18 01:52:07  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.6  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.5  2003/09/11 22:54:05  tim
 * - background color now in camera images, too
 *
 * Revision 1.4  2003/09/10 10:09:14  tim
 * - added comments
 * - small interface changes
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */