/**
 * @file SimGeometry.cpp
 * 
 * Implementation of class SimGeometry
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include "SimGeometry.h"


bool SimGeometry::pointInsidePolygon(const Vector3d& p, 
                                     const std::vector<Vector3d>& polygon,
                                     const unsigned int numberOfPoints,
                                     double maxDistance) 
{
  unsigned int point;
  //Compute a ray v to another point, which is definitely outside the polygon
  //and intersects the first side of the polygon
  Vector3d pointOnSide(polygon[0] + ((polygon[1] - polygon[0])*0.5));
  Vector3d v(pointOnSide - p);
  v.normalize();
  v *= 2.0*maxDistance;
  //Now check for another intersection of the ray with the polygon
  Vector3d p1,p2;
  for(point=1; point < numberOfPoints; point++)
  {
    p1 = (polygon[point]);
    p2 = (polygon[(point+1) % numberOfPoints]);
    if(intersectRayAndLine(p,v,p1,p2))
    {
      return false;
    }
  }
  return true;
}


bool SimGeometry::intersectRayAndPlane(const Vector3d& point, const Vector3d& v, 
                                       const Vector3d& plane, const Vector3d& n, 
                                       Vector3d& intersection) 
{
  Vector3d p(plane - point);
  double denominator(n.vecmul(v));
  if(denominator == 0.0)
  {
    return false;
  }
  double r(n.vecmul(p) / denominator);
  if(r < 0.0)
  {
    return false;
  }
  intersection = v;
  intersection *=r;
  intersection += point;
  return true;
}


bool SimGeometry::fourPointsInPlane(const Vector3d& p, const Vector3d& a,
                                    const Vector3d& b, const Vector3d& c) 
{
  Vector3d va(a-p);
  Vector3d vb(b-p);
  Vector3d pc(c-p);
  Vector3d n;
  n = va * vb;
  double value(n.vecmul(pc));
  double epsilon(0.0001);
  return (fabs(value)<epsilon);
}


bool SimGeometry::intersectRayAndLine(const Vector3d& p, const Vector3d& v,
                                      const Vector3d& p1, const Vector3d& p2)
{
  Vector3d w(p2-p1);
  Vector3d q(p1);
  Vector3d c(q-p);
  double denominator(v.v[0]*w.v[1]-v.v[1]*w.v[0]);
  double m;
  if(denominator == 0.0)
  {
    denominator = v.v[0]*w.v[2]-v.v[2]*w.v[0];
    if(denominator == 0.0)
    {
      denominator = v.v[1]*w.v[2]-v.v[2]*w.v[1];
      if(denominator == 0.0)
      {
        return false;
      }
      else
      {
        m = (c.v[1]*w.v[2] - c.v[2]*w.v[1])/denominator;
      }
    }
    else
    {
      m = (c.v[0]*w.v[2] - c.v[2]*w.v[0])/denominator;
    }
  }
  else
  {
    m = (c.v[0]*w.v[1] - c.v[1]*w.v[0])/denominator;
  }
  if((m >= 0.0) && (m < 1.0))
  {
    double n;
    if(w.v[0] != 0.0)
    {
      n = (m*v.v[0]-c.v[0]) / w.v[0];
    }
    else if(w.v[1] != 0.0)
    {
      n = (m*v.v[1]-c.v[1]) / w.v[1];
    }
    else if(w.v[2] != 0.0)
    {
      n = (m*v.v[2]-c.v[2]) / w.v[2];
    }
    else
    {
      return false;
    }
    return ((n>=0.0) && (n<=1.0));
  }
  else //Wrong side of the ray or too far away... :-(
  {
    return false;
  }
}


/*
 * $Log: SimGeometry.cpp,v $
 * Revision 1.3  2003/12/09 13:40:52  roefer
 * href attribute corrected
 *
 * Revision 1.3  2003/12/09 12:38:27  roefer
 * href attribute corrected
 *
 * Revision 1.2  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.1  2003/10/18 11:26:05  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 */