/**
 * @file Sensor.h
 * 
 * Definition of class Sensor
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef SENSORPORT_H_
#define SENSORPORT_H_

#include "Sensor.h"
#include <string>
#include <vector>

/** An identifier type for the different kinds of sensors*/
enum SensorType
{
  boolSensor,
  intSensor,
  doubleSensor,
  cameraSensor,
  doubleFieldSensor,
  viewSensor,
  noSensor
};


/**
* @class Sensorport
*
* A class representing a sensor port
*/
class Sensorport
{
private:
  /** The name of the port*/
  std::string name;
  /** The dimensions of the sensor data*/
  std::vector<int> dimensions;
  /** The type of the sensor*/
  SensorType sensorType;
  /** A pointer to the sensor*/
  Sensor* sensor;
  /** The id of the port*/
  int portId;
  /** The minimum sensor value*/
  double minValue;
  /** The maximum sensor value*/
  double maxValue;

public:
  /** Constructor
  * @param name The name of the port
  * @param portId The id of the port
  * @param sensorType The type of the sensor
  * @param sensor A pointer to the sensor
  * @param minValue The minimum sensor value
  * @param maxValue The maximum sensor value
  * @param dimensions The dimensions of the sensor data
  */
  Sensorport(const std::string& name, int portId, SensorType sensorType, 
             Sensor* sensor, double minValue, double maxValue,
             std::vector<int> dimensions)
  {
    this->sensor = sensor; 
    this->portId = portId;
    this->sensorType = sensorType;
    this->name = sensor->getFullName() + "." + name;
    this->minValue = minValue;
    this->maxValue = maxValue;
    this->dimensions = dimensions;
  }

  /** Constructor
  * Sets the dimension of the sensor data to 1
  * @param name The name of the port
  * @param portId The id of the port
  * @param sensorType The type of the sensor
  * @param sensor A pointer to the sensor
  * @param minValue The minimum sensor value
  * @param maxValue The maximum sensor value
  */
  Sensorport(const std::string& name, int portId, SensorType sensorType, 
             Sensor* sensor, double minValue, double maxValue)
  {
    this->sensor = sensor; 
    this->portId = portId;
    this->sensorType = sensorType;
    this->name = sensor->getFullName() + (name == "" ? "" : "." + name);
    this->minValue = minValue;
    this->maxValue = maxValue;
    dimensions.push_back(1);
  }

  /** Returns the name of the port
  * @return The name
  */
  const std::string& getName() const {return name;}

  /** Returns the type of the sensor
  * @return The type
  */
  SensorType getType() const {return sensorType;}

  /** Returns the dimensions of the sensor data
  * @return the dimensions
  */
  const std::vector<int>& getDimensions() const {return dimensions;}

  /** Returns a floating point sensor value
  * @return The value
  */
  void getValue(double& value) {assert(sensorType == doubleSensor); sensor->computeValue(value,portId);}

  /** Returns a boolean sensor value
  * @return The value
  */
  void getValue(bool& value) {assert(sensorType == boolSensor); sensor->computeValue(value,portId);}

  /** Returns an integer sensor value
  * @return The value
  */
  void getValue(int& value) {assert(sensorType == intSensor); sensor->computeValue(value,portId);}

  /** Returns a list of floating point sensor values
  * @return The value
  */
  void getValue(double*& value) {assert(sensorType == doubleFieldSensor); sensor->computeValue(value,portId);}

  /** Returns a list of unsigned char* sensor values (e.g. an image)
  * @return The value
  */
  void getValue(unsigned char*& value) {assert(sensorType == cameraSensor); sensor->computeValue(value,portId);}

  /** Returns the minimum sensor value
  * @return The value
  */
  double getMinValue() const {return minValue;}

  /** Returns the maximum sensor value
  * @return The value
  */
  double getMaxValue() const {return maxValue;}

  /** Returns the associated sensor
  * @return The sensor
  */
  Sensor* getSensor() {return sensor;}
};

#endif //SENSORPORT_H_

/*
 * $Log: Sensorport.h,v $
 * Revision 1.3  2004/04/09 18:55:16  roefer
 * Better structure for views
 * cp command added
 * Timing view added
 *
 * Revision 1.2  2003/12/09 13:40:52  roefer
 * href attribute corrected
 *
 * Revision 1.9  2003/12/09 12:38:27  roefer
 * href attribute corrected
 *
 * Revision 1.8  2003/10/18 20:07:30  roefer
 * DirectView added
 *
 * Revision 1.7  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.6  2003/10/17 22:16:26  roefer
 * New files in GT2004 added
 *
 * Revision 1.5  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.4  2003/09/08 11:46:54  tim
 * - added Simulation::getSensorportMinValue
 * - added Simulation::getSensorportMaxValue
 *
 * Revision 1.3  2003/09/07 17:51:08  roefer
 * Sensor and actuator view added, interface changes for simulation
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */
