/**
 * @file Sensor.h
 * 
 * Definition of class Sensor
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef SENSOR_H_
#define SENSOR_H_

#include "SimObject.h"

/**
* @class Sensor
*
* An abstract class simulating a sensor
* All sensor classes are derivated from Sensor and 
* reimplement the functions needed
*/
class Sensor : public virtual SimObject
{
protected:
  /** The number of step, the last value has been computed*/
  int lastComputationStep;
  
public:
  /** Constructor*/
  Sensor() {lastComputationStep = -1;}

  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "sensor";}

  /** Computes a floating point value
  * @param value The value
  * @param portId The id of the sensorport
  */
  virtual void computeValue(double& value, int portId) {}

  /** Computes a boolean value
  * @param value The value
  * @param portId The id of the sensorport
  */
  virtual void computeValue(bool& value, int portId) {}

  /** Computes an integer value
  * @param value The value
  * @param portId The id of the sensorport
  */
  virtual void computeValue(int& value, int portId) {}

  /** Computes an array of floating point values
  * (memory allocation is done internally, only a pointer will be returned)
  * @param value The value
  * @param portId The id of the sensorport
  */
  virtual void computeValue(double*& value, int portId) {}

  /** Computes an array of bytes (e.g. an image)
  * (memory allocation is done internally, only a pointer will be returned)
  * @param value The value
  * @param portId The id of the sensorport
  */
  virtual void computeValue(unsigned char*& value, int portId) {}
};

#endif

/*
 * $Log: Sensor.h,v $
 * Revision 1.2  2003/12/09 13:40:51  roefer
 * href attribute corrected
 *
 * Revision 1.9  2003/12/09 12:38:27  roefer
 * href attribute corrected
 *
 * Revision 1.8  2003/10/18 20:07:29  roefer
 * DirectView added
 *
 * Revision 1.7  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.6  2003/09/18 05:27:37  roefer
 * simulationStep is an int now, so -1 is a possible value
 *
 * Revision 1.5  2003/09/12 05:59:40  roefer
 * SensorportMin/MaxValue bugs fixed
 *
 * Revision 1.4  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.3  2003/09/08 11:46:54  tim
 * - added Simulation::getSensorportMinValue
 * - added Simulation::getSensorportMaxValue
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */