/**
 * @file Polyeder.cpp
 * 
 * Implementation of class Polyeder
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include "Platform/OpenGL.h"
#include "Polyeder.h"
#include "Surface.h"
#include "SimGeometry.h"


Polyeder::Polyeder():vertices(0),normalsValid(false),
                     numOfPoints(0), numOfVertices(0)
{
}


Polyeder::~Polyeder()
{
  if(vertices != 0)
  {
    delete[] vertices;
  }
  for(unsigned int i=0; i<polygonIDX.size(); i++)
  {
    delete[] polygonIDX[i];
  }
}


void Polyeder::draw(const Vector3d& pointOfView,
                    const VisualizationParameterSet& visParams) 
{
  if(invisible && !drawForCamera)
  {
    return;
  }
  Vector3d color(surface->getColor(inverted && !drawForCamera));
  glColor3f((float) color.v[0],(float) color.v[1],(float) color.v[2]);
  glEnableClientState(GL_VERTEX_ARRAY);
  glVertexPointer(3, GL_DOUBLE,0, vertices);
  if(visParams.surfaceStyle == VisualizationParameterSet::SMOOTH_SHADING)
  {
    updateNormals();
  }
  for(unsigned int polygon=0; polygon < numOfPolygons; polygon++)
  {
    if(visParams.surfaceStyle == VisualizationParameterSet::SMOOTH_SHADING)
    {
      const Vector3d& n = polygonNormals[polygon];
      glNormal3d(n.v[0], n.v[1], n.v[2]);
    }   
    if(visParams.surfaceStyle == VisualizationParameterSet::WIREFRAME)
    {
      glDrawElements(GL_LINE_LOOP,numOfPointsPerPolygon[polygon], 
                     GL_UNSIGNED_INT, polygonIDX[polygon]);
    }
    else
    {
      glDrawElements(GL_POLYGON,numOfPointsPerPolygon[polygon], 
                     GL_UNSIGNED_INT, polygonIDX[polygon]);
    }
  }
  SimObject::draw(pointOfView, visParams);
}


void Polyeder::setPointsAndSurfaces(const std::vector<Vector3d>& points, 
                                    const std::vector< std::vector<unsigned int> >& surfaces)
{
  //Set all vertices
  numOfPoints = points.size();
  numOfVertices = numOfPoints*3;
  vertices = new double[numOfVertices];
  for(unsigned int v=0; v < points.size(); v++)
  {
    Vector3d point(points[v]);
    point.rotate(rotation);
    point += position;
    setVerticesFromVector(v*3, point);
  }
  
  //Store the sizes of the polygons and init polygon index lists
  numOfPointsPerPolygon.clear();
  numOfPointsPerPolygon.resize(surfaces.size());
  polygonIDX.clear();
  polygonIDX.resize(surfaces.size());
  maxNumOfPointsPerPolygon=0;
  for(unsigned int surface=0; surface<surfaces.size(); ++surface)
  {
    unsigned int pointsInPolygon(surfaces[surface].size());
    if(pointsInPolygon>maxNumOfPointsPerPolygon)
    {
      maxNumOfPointsPerPolygon = pointsInPolygon;
    }
    numOfPointsPerPolygon[surface] = pointsInPolygon;
    polygonIDX[surface] = new unsigned int[pointsInPolygon];
    for(unsigned int i=0; i<pointsInPolygon; i++)
    {
      polygonIDX[surface][i] = surfaces[surface][i];
    }
  }
  numOfPolygons = polygonIDX.size();
  intersectionBuffer.clear();
  intersectionBuffer.resize(maxNumOfPointsPerPolygon);

  //Compute the intersection sphere for each surface:
  intersectionSpheres.clear();
  intersectionSpheres.resize(numOfPolygons);
  for(unsigned int polygon=0; polygon<numOfPolygons; polygon++)
  {
    double radius(0);
    Vector3d vec0;
    getVectorFromVertices(polygonIDX[polygon][0]*3, vec0);
    for(unsigned int point=1; point < numOfPointsPerPolygon[polygon]; point++)
    {
      Vector3d vec;
      getVectorFromVertices(polygonIDX[polygon][point]*3, vec);
      double dist((vec-vec0).getLength());
      if(dist>radius)
      {
        radius = dist;
      }
    }
    intersectionSpheres[polygon] = radius;
  }

  //Initialize polygon normals
  polygonNormals.clear();
  polygonNormals.resize(numOfPolygons);
  normalsValid=false;
  updateNormals();
}


void Polyeder::setPosition(const Vector3d& pos)
{
  Vector3d translation(pos - position);
  position = pos;
  for(unsigned int v=0; v<numOfVertices; ++v)
  {
    vertices[v] += translation.v[0];
    vertices[++v] += translation.v[1];
    vertices[++v] += translation.v[2];
  }
}


void Polyeder::translate(const Vector3d& translation)
{
  for(unsigned int v=0; v<numOfVertices; ++v)
  {
    vertices[v] += translation.v[0];
    vertices[++v] += translation.v[1];
    vertices[++v] += translation.v[2];
  }
  SimObject::translate(translation);
}


void Polyeder::rotate(const Matrix3d& rotation, 
                      const Vector3d& origin)
{
  position -= origin;
  position.rotate(rotation);
  position += origin;
  Matrix3d newRotation(rotation);
  newRotation *= this->rotation;
  this->rotation = newRotation;
  for(unsigned int i=0; i<numOfPoints; i++)
  {
    unsigned int pos(i*3);
    Vector3d point;
    getVectorFromVertices(pos, point);
    point -= origin;
    point.rotate(rotation);
    point += origin;
    setVerticesFromVector(pos,point);
  }
  normalsValid=false;
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    (*pos)->rotate(rotation, origin);
  }
}


SimObject* Polyeder::clone() const
{
  Polyeder* newPoly = new Polyeder();
  newPoly->setName(name);
  newPoly->setFullName(fullName);
  newPoly->position = position;
  newPoly->rotation = rotation;
  newPoly->surface = surface;
  newPoly->invisible = invisible;
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    SimObject* childNode = (*pos)->clone();
    newPoly->addChildNode(childNode, false);
  }
  newPoly->intersectionSpheres = intersectionSpheres;
  newPoly->intersectionBuffer = intersectionBuffer;
  newPoly->polygonNormals = polygonNormals;
  newPoly->normalsValid = false;
  newPoly->numOfPointsPerPolygon = numOfPointsPerPolygon;
  newPoly->maxNumOfPointsPerPolygon = maxNumOfPointsPerPolygon;;
  newPoly->numOfPoints = numOfPoints;
  newPoly->numOfVertices = numOfVertices;
  newPoly->numOfPolygons = numOfPolygons;
  newPoly->vertices = new double[numOfVertices];
  memcpy(newPoly->vertices, vertices, sizeof(double)*numOfVertices);
  newPoly->polygonIDX = polygonIDX;
  for(unsigned int i=0; i<polygonIDX.size(); i++)
  {
    newPoly->polygonIDX[i] = new unsigned int[numOfPointsPerPolygon[i]];
    for(unsigned int j=0; j<numOfPointsPerPolygon[i]; j++)
    {
      newPoly->polygonIDX[i][j] = polygonIDX[i][j];
    }
  }
  SimObject* newObject = newPoly;
  return newObject;
}


double Polyeder::getMaxDistanceTo(Vector3d& base) const
{
  Vector3d vec(base - position);
  double vecLen(vec.getQuadLength());
  double dist(vecLen);
  for(unsigned int i=0; i < numOfVertices; i+=3)
  {
    Vector3d currentVec;
    getVectorFromVertices(i, currentVec);
    vec = base - currentVec;
    vecLen = vec.getQuadLength();
    if(vecLen > dist)
    {
      dist = vecLen;
    }
  }
  dist = sqrt(dist);
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    double childDist = (*pos)->getMaxDistanceTo(base);
    if(dist < childDist)
    {
      dist = childDist;
    }
  }
  return dist;
}


bool Polyeder::intersectWithRay(const Vector3d& pos, 
                                const Vector3d& ray, 
                                Vector3d& intersectPos) 
{
  bool result = false;
  std::vector<Vector3d> pointList;
  Vector3d intersectionPos;
  updateNormals();
  for(unsigned int polygon = 0; polygon < numOfPolygons; polygon++)
  {
    Vector3d planePos;
    getVectorFromVertices(polygonIDX[polygon][0]*3, planePos);
    Vector3d n(polygonNormals[polygon]);
    if(SimGeometry::intersectRayAndPlane(pos, ray, planePos, n, intersectionPos))
    {
      if((intersectionPos-planePos).getLength() < intersectionSpheres[polygon])
      {
        copyPolygonToIntersectionBuffer(polygon);
        if(SimGeometry::pointInsidePolygon(intersectionPos, 
                                           intersectionBuffer,
                                           numOfPointsPerPolygon[polygon],
                                           intersectionSpheres[polygon]))
        {
          pointList.push_back(intersectionPos);
          result = true;
        }
      }
    }
  }
  //find nearest intersection
  if(result)
  {
    unsigned int shortestIdx = 0;
    Vector3d distVec = pointList[shortestIdx] - pos;
    double minQuadDist = distVec.getQuadLength();
    for(unsigned int i = 1; i < pointList.size(); i++)
    {
      distVec = pointList[i] - pos;
      double currentQuadDist = distVec.getQuadLength();
      if(currentQuadDist < minQuadDist)
      {
        minQuadDist = currentQuadDist;
        shortestIdx = i;
      }
    }
    intersectPos = pointList[shortestIdx];
  }
  return result;
}


void Polyeder::updateNormals()
{
  if(!normalsValid)
  {
    Vector3d crossProd;
    for(unsigned int i=0; i<numOfPolygons; ++i)
    {
      Vector3d vec0, vec1, vec2;
      getVectorFromVertices(polygonIDX[i][0]*3, vec0);
      getVectorFromVertices(polygonIDX[i][1]*3, vec1);
      getVectorFromVertices(polygonIDX[i][2]*3, vec2);
      crossProd = ((vec1 - vec0) * (vec2 - vec0));
      crossProd.normalize();
      polygonNormals[i] = crossProd;
    }
    normalsValid = true;
  }
}


inline void Polyeder::getVectorFromVertices(unsigned int pos,
                                            Vector3d& vec) const
{
  memcpy(&vec, &vertices[pos], sizeof(double)*3);
}


inline void Polyeder::setVerticesFromVector(unsigned int pos,
                                            const Vector3d& vec)
{
  memcpy(&vertices[pos], &vec, sizeof(double)*3);
}


inline void Polyeder::copyPolygonToIntersectionBuffer(unsigned int polygon)
{
  for(unsigned int i=0; i<numOfPointsPerPolygon[polygon]; i++)
  {
    getVectorFromVertices(polygonIDX[polygon][i]*3, intersectionBuffer[i]);
  }
}


/*
 * $Log: Polyeder.cpp,v $
 * Revision 1.3  2003/12/09 13:40:51  roefer
 * href attribute corrected
 *
 * Revision 1.18  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.17  2003/12/03 18:12:37  roefer
 * Compatibility with VC2003.NET, GUI does still not work completely there
 *
 * Revision 1.16  2003/10/26 14:47:03  tim
 * - fixed cylinder backtransformation problem
 * - fixed wireframe polyeder drawing problem
 *
 * Revision 1.15  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.14  2003/10/23 22:19:59  roefer
 * Minimal change
 *
 * Revision 1.13  2003/10/23 21:30:31  tim
 * - normal computation only after rotations
 * - first steps to faster geometry rendering
 *
 * Revision 1.12  2003/10/18 20:05:57  roefer
 * Bug fixed, rotation was ignored
 *
 * Revision 1.11  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.10  2003/10/11 14:53:57  tim
 * - added standard length
 * - added invisibility for objects
 * - changed parser implementation
 *
 * Revision 1.9  2003/10/05 15:24:30  tim
 * - changed drag & drop visualization
 *
 * Revision 1.8  2003/09/18 10:42:17  tim
 * - clean up
 *
 * Revision 1.7  2003/09/18 02:09:09  tim
 * - commented out some code
 *
 * Revision 1.6  2003/09/18 01:52:07  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.5  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.4  2003/09/11 22:10:57  tim
 * - added new zoom functions
 * - added changing of opening angle
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */

