/**
 * @file MovableObject.h
 * 
 * Definition of class MovableObject
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef MOVABLEOBJECT_H_
#define MOVABLEOBJECT_H_

#include "Actuator.h"
#include "Sensor.h"

/**
* @class MovableObject
*
* A class representing a movable object
*/
class MovableObject : public Actuator, public Sensor
{
protected:
  /** An internal type for assigning names to ports*/
  enum Ports
  {
    xSpeed = 0,
      ySpeed,
      zSpeed,
      xRotation,
      yRotation,
      zRotation
  };
  /** Data about the speed in x-direction*/
  ActuatorportData speedX;
  /** Data about the speed in y-direction*/
  ActuatorportData speedY;
  /** Data about the speed in z-direction*/
  ActuatorportData speedZ;
  /** Data about the rotation speed around the x-axis*/
  ActuatorportData rotationSpeedX;
  /** Data about the rotation speed around the y-axis*/
  ActuatorportData rotationSpeedY;
  /** Data about the rotation speed around the z-axis*/
  ActuatorportData rotationSpeedZ;
  
public:
  /** Constructor*/
  MovableObject();

  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "movable";}
  
  /** Adds the object to some internal lists
  * @param sensorportList A list of all sensor ports in the scene
  * @param actuatorportList A list of all actuator ports in the scene
  * @param actuatorList A list of all actuators in the scene
  */
  virtual void addToLists(std::vector<Sensorport*>& sensorportList,
                          std::vector<Actuatorport*>& actuatorportList,
                          std::vector<Actuator*>& actuatorList);

  /** Adds descriptions of the object and its children to a list
  * @param objectDescriptionTree The list of descriptions
  * @param The depth of the object in the object tree
  */
  virtual void addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree, 
                                 int depth);

  /** Sets the value of one actuatorport
  * @param value The value
  * @param port The port
  */
  virtual void setValue(double value, int port);

  /** Returns the value of a specified sensorport
  * @param value The value
  * @param portId The id of the sensorport
  */
  virtual void computeValue(double& value, int portId);

  /** Let the actuator do what it is supposed to do
   * @param initial Is this the initial step?
   */
  virtual void act(bool initial);

   /** Clones the object and its child nodes
  * @return A pointer to a copy of the object
  */
  virtual SimObject* clone() const;

  /** Returns true, since this object is movable
  * @param nameOfObject The name of this object will be copied to this variable
  * @param testClickable Flag: Also test for interactive buttons, if true
  * @return Always true
  */
  virtual bool isMovableOrClickable(std::string& nameOfObject,
                                    bool testClickable) const;
  
  /** Sets the minimum speed in x-direction
  * @param v The speed
  */
  void setSpeedXMin(double v) {speedX.min =v;}

  /** Sets the maximum speed in x-direction
  * @param v The speed
  */
  void setSpeedXMax(double v) {speedX.max =v;}

  /** Sets the minimum speed in y-direction
  * @param v The speed
  */
  void setSpeedYMin(double v) {speedY.min =v;}

  /** Sets the maximum speed in y-direction
  * @param v The speed
  */
  void setSpeedYMax(double v) {speedY.max =v;}

  /** Sets the minimum speed in z-direction
  * @param v The speed
  */
  void setSpeedZMin(double v) {speedZ.min =v;}

  /** Sets the maximum speed in z-direction
  * @param v The speed
  */
  void setSpeedZMax(double v) {speedZ.max =v;}

  /** Sets the minimum speed for x-rotation
  * @param v The speed
  */
  void setRotationSpeedXMin(double v) {rotationSpeedX.min =v;}

  /** Sets the maximum speed for x-rotation
  * @param v The speed
  */
  void setRotationSpeedXMax(double v) {rotationSpeedX.max =v;}

  /** Sets the minimum speed for y-rotation
  * @param v The speed
  */
  void setRotationSpeedYMin(double v) {rotationSpeedY.min =v;}

  /** Sets the maximum speed for y-rotation
  * @param v The speed
  */
  void setRotationSpeedYMax(double v) {rotationSpeedY.max =v;}

  /** Sets the minimum speed for z-rotation
  * @param v The speed
  */
  void setRotationSpeedZMin(double v) {rotationSpeedZ.min =v;}

  /** Sets the maximum speed in z-direction
  * @param v The speed
  */
  void setRotationSpeedZMax(double v) {rotationSpeedZ.max =v;}
};

#endif	//MOVABLEOBJECT_H_

/*
 * $Log: MovableObject.h,v $
 * Revision 1.2  2003/12/09 13:40:50  roefer
 * href attribute corrected
 *
 * Revision 1.7  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.6  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.5  2003/10/12 13:19:13  tim
 * - added interactive buttons
 *
 * Revision 1.4  2003/09/28 14:50:04  roefer
 * Planes changed, initialValue for joints added
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */