/**
 * @file Cylinder.h
 * 
 * Definition of class Cylinder
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef CYLINDER_H_
#define CYLINDER_H_

#include "DrawableObject.h"


/**
* @class Cylinder
*
* A class representing a cylinder in 3D space.
*/
class Cylinder : public DrawableObject
{
protected:
  /** The radius of the cylinder*/
  double radius;
  /** The height of the cylinder*/
  double height;
  /** Flag: The cylinder is closed, if true*/
  bool closed;
  
  /** Intersects the cylinder and its children with a ray
  * @param pos The starting position of the ray
  * @param ray The ray
  * @param intersectPos The intersection point nearest to pos (if found)
  * @return true, if an intersection point has been found
  */
  virtual bool intersectWithRay(const Vector3d& pos, const Vector3d& ray, Vector3d& intersectPos);

public:
  /** Constructor*/
  Cylinder();

  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "cylinder";}

  /** Draws the cylinder and its child nodes
  * @param pointOfView The point of view from which the scene is drawn
  * @param visParams Parameters for Visualization
  */
  virtual void draw(const Vector3d& pointOfView,
                    const VisualizationParameterSet& visParams);

  /** Sets the radius of the cylinder
  * @param radius The radius
  */
  void setRadius(double radius) {this->radius = radius;}

  /** Sets the height of the cylinder
  * @param height The height
  */
  void setHeight(double height) {this->height = height;};

  /** Sets the closed flag
  * @param closed The closed flag
  */
  void setClosed(bool closed) {this->closed = closed;}

  /** Clones the object and its child nodes
  * @return A pointer to a copy of the object
  */
  virtual SimObject* clone() const;

  /** Returns the maximum distance of any point in the subtree to a given point
  * @param base The point to compute the distance to
  * @return The distance
  */
  virtual double getMaxDistanceTo(Vector3d& base) const;
};

#endif //CYLINDER_H_

/*
 * $Log: Cylinder.h,v $
 * Revision 1.3  2003/12/09 13:40:50  roefer
 * href attribute corrected
 *
 * Revision 1.7  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.6  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.5  2003/09/18 01:52:06  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.4  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.3  2003/09/08 22:32:07  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */