/**
 * @file APIDatatypes.h
 * 
 * Definition of types used for interfacing SimRobotCore
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef API_DATATYPES_H_
#define API_DATATYPES_H_


#include <string>

/** A type for distinguishing between several types of objects*/
enum ObjectType 
{
  OBJECT_TYPE_NONE,
  OBJECT_TYPE_OBJECT,
  OBJECT_TYPE_SENSORPORT, 
  OBJECT_TYPE_ACTUATORPORT
};

/** Describes the current status of the simulation*/
enum SimulationStatus 
{
  NOT_LOADED, 
  SCENE_LOADED, 
  SIM_RUNNING,
  SIM_PAUSE
};

/** Specifies the current plane for Drag&Drop translations*/
enum DragAndDropPlane 
{
  XY_PLANE,
  XZ_PLANE,
  YZ_PLANE
};

/** Used for selection of zoom types*/
enum ZoomFitType
{
  ZOOM_FIT=0, ZOOM_FIT_WIDTH, ZOOM_FIT_HEIGHT
};

/** Specifies the result of an interactive selection*/
enum InteractiveSelectionType
{
  NO_SELECTION=0, BUTTON_SELECTION, OBJECT_SELECTION
};

/** The set of parameters for visualizing the scene*/
class VisualizationParameterSet
{
public:
  enum SurfaceStyle {WIREFRAME = 0, FLAT_SHADING, SMOOTH_SHADING} surfaceStyle;

  /** Constructor*/
  VisualizationParameterSet():surfaceStyle(SMOOTH_SHADING) {}
};

/** 
* @class ObjectDescription
* Describes an object 
* (used for displaying information in a GUI)
*/
class ObjectDescription
{
public:
  /** Constructor*/
  ObjectDescription()
  {kind = ""; name = ""; fullName = ""; type = OBJECT_TYPE_OBJECT; depth=0;}
  
  /** Copy-Constructor
  * @param other Another ObjectDesciption
  */
  ObjectDescription(const ObjectDescription& other)
  {*this = other;}
  
  /** Copy-Operator
  * @param other Another ObjectDesciption
  */
  void operator =	(const ObjectDescription& other)
  {kind = other.kind; name = other.name; fullName = other.fullName; type = other.type; depth = other.depth;}
  
  /** A description of the kind of the object*/
  std::string kind;
  /** The name*/
  std::string name;
  /** The name together with the full path in the object tree*/
  std::string fullName;
  /** The type of the object*/
  ObjectType type;
  /** The depth in the object tree (useful for displaying in a tree)*/
  int depth;
};

/**
* @class ErrorDescription
* Describes an error
*/
class ErrorDescription
{
public:
  /** The title of the error (very short description)*/
  std::string title;
  /** The complete error message*/
  std::string text;
  /** Flag: true, if the position of the error is known*/
  bool position;
  /** The line in the XML file in which the error occured.
  * Only used if position is true
  */
  int line;
  /** The column in the XML file in which the error occured
  * Only used if position is true
  */
  int column;
};


#endif //API_DATATYPES_H_

/*
 * $Log: APIDatatypes.h,v $
 * Revision 1.2  2003/12/09 13:40:49  roefer
 * href attribute corrected
 *
 * Revision 1.10  2003/12/09 12:38:25  roefer
 * href attribute corrected
 *
 * Revision 1.9  2003/10/12 13:19:13  tim
 * - added interactive buttons
 *
 * Revision 1.8  2003/09/28 14:50:04  roefer
 * Planes changed, initialValue for joints added
 *
 * Revision 1.7  2003/09/11 22:10:57  tim
 * - added new zoom functions
 * - added changing of opening angle
 *
 * Revision 1.6  2003/09/08 22:32:07  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.5  2003/09/08 14:46:59  tim
 * - added surface style selection
 *
 * Revision 1.4  2003/09/08 12:34:26  tim
 * - added comments
 * - added simple VisualizationParameterSet
 *
 * Revision 1.3  2003/09/05 08:46:04  roefer
 * ObjectView added
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */
