// srObject.cpp : implementation file
//

#include "StdAfx.h"
#include "SimRobXP.h"
#include "srDoc.h"
#include "srObject.h"
#include "srBar.h"
#include "srConsole.h"
#include "srFrame.h"
#include "srObjectFrame.h"
#include <gl/glu.h>
#include "Platform/OffScreenRenderer.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CObjectView

IMPLEMENT_DYNCREATE(CObjectView, CView)

void CObjectView::WriteLayout()
{
  ((CObjectFrame*) GetParent())->WriteLayout();
  CDoc* pDoc = GetDocument();
  pDoc->NewSection("object");
  pDoc->WriteString("object",m_sObject);
  pDoc->WriteDouble("zoomDistance",zoomDistance);
  pDoc->WriteInt("distortion",verticalOpeningAngleIdx);
  pDoc->WriteInt("surfaceStyle", (int)(visualizationParameters.surfaceStyle));
  pDoc->WriteInt("showSensors", showSensors ? 1 : 0);
  pDoc->WriteInt("plane", (int) m_plane);
}

bool CObjectView::RestoreLayout()
{
  CDoc* pDoc = GetDocument();
  if(!pDoc->IsRestoringLayout())
    return false;
  ((CObjectFrame*) GetParent())->RestoreLayout();
  pDoc->NewSection("object");
  m_sObject = pDoc->ReadString("object");
  verticalOpeningAngleIdx = pDoc->ReadInt("distortion");
  if(verticalOpeningAngleIdx == 0)
    verticalOpeningAngleIdx = 3;
  m_nDL = pDoc->ReadInt("detailLevel");
  visualizationParameters.surfaceStyle = 
    (VisualizationParameterSet::SurfaceStyle)(pDoc->ReadInt("surfaceStyle"));
  zoomDistance = pDoc->ReadDouble("zoomDistance");
  showSensors = pDoc->ReadInt("showSensors") != 0;
  if(zoomDistance == 0.0)
    computeZoom(ZOOM_FIT);
  m_plane = DragAndDropPlane(pDoc->ReadInt("plane"));
  return true;
}

void CObjectView::computeZoom(ZoomFitType zoomFitType)
{
  if(m_sObject != "")
  {
    CRect rect;
    GetClientRect(&rect);
    Simulation* sim = GetDocument()->GetSimulation();
    double fovy((verticalOpeningAngleIdx)*20.0);
    zoomDistance = sim->getOptimalZoomForObject(zoomFitType, rect.right, rect.bottom, 
                                                fovy, (const char*)m_sObject);
  }
}


BEGIN_MESSAGE_MAP(CObjectView, CView)
  //{{AFX_MSG_MAP(CObjectView)
  ON_UPDATE_COMMAND_UI(ID_DIST0, OnUpdateDist)
  ON_WM_RBUTTONDOWN()
  ON_WM_RBUTTONUP()
  ON_COMMAND(ID_DIST0, OnDist)
  ON_COMMAND(ID_ZOOM_IN, OnZoom)
  ON_COMMAND(ID_SURFACE_WIRE, OnSelectSurfaceStyle)
  ON_UPDATE_COMMAND_UI(ID_SURFACE_WIRE, OnUpdateSurfaceStyle)
  ON_COMMAND(ID_DETAIL_SENSORS, OnShowSensors)
  ON_UPDATE_COMMAND_UI(ID_DETAIL_SENSORS, OnUpdateShowSensors)
  ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
  ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateEditCopy)
  ON_WM_ERASEBKGND()
  ON_WM_LBUTTONDOWN()
  ON_WM_MOUSEMOVE()
  ON_WM_LBUTTONUP()
  ON_WM_DESTROY()
	ON_WM_CREATE()
	ON_WM_TIMER()
	ON_COMMAND(ID_PLANE_XY, OnPlaneXY)
	ON_UPDATE_COMMAND_UI(ID_PLANE_XY, OnUpdatePlaneXY)
	ON_COMMAND(ID_PLANE_XZ, OnPlaneXZ)
	ON_UPDATE_COMMAND_UI(ID_PLANE_XZ, OnUpdatePlaneXZ)
	ON_COMMAND(ID_PLANE_YZ, OnPlaneYZ)
	ON_UPDATE_COMMAND_UI(ID_PLANE_YZ, OnUpdatePlaneYZ)
  ON_WM_LBUTTONDOWN()
  ON_COMMAND(ID_ZOOM_OUT, OnZoom)
  ON_COMMAND(ID_ZOOM_FIT, OnZoom)
  ON_COMMAND(ID_ZOOM_FIT_WIDTH, OnZoom)
  ON_COMMAND(ID_ZOOM_FIT_HEIGHT, OnZoom)
  ON_COMMAND(ID_DIST1, OnDist)
  ON_COMMAND(ID_DIST2, OnDist)
  ON_COMMAND(ID_DIST3, OnDist)
  ON_COMMAND(ID_DIST4, OnDist)
  ON_COMMAND(ID_DIST5, OnDist)
  ON_UPDATE_COMMAND_UI(ID_DIST1, OnUpdateDist)
  ON_UPDATE_COMMAND_UI(ID_DIST2, OnUpdateDist)
  ON_UPDATE_COMMAND_UI(ID_DIST3, OnUpdateDist)
  ON_UPDATE_COMMAND_UI(ID_DIST4, OnUpdateDist)
  ON_UPDATE_COMMAND_UI(ID_DIST5, OnUpdateDist)
  ON_COMMAND(ID_SURFACE_FLAT, OnSelectSurfaceStyle)
  ON_COMMAND(ID_SURFACE_SMOOTH, OnSelectSurfaceStyle)
  ON_UPDATE_COMMAND_UI(ID_SURFACE_FLAT, OnUpdateSurfaceStyle)
  ON_UPDATE_COMMAND_UI(ID_SURFACE_SMOOTH, OnUpdateSurfaceStyle)
	ON_WM_LBUTTONDBLCLK()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CObjectView drawing

void CObjectView::DoDraw(const CRect& rect)
{
  double fovy((verticalOpeningAngleIdx)*20.0);
  GetDocument()->GetSimulation()->setVisualizeSensors(showSensors);
  GetDocument()->GetSimulation()->draw(GetParent()->GetScrollPos(SB_VERT) + 90, 0,
                                       GetParent()->GetScrollPos(SB_HORZ), 
                                       zoomDistance, rect.right, rect.bottom, fovy,
                                       visualizationParameters,
                                       (const char*) m_sObject);
  glFinish();
}

void CObjectView::OnDraw(CDC* pDC)
{
  CRect rect;
  GetClientRect(&rect);
  if(m_bValid)
  {
    wglMakeCurrent(m_hDC,m_hGLContext);
    DoDraw(rect);
    SwapBuffers(pDC->m_hDC);
    wglMakeCurrent(NULL,NULL);
  }
  else
    pDC->FillSolidRect(&rect,0xffffff);
}

/////////////////////////////////////////////////////////////////////////////
// CObjectView diagnostics

#ifdef _DEBUG
void CObjectView::AssertValid() const
{
  CView::AssertValid();
}

void CObjectView::Dump(CDumpContext& dc) const
{
  CView::Dump(dc);
}

CDoc* CObjectView::GetDocument() // non-debug version is inline
{
  ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CDoc)));
  return (CDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CObjectView message handlers

void CObjectView::OnInitialUpdate()
{
  CObjectFrame* pFrame = (CObjectFrame*) GetParent();
  pFrame->SetObject(this);
  if(!RestoreLayout())
  {
    m_sObject = GetDocument()->GetSelectedObject();
    verticalOpeningAngleIdx = 3;
    computeZoom(ZOOM_FIT);
    m_nDL = 9;
    visualizationParameters.surfaceStyle = VisualizationParameterSet::SMOOTH_SHADING;
    showSensors = false;
    m_plane = XY_PLANE;
  }
  m_bValid = false;
  pFrame->SetWindowText(CString("Object - ") +  m_sObject);
  m_menuPopup.LoadMenu(IDP_OBJECT);
  CView::OnInitialUpdate();
}

void CObjectView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint)
{
  if(lHint != UPDATE_ACTORS)
  {
    if(lHint != UPDATE_SENSORS)
      m_bValid = GetDocument()->GetType(m_sObject) == OBJECT_TYPE_OBJECT;
    Invalidate(false);
  }
}

void CObjectView::OnZoom() 
{
  if(m_sObject != "")
  {
    int zoomIndex(LOWORD(GetCurrentMessage()->wParam) - ID_ZOOM_IN);
    switch(zoomIndex)
    {
    case 0: zoomDistance*=0.8; break;
    case 1: zoomDistance/=0.8; break;
    default:
      ZoomFitType zoomType((ZoomFitType)(zoomIndex-2));
      computeZoom(zoomType);
      break;
    }
    Invalidate(false);
  }
}

void CObjectView::OnDist() 
{
  if(m_sObject != "")
  {
    double oldAlpha((verticalOpeningAngleIdx * 10.0) * M_PI / 180.0);
    double oldZoom(zoomDistance);
    verticalOpeningAngleIdx = LOWORD(GetCurrentMessage()->wParam) - ID_DIST0 + 1;
    double r(tan(oldAlpha)*oldZoom);
    double newAlpha((verticalOpeningAngleIdx * 10.0) * M_PI / 180.0);
    zoomDistance = r / tan(newAlpha);
    Invalidate(false);
  }
}

void CObjectView::OnUpdateDist(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sObject != "");
  pCmdUI->SetCheck((int) pCmdUI->m_nID == ID_DIST0 + verticalOpeningAngleIdx -1);
}

void CObjectView::OnEditCopy() 
{
  if(m_sObject != "")
  {
    CRect rect;
    GetClientRect(&rect);
    OffscreenRenderer osRenderer;
    char* image = new char[rect.right * rect.bottom * 3];
    osRenderer.prepareRendering(image, rect.right, rect.bottom);
    DoDraw(rect);
    osRenderer.finishRendering(image, rect.right, rect.bottom);

    GLOBALHANDLE hGMem = GlobalAlloc(GHND,sizeof(BITMAPINFOHEADER) + 
                                          ((rect.right * 3 + 3) & 0xfffffffc) * rect.bottom);
    BITMAPINFOHEADER* bmiHeader = (BITMAPINFOHEADER*) GlobalLock(hGMem);
    memset(bmiHeader, 0, sizeof(BITMAPINFO));
	  bmiHeader->biSize = sizeof(BITMAPINFOHEADER);
	  bmiHeader->biWidth = rect.right;
	  bmiHeader->biHeight = rect.bottom;
	  bmiHeader->biPlanes = 1;
	  bmiHeader->biBitCount = 24;
	  bmiHeader->biCompression	= BI_RGB;

    char* pSrc = image,
        * pDest = (char*)(bmiHeader + 1) + ((rect.right * 3 + 3) & 0xfffffffc) * rect.bottom;
    for(int i = 0; i < rect.bottom; ++i)
    {
      pDest -= (rect.right * 3 + 3) & 0xfffffffc;
      memcpy(pDest, pSrc, rect.right * 3);
      pSrc += rect.right * 3;
    }

    GlobalUnlock(hGMem);
    delete [] image;

    OpenClipboard();
    EmptyClipboard();
    SetClipboardData(CF_DIB,hGMem);
    CloseClipboard();
  }
}

void CObjectView::OnUpdateEditCopy(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sObject != "");
}

BOOL CObjectView::OnEraseBkgnd(CDC* pDC) 
{
  return true;
}

void CObjectView::OnActivateView(BOOL bActivate, CView* pActivateView, CView* pDeactiveView) 
{
  ((CFrame*) AfxGetMainWnd())->m_wndToolBar.SetObject(bActivate ? this : 0);
  CView::OnActivateView(bActivate, pActivateView, pDeactiveView);
}

void CObjectView::OnLButtonDown(UINT nFlags, CPoint point) 
{
  if(m_sObject != "" && !GetCapture())
  {
    CRect rect;
    GetClientRect(&rect);
    double fovy((verticalOpeningAngleIdx)*20.0);
    interactiveSelection = GetDocument()->GetSimulation()->selectObject(point.x,point.y,
                                                    GetParent()->GetScrollPos(SB_VERT) + 90, 0,
                                                    GetParent()->GetScrollPos(SB_HORZ), 
                                                    zoomDistance, rect.right, rect.bottom, fovy, m_plane,
                                                    (const char*) m_sObject, true);
    if(interactiveSelection != NO_SELECTION)
    {
      m_bDrag = (interactiveSelection == OBJECT_SELECTION);
      selectedObject = GetDocument()->GetSimulation()->getSelectedObject();
      SetCapture();
      GetDocument()->UpdateAllViews(0,UPDATE_OBJECTS,0);
    }
  }
}

void CObjectView::OnMouseMove(UINT nFlags, CPoint point) 
{
  if(GetCapture() == this)
  {
    CRect rect;
    GetClientRect(&rect);
    double fovy((verticalOpeningAngleIdx)*20.0);
    if(interactiveSelection == OBJECT_SELECTION)
    {
      if(m_bDrag)
        GetDocument()->GetSimulation()->translateObject(point.x,point.y);
      else
        GetDocument()->GetSimulation()->rotateObject(point.x,point.y);
    }
    GetDocument()->UpdateAllViews(0,UPDATE_OBJECTS,0);
  }
}

void CObjectView::OnLButtonUp(UINT nFlags, CPoint point) 
{
  if(GetCapture() == this && m_sObject != "")
  {
    if((interactiveSelection == OBJECT_SELECTION)  && m_bDrag)
    {
      CRect rect;
      GetClientRect(&rect);
      double fovy((verticalOpeningAngleIdx)*20.0);
      GetDocument()->GetSimulation()->translateObject(point.x,point.y);
    }
    GetDocument()->GetSimulation()->unselectObject();
    ReleaseCapture();
    GetDocument()->UpdateAllViews(0,UPDATE_OBJECTS,0);
  }
}


void CObjectView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
  if(interactiveSelection != NO_SELECTION)
    GetDocument()->OnSelected(selectedObject);
}

void CObjectView::OnRButtonDown(UINT nFlags, CPoint point) 
{
  if(m_sObject != "" && !GetCapture())
  {
    CRect rect;
    GetClientRect(&rect);
    double fovy((verticalOpeningAngleIdx)*20.0);
    interactiveSelection = GetDocument()->GetSimulation()->selectObject(point.x,point.y,
                                                    GetParent()->GetScrollPos(SB_VERT) + 90, 0,
                                                    GetParent()->GetScrollPos(SB_HORZ), 
                                                    zoomDistance, rect.right, rect.bottom, fovy, m_plane,
                                                    (const char*) m_sObject);
    if(interactiveSelection != NO_SELECTION)
    {
      m_bDrag = false;
      SetCapture();
      GetDocument()->UpdateAllViews(0,UPDATE_OBJECTS,0);
    }
  }
}

void CObjectView::OnRButtonUp(UINT nFlags, CPoint point) 
{
  if(GetCapture() == this && m_sObject != "" && !m_bDrag)
  {
    CRect rect;
    GetClientRect(&rect);
    double fovy((verticalOpeningAngleIdx)*20.0);
    GetDocument()->GetSimulation()->rotateObject(point.x,point.y);
    GetDocument()->GetSimulation()->unselectObject();
    ReleaseCapture();
    GetDocument()->UpdateAllViews(0,UPDATE_OBJECTS,0);
  }
  else
  {  
    ClientToScreen(&point);
    m_menuPopup.GetSubMenu(0)
      ->TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RIGHTBUTTON,
                       point.x,point.y,AfxGetMainWnd());
    // Force repaint after menu has completely disappeared.
    SetTimer(0,300,NULL);
  }
}

void CObjectView::OnDestroy() 
{
  if(GetCapture() == this)
    ReleaseCapture();
  
  if(wglGetCurrentContext() != NULL)
    wglMakeCurrent(NULL,NULL);
  
  if(m_hGLContext != NULL)
  {
    wglDeleteContext(m_hGLContext);
    m_hGLContext = NULL;
  }
  CView::OnDestroy();
}

int CObjectView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CView::OnCreate(lpCreateStruct) == -1)
		return -1;
	
  m_hDC = ::GetDC(GetSafeHwnd());
  
  PIXELFORMATDESCRIPTOR pixelDesc;
  
  pixelDesc.nSize = sizeof(PIXELFORMATDESCRIPTOR);
  pixelDesc.nVersion = 1;
  
  pixelDesc.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER | PFD_STEREO_DONTCARE;
  
  pixelDesc.iPixelType = PFD_TYPE_RGBA;
  pixelDesc.cColorBits = 32;
  pixelDesc.cRedBits = 8;
  pixelDesc.cRedShift = 16;
  pixelDesc.cGreenBits = 8;
  pixelDesc.cGreenShift = 8;
  pixelDesc.cBlueBits = 8;
  pixelDesc.cBlueShift = 0;
  pixelDesc.cAlphaBits = 0;
  pixelDesc.cAlphaShift = 0;
  pixelDesc.cAccumBits = 64;
  pixelDesc.cAccumRedBits = 16;
  pixelDesc.cAccumGreenBits = 16;
  pixelDesc.cAccumBlueBits = 16;
  pixelDesc.cAccumAlphaBits = 0;
  pixelDesc.cDepthBits = 32;
  pixelDesc.cStencilBits = 8;
  pixelDesc.cAuxBuffers = 0;
  pixelDesc.iLayerType = PFD_MAIN_PLANE;
  pixelDesc.bReserved = 0;
  pixelDesc.dwLayerMask = 0;
  pixelDesc.dwVisibleMask = 0;
  pixelDesc.dwDamageMask = 0;

  int glPixelIndex = ChoosePixelFormat(m_hDC,&pixelDesc);
  if(glPixelIndex==0) // Choose default
  {
    glPixelIndex = 1;
    if(!DescribePixelFormat(m_hDC,glPixelIndex,sizeof(PIXELFORMATDESCRIPTOR),&pixelDesc))
      return false;
  }
  
  if(!::SetPixelFormat(m_hDC,glPixelIndex,&pixelDesc))
    return -1;

  m_hGLContext = wglCreateContext(m_hDC); 
  if(!m_hGLContext)
    return -1;

  if(!wglMakeCurrent(m_hDC,m_hGLContext))
    return -1;

  return 0;
}

void CObjectView::OnTimer(UINT nIDEvent) 
{
  KillTimer(0);
  Invalidate(false);
  	
	CView::OnTimer(nIDEvent);
}

void CObjectView::OnSelectSurfaceStyle()
{
  if(m_sObject != "")
  {
    int updatedStyle = LOWORD(GetCurrentMessage()->wParam) - ID_SURFACE_WIRE;
    visualizationParameters.surfaceStyle = 
    (VisualizationParameterSet::SurfaceStyle)updatedStyle;
    Invalidate(false);
  }
}

void CObjectView::OnUpdateSurfaceStyle(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sObject != "");
  int currentStyle((int)visualizationParameters.surfaceStyle);
  int updatedStyle(pCmdUI->m_nID - ID_SURFACE_WIRE);
  pCmdUI->SetCheck(currentStyle == updatedStyle);
}

void CObjectView::OnShowSensors() 
{
  if(m_sObject != "")
  {
    showSensors ^= true;
    Invalidate(false);
  }
}
 
void CObjectView::OnUpdateShowSensors(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sObject != "");
  pCmdUI->SetCheck(showSensors);
}

void CObjectView::OnPlaneXY() 
{
  if(m_sObject != "")
    m_plane = XY_PLANE;
}

void CObjectView::OnUpdatePlaneXY(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sObject != "");
  pCmdUI->SetCheck(m_plane == XY_PLANE);
}

void CObjectView::OnPlaneXZ() 
{
  if(m_sObject != "")
    m_plane = XZ_PLANE;
}

void CObjectView::OnUpdatePlaneXZ(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sObject != "");
  pCmdUI->SetCheck(m_plane == XZ_PLANE);
}

void CObjectView::OnPlaneYZ() 
{
  if(m_sObject != "")
    m_plane = YZ_PLANE;
}

void CObjectView::OnUpdatePlaneYZ(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sObject != "");
  pCmdUI->SetCheck(m_plane == YZ_PLANE);
}
