#ifndef _CONTROLLER_H_
#define _CONTROLLER_H_

#ifndef _NOLINK
#include "../GUI/StdAfx.h"
#include "../GUI/SimRobXP.h"
#endif
#include <string>
#include <vector>
#include "Platform/Globals.h"
#include "SimRobotCore/SimMath.h"
#include "SimRobotCore/View.h"

class Simulation;
class SimObject;
class Surface;

/**
 * The class is the interface between user-written robot controllers
 * and the simulator. A user-written controller has to be derived from
 * this class and the following macro has to be placed in the code:
 * CONNECT_CONTROLLER_TO_SCENE(ControllerClass, "SceneName");
 */
class Controller
{
public:
  /** 
   * Constructor.
   */
  Controller();

  /**
   * Destructor.
   */
  virtual ~Controller();

  /** 
   * The function returns the name of the simulation file.
   * @return The file name.
   */
  std::string Controller::getSimulationFileName() const;

  /** 
   * The function returns the current simulation step.
   * @return The number of the current step, starting with 0.
   */
  unsigned int getSimulationStep() const;

  /** 
   * The function returns a pointer to an object.
   * An assertion will fail if the object does not exist.
   * @param objectName The full name of the object.
   * @return A pointer to the object.
   */
  SimObject* getObjectReference(const std::string& objectName);

  /** 
   * The function returns the surface with a given name.
   * An assertion will fail if the surface does not exist.
   * @param name The name of the surface searched for.
   * @return A pointer to the surface.
   */
  Surface* getSurface(const std::string& name) const;

  /** 
   * The function sets the color of the background of the scene.
   * The color has to be a triple of (R,G,B) values in the range of [0..1].
   * @param color The color
   */
  void setBackgroundColor(const Vector3d& color);

  /** 
   * The function sets the color of the ambient light.
   * The color has to be a triple of (R,G,B) values in the range of [0..1].
   * @param color The color
   */
  void setAmbientColor(const Vector3d& color);

  /**
   * The function returns the id of an actuatorport with a given name.
   * An assertion will fail if the actuatorport does not exist.
   * @param portName The name of the actuatorport.
   * @return The id.
   */
  int getActuatorportId(std::string portName) const;

  /** 
   * The function returns the minimum value to be set for a certain actuatorport.
   * @param id The id of the port.
   * @return The minimum value.
   */
  double getActuatorportMinValue(int id) const;

  /** 
   * The function returns the maximum value to be set for a certain actuatorport.
   * @param id The id of the port.
   * @return The maximum value.
   */
  double getActuatorportMaxValue(int id) const;

  /** 
   * The function sets the value of an actuatorport.
   * @param id The id of the port.
   * @param value The new value.
   */
  void setActuatorport(int id, double value);

  /**
   * The function returns the id of a sensorport with a given name.
   * An assertion will fail if the sensorport does not exist.
   * @param portName The name of the sensorport.
   * @return The id.
   */
  int getSensorportId(std::string portName) const;

  /** 
   * The function returns the dimensions of the values returned by a certain sensorport.
   * @param id The id of the port.
   * @return An array. The number of values equals the number of dimensions.
   *         Each value specifies the size of the corresponding dimension.
   */
  const std::vector<int>& getSensorDimensions(int id) const;

  /** 
   * The function returns the current value of a certain sensorport.
   * An assertion fails if the type of the sensorport is not booleanSensor.
   * @param id The id of the port.
   * @param value The current value as Boolean.
   */
  void getSensorValue(int id, bool& value);

  /** 
   * The function returns the current value of a certain sensorport.
   * An assertion fails if the type of the sensorport is not doubleSensor.
   * @param id The id of the port.
   * @param value The current value.
   */
  void getSensorValue(int id, double& value);

  /** 
   * The function returns the current value of a certain sensorport.
   * An assertion fails if the type of the sensorport is not intSensor.
   * @param id The id of the port.
   * @param value The current value.
   */
  void getSensorValue(int id, int& value);

  /** 
   * The function returns the current value of a certain sensorport.
   * An assertion fails if the type of the sensorport is not doubleFieldSensor.
   * @param id The id of the port.
   * @param value The current value.
   */
  void getSensorValue(int id, double*& value);

  /** 
   * The function returns the current value of a certain sensorport.
   * An assertion fails if the type of the sensorport is not cameraSensor.
   * @param id The id of the port
   * @param value The current value.
   */
  void getSensorValue(int id, unsigned char*& value);
  
  /**
   * The function adds a view to the scene.
   * The view can be found in the object tree in the group "views".
   * @param The view.
   * @param name The name of the view will have in the group "views".
   */
  void addView(View* view, const std::string& name);

  /**
   * The function prints text into the console window.
   * @param text The text to be printed .
   */
  void print(const std::string& text);

  /**
   * The function prints text followed by a line break into the console window.
   * @param text The text to be printed.
   */
  void printLn(const std::string& text);

  /**
   * The function clears the console window.
   */
  void clear();

  /**
   * The function is called before every simulation step.
   * It must be implemented in a derived class.
   */
  virtual void execute() = 0;

  /**
   * The function is called before the first call to execute().
   * When this function is called the controller is already constructed.
   */
  virtual void init() {}

  /**
   * The function is called before the controller is destructed.
   */
  virtual void destroy() {}

  /**
   * The function is called when a key on the numerical keypad was pressed.
   * @param key 0..9 for the keys 0..9 and 10 for the decimal point.
   */
  virtual void onKeyPressed(int key) {}

  /**
    * The function is called when a movable object has been selected.
    * @param obj The object.
    */
  virtual void onSelected(SimObject* obj) {}

  /**
    * The function is called when a console command has been entered.
    * @param command The command.
    */
  virtual void onConsoleCommand(const std::string&) {}

  /**
    * The function is called when the tabulator key is pressed.
    * It can replace the given command line by a new one.
    * @param command The command.
    * @param forward Complete in forward direction.
    */
  virtual void onConsoleCompletion(std::string& command, bool forward) {}

private:
  Simulation* simulation; /**< A pointer to the simulation. */
};

/**
 * The class is a helper to connect controllers to simulation scenes.
 */
class Connection
{
  public:
    static Connection* start; /**< The start of the list of all controllers. */
    const char* sceneName; /**< The name of the scene the controller is connected to. */
    Connection* next; /** The next entry in the list of all controllers. */
    
    /**
     * Constructor.
     * @param The name of the scene the controller is connected to. 
     */
    Connection(char* sceneName);

    /**
     * The function is called to create an instance of the controller.
     * @return A pointer to a new instance of the controller.
     */
    virtual Controller* createController() = 0;
};

/**
 * The template class is also helper to connect controllers to simulation scenes.
 * The template parameter is the controller class to construct.
 */
template <class T> class ConnectionTemplate : public Connection
{
  public:
    /**
     * Constructor.
     * @param The name of the scene the controller is connected to. 
     */
    ConnectionTemplate(char* sceneName) : Connection(sceneName) {}

    /**
     * The function is called to create an instance of the controller.
     * @return A pointer to a new instance of the controller.
     */
    Controller* createController() {return new T;}
};

/**
 * The macro connects a controller to a scene.
 * @param Controller The class of the controller.
 * @param sceneName The name of the scene.
 */
#define CONNECT_CONTROLLER_TO_SCENE(Controller,sceneName) \
  ConnectionTemplate<Controller> dummy##ClassName(sceneName);

/**
 * The class is the base class for direct views, i.e. views that draw themselves.
 */
class DirectView : public View
{
  public:
    /**
     * The function is called when the view has to be drawn.
     * @param dc A device context to draw to.
     */
    virtual void draw(CDC& dc) = 0;

  private:
    /**
     * The function is called to update the data of the view.
     * However, direct views store no data.
     */
    virtual void update() {}
};

//////////////////////////////////////////////////////////////////////////
// Force linking of GUI.lib

#ifndef _NOLINK
extern "C" int WINAPI WinMain(HINSTANCE hInstance,HINSTANCE,LPSTR,int);
extern class CApp theApp;
static int *__p1 = (int*) WinMain,
           *__p2 = (int*) &theApp;
#endif

#endif
