/**
* @file WorldstatePlayer.cpp 
*
* Implementation of class WorldstatePlayer.
*
*/

#include "WorldstatePlayer.h"
#include "Bars/WorldstateAnalysisDlgBar.h"
#include "Platform/SystemCall.h"
#include "Representations/RoboCup/GameControlData.h"
#include "../Tools/Player.h"

/*
//\\// CMessageHandlerForSync \\//\\
//-----------------------------------------------------------------------------
CMessageHandlerForSync::CMessageHandlerForSync()
: m_timeStartgame(0)
, m_timeMax(0)
, m_timeMin(0xffffffff)
{
}

//-----------------------------------------------------------------------------
CMessageHandlerForSync::~CMessageHandlerForSync()
{
}

//-----------------------------------------------------------------------------
TTIME CMessageHandlerForSync::getStartTime()
{
	if (m_timeStartgame == 0)
		return getMinTime();
	return m_timeStartgame;
}

TTIME CMessageHandlerForSync::getMaxTime()
//-----------------------------------------------------------------------------
{
	return m_timeMax;
}

//-----------------------------------------------------------------------------
TTIME CMessageHandlerForSync::getMinTime()
{
	return m_timeMin;
}

//-----------------------------------------------------------------------------
bool CMessageHandlerForSync::handleMessage(InMessage& message)
{
	TTIME tmp = message.getTimeStamp();
	switch (message.getMessageID())
	{
		case idPercepts:
		case idGlobalGameControlData:
			if (tmp < m_timeMin)
				m_timeMin = tmp;
			if (tmp > m_timeMax)
				m_timeMax = tmp;
			break;
	}
	if (m_timeStartgame == 0)
	{
		switch (message.getMessageID())
		{
			case idPercepts:
			{
				Player	gPlayer;
				CameraMatrix cameraMatrix;
				BallPercept	ballPercept;
				LandmarksPercept landmarksPercept;
				PlayersPercept			playersPercept;
				ObstaclesPercept		obstaclesPercept;
				PSDPercept				psdPercept;
				CollisionPercept		collisionPercept;
				LinesPercept			linesPercept;

				message.bin >> RECEIVE_PERCEPTS(gPlayer,cameraMatrix,ballPercept,
				landmarksPercept,linesPercept,playersPercept, obstaclesPercept, psdPercept, collisionPercept);
				if (( ballPercept.ballWasSeen )
				||	( obstaclesPercept.numberOfPoints > 2 ))
					m_timeStartgame = tmp;
			}
			break;
			case idGlobalGameControlData:
			{
			  GlobalGameControlData ggcd;
			  message.bin >> ggcd;
			  if (ggcd.state == ggcd.playing)
				  m_timeStartgame = tmp;
			}
			break;
		}
	}

	return true;
}
*/

//\\// WorldstatePlayer \\//\\
//-----------------------------------------------------------------------------
WorldstatePlayer::WorldstatePlayer(CRobotControlQueues& queues,HWND hwnd,HINSTANCE instance, CWorldstateAnalysisDlgBar* bar)
: m_speed(1)
{
	m_bar = bar;

	m_statusCam = deleted;

	for (int i=0; i< Player::numOfPlayerNumbers ;++i)
	{

		if (i==5)
		{
			m_logplayer[i] = new LogPlayerWithSync(queues.toSimulated.robot[i],&m_playTime);
		}
		else
		{
			m_logplayer[i] = new LogPlayerWithSync(m_queue[i],&m_playTime);
		}

	}

	m_movieplayer = new MoviePlayer(hwnd, instance);
	m_statusCam = empty;

	m_statusPlayer = empty;
}

WorldstatePlayer::~WorldstatePlayer()
{
	delete m_movieplayer;
	m_movieplayer = 0;
	m_statusCam = deleted;
	
	for (int i=0; i< Player::numOfPlayerNumbers ;++i)
	{
		delete m_logplayer[i];
		m_logplayer[i] = 0;
		m_queue[i].clear();
	}
}

bool WorldstatePlayer::setStarttime(int num, string value)
{
	int i = value.find("!");
	if (i>=0)
	{
		value.replace(i,1,"");
		long tmp;
		i = 0;
		while( value[i] == '0')	i++;
		sscanf( &value[i],"%i",&tmp);

		if (num >= Player::numOfPlayerNumbers)
		{
			m_movieplayer->m_synctime = tmp;
		}else{
			m_logplayer[num]->setSync(tmp);
		}
		return true;
	}
	i = value.find("+");
	if (i>=0)
	{
		value.replace(i,1,"");
		long tmp;
		i = 0;
		while( value[i] == '0')	i++;
		sscanf( &value[i],"%i",&tmp);
		if (num >= Player::numOfPlayerNumbers)
		{
			m_movieplayer->m_synctime += tmp;
		}else{
			m_logplayer[num]->setSync(m_logplayer[num]->getSyncTime()+tmp);
		}
		return true;
	}
	i = value.find("-");
	if (i>=0)
	{
		value.replace(i,1,"");
		long tmp;
		i = 0;
		while( value[i] == '0')	i++;
		sscanf( &value[i],"%i",&tmp);
		if (num >= Player::numOfPlayerNumbers)
		{
			m_movieplayer->m_synctime -= tmp;
		}else{
			m_logplayer[num]->setSync(m_logplayer[num]->getSyncTime()-tmp);
		}
		return true;
	}

	return false;
}

void WorldstatePlayer::play()
{
	switch (m_statusCam)
	{
		case initial:
			m_movieplayer->play();
			m_statusCam = playing;
			break;
		case paused:
			m_movieplayer->play();
			m_statusCam = playing;
			break;
	}


    m_lasttimeSystem = SystemCall::getCurrentSystemTime();
	m_statusPlayer = playing;
}

void WorldstatePlayer::stop()
{
	if (m_statusCam >= initial)
	{
		m_movieplayer->stop();
		m_statusCam = initial;
	}

	m_playTime = 0;
	m_statusPlayer = initial;
	for(int i=0; i<Player::numOfPlayerNumbers; ++i)
	{
		m_logplayer[i]->seek(m_playTime);

	}
}

void WorldstatePlayer::pause()
{
	m_lasttimeSystem = SystemCall::getCurrentSystemTime();
	int i(0);
	switch(	m_statusPlayer )
	{
	case playing:
			m_statusPlayer = paused;
			break;
		case paused:
			m_statusPlayer = playing;
			break;
	}

	switch(	m_statusCam )
	{
		case playing:
			m_statusPlayer = paused;
			if (m_statusCam == playing)
			{
				m_movieplayer->pause();
				m_statusCam = paused;
			}
			break;
		case paused:
			m_statusPlayer = playing;
			if (m_statusCam == paused)
			{
				m_movieplayer->play();
				m_statusCam = playing;
			}
			break;
	}
}

void WorldstatePlayer::speed(int sp)
{
	m_speed = (double)sp / 100;
	m_movieplayer->speed(sp);
}

bool WorldstatePlayer::hasTimeDelay()
{
	if (m_statusPlayer != playing)
		return false;
	for(int i=0; i < Player::numOfPlayerNumbers; ++i)
		if (this->isLogfileActive((Player::playerNumber)i)
		&& m_logplayer[i]->hasTimeDelay())
		return true;
	return false;
}

void WorldstatePlayer::load()
{
	OPENFILENAME ofn ;
			char fileName[1000] ;
			char filePathName[1000] ;
			m_statusCam = empty;
			fileName[0] = '\0' ;
			filePathName[0] = '\0' ;

			ZeroMemory(&ofn, sizeof(ofn));
			ofn.lStructSize = sizeof(ofn);
			ofn.lpstrFilter = "AVI Files (*.avi)\0*.avi\0\0";
			ofn.Flags = OFN_NOCHANGEDIR ;
			ofn.lpstrFile = filePathName ;
			ofn.nMaxFile = 999 ;
			ofn.lpstrFileTitle = fileName ;
			ofn.nMaxFileTitle = 999 ;
		
			GetOpenFileName(&ofn) ;
			if(fileName[0]!='\0') { // If a file is selected...
			// Stop (if any) active movie
			
			m_movieplayer->open(filePathName);
			m_statusCam = initial;
			}
			m_timeMin = 0xffffffff;
			m_timeMax = 0;
			
			filePathName[0] = '\0' ;
			fileName[0] = '\0' ;
			for(int i=0; i<Player::numOfPlayerNumbers; ++i)
			{
				m_logplayer[i]->stop();
				m_logplayer[i]->_new();
				fileName[0] = '\0' ;
				for(int ii=1; ii< 900; ++ii)
				{
					if (filePathName[ii] == '.')
					{
						filePathName[ii-1] = filePathName[ii-1]+1;
					}
				}
				ZeroMemory(&ofn, sizeof(ofn));
				ofn.lStructSize = sizeof(ofn);
				ofn.lpstrFilter = "LOG Files (*.log)\0*.log\0\0";
				ofn.Flags = OFN_NOCHANGEDIR ;
				ofn.lpstrFile = filePathName ;
				ofn.nMaxFile = 999 ;
				ofn.lpstrFileTitle = fileName ;
				ofn.nMaxFileTitle = 999 ;
			
				GetOpenFileName(&ofn) ;
				if(fileName[0]!='\0') {
				  m_logplayer[i]->open(filePathName);
//				  m_statusLogfile[i]=initial;
				  syncLogfile( (Player::playerNumber)( i ) );
				}
			}

	if (m_timeMax <= 0)
		m_timeMax = 10000; //standart Spielzeit
	m_statusPlayer = initial;

/*	if (m_statusCam == initial)
	{
		m_movieplayer->m_synctime = 0;// besser 0 oder Synchronisieren
		if (m_timeMax < int(m_movieplayer->getLength() + m_movieplayer->m_synctime))
			m_timeMax = int(m_movieplayer->getLength() + m_movieplayer->m_synctime);
	}
	*/

    m_lasttimeSystem = SystemCall::getCurrentSystemTime();
	m_playTime = 0;
}

void WorldstatePlayer::stepForward()
{
	if ( m_statusPlayer == playing )
		pause();

	if ( m_statusPlayer != paused )
		return;

	long ntime = m_playTime + 1000;
	// search minimal step
	Player::playerNumber num;
	long tmp;
	for(int i= 0; i<Player::numOfPlayerNumbers; ++i)
	{
		if (!isLogfileActive( (Player::playerNumber)i) )
			continue;
		num = (Player::playerNumber)i;
		tmp = getLogfilePos(num);
		tmp = m_logplayer[num]->getNextStepForward();
		if (tmp < ntime)
			ntime = tmp;
	}
	if (m_playTime <= ntime)
	m_playTime = ntime+1;

	if (m_statusCam >= initial)
	{
		m_movieplayer->seekPos(m_playTime);//+m_starttimeCam);
//		m_movieplayer->stepForward();
	}
}

void WorldstatePlayer::stepBackward()
{
	if ( m_statusPlayer == playing )
		pause();
	if ( m_statusPlayer != paused )
		return;

	long ntime = m_playTime-1000;

	// search minimal step
	Player::playerNumber num;
	long tmp;
	for(int i= 0; i<Player::numOfPlayerNumbers; ++i)
	{
		num = (Player::playerNumber)i;
		if (!isLogfileActive( num ) )
			continue;
		tmp = m_logplayer[num]->getNextStepBackward();
		if (tmp > ntime)
			ntime = tmp;
	}

	if (m_playTime > ntime)
	m_playTime = ntime -1 ;

	if (m_playTime < m_timeMin)
		m_playTime = m_timeMin;

	if (m_statusCam >= initial)
	{
		m_movieplayer->seekPos(m_playTime);//+m_starttimeCam);
//		m_movieplayer->stepBackward();
	}
}

void WorldstatePlayer::seekPos(int pos)
{
	m_playTime = pos;
	for(int i=0; i< Player::numOfPlayerNumbers; ++i)
		m_logplayer[i]->seek(m_playTime);
    m_lasttimeSystem = SystemCall::getCurrentSystemTime();
	m_movieplayer->seekPos(m_playTime);//+m_starttimeCam);
}

void WorldstatePlayer::seekPosPerCent(int pos)
{
	long tmp;
	tmp = m_timeMax-m_timeMin;
	tmp = tmp * pos / 100;
	tmp = tmp +m_timeMin;
	seekPos(tmp);
}

bool WorldstatePlayer::handleMessage(InMessage &message)
{
	MessageID messageID;
	messageID = message.getMessageID();
	if (message.getMessageID() == idText)
		return false;
	if (m_bar != 0)
		m_bar->paintMessage(message, m_actMessage);
	return true;
}

void WorldstatePlayer::OnIdle()
{
	switch (m_statusPlayer)
	{
		case playing:
		case recording:
			updatePlaytime();
			break;
		case deleted:
		case empty:
			return;
		case initial:
		case paused:
			break;
	}
	
	int i;
	for(i=0; i<Player::numOfPlayerNumbers; ++i)
	{
		m_actMessage = (Player::playerNumber)i;
		m_logplayer[i]->onIdle();
		m_queue[i].handleAllMessages(*(WorldstatePlayer*)(void*)this);
	}

	if (m_movieplayer != 0)
		m_movieplayer->seekPos(m_playTime);

}

int WorldstatePlayer::getStatus()
{
	return m_statusPlayer;
}

TTIME WorldstatePlayer::getStarttime(int num)
{
	if (num >= Player::numOfPlayerNumbers)
	{
		return m_movieplayer->m_synctime;
	}else{
		return m_logplayer[num]->getSyncTime();
	}
}

long WorldstatePlayer::getPos()
{
	return m_playTime;
}

void WorldstatePlayer::syncLogfile(Player::playerNumber num)
{
	unsigned long tmpstarttime = m_logplayer[num]->getGamestartTime();

	long tmp;
	tmp = m_logplayer[num]->getLogstartTime() - tmpstarttime;
	if (tmp < m_timeMin)
		m_timeMin	= tmp;
	tmp = m_logplayer[num]->getLogendTime() - tmpstarttime;
	if (tmp > m_timeMax)
		m_timeMax	= tmp;
	m_logplayer[num]->seek(tmpstarttime);
	m_logplayer[num]->setSync(tmpstarttime);
}

long WorldstatePlayer::getLogfilePos(Player::playerNumber num)
{
	if (!isLogfileActive(num))
		return m_playTime-1;
	return m_logplayer[num]->getGameTime();
}

bool WorldstatePlayer::isLogfileActive(Player::playerNumber num)
{
	return ((m_logplayer[num] != 0)
		&& (m_logplayer[num]->getNumberOfMessages() > 0));
}

void WorldstatePlayer::updatePlaytime()
{
	long tmp = SystemCall::getTimeSince(m_lasttimeSystem);
	if (tmp > 4000)
		tmp = 4000;
	tmp = long(tmp * m_speed);
	m_playTime += tmp;
	m_lasttimeSystem = SystemCall::getCurrentSystemTime();
}

double WorldstatePlayer::getPosPerCent()
{
	if (m_playTime >= m_timeMax)
		return 100;
	if (m_playTime <= m_timeMin)
		return 0;
	return ((double)(m_playTime-m_timeMin)/(double)(m_timeMax-m_timeMin))*100;
}