/** 
 * @file PaintMethodsWin32.h
 * Declaration of class PaintMethodsWin32.
 *
 * @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
 */

#ifndef __PaintMethodsWin32_h__
#define __PaintMethodsWin32_h__

#include "Visualization/DebugDrawing.h"
#include "Representations/Perception/ColorTable.h"

#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/PlayersPercept.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/PSDPercept.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/ObstaclesModel.h"

#include "Tools/KickSelectionTable.h"

/**
* @class PaintMethodsWin32
*
* Defines static methods to paint debug drawings to CDCs.
*/
class PaintMethodsWin32 
{
public:
  /** 
  * Paints a DebuDrawing to a CDC. 
  * @param debugDrawing The DebugDrawing to paint.
  * @param dc The CDC the DebugDrawing is painted to.
  */
  static void paintDebugDrawingToCDC(const DebugDrawing& debugDrawing, CDC& dc);

  /** 
  * Paints a DebuDrawing to a CDC. 
  * @param debugDrawing The DebugDrawing to paint.
  * @param dc The CDC the DebugDrawing is painted to.
  * @param scale Specifies how the drawing is magnified.
  */
  static void paintDebugDrawingToCDC(const DebugDrawing& debugDrawing, CDC& dc, double scale);

  /** 
  * Paints a ColorTable to a bitmap. 
  * The view to the color table is along the y axis.
  * @param colorTable The ColorTable to paint.
  * @param pBuffer Points to the Bitmap, where the ColorTable is painted to.
  */
  static void paintColorClassUV2CDC(BITMAPINFOHEADER* pBuffer, ColorTable& colorTable);

  /** 
  * Paints a ColorTable to a CDC. 
  * The view to the color table is along the v axis.
  * @param colorTable The ColorTable to paint.
  * @param pBuffer Points to the Bitmap, where the ColorTable64 is painted to.
  */
  static void paintColorClassUY2CDC(BITMAPINFOHEADER* pBuffer, ColorTable& colorTable);

  /** 
  * Paints a slice of a ColorTable to a CDC. 
  * The view to the color table is along the y axis.
  * @param pColorTable Points to the ColorTable to paint.
  * @param dcMem The CDC the ColorTable is painted to.
  * @param slice y channel of the slice to paint.
  * @param init If true, paint a black background rectangle before painting the slice.
  */
  static void paintColorClassUV2CDC(
    CDC* dcMem, 
    unsigned char slice,
    bool init, 
    ColorTable* pColorTable
    );

  /** 
  * Paints a slice of a ColorTable to a CDC.
  * The view to the color table is along the y axis.
  * @param pColorTable Points to the ColorTable to paint.
  * @param dcMem The CDC the ColorTable is painted to.
  * @param slice y channel of the slice to paint.
  * @param init If true, paint a black background rectangle before painting the slice.
  */
  static void paintColorClassUY2CDC(
    CDC* dcMem,
    unsigned char slice,
    bool init, 
    ColorTable* pColorTable
    );

  static void createColoredPens(CBrush* coloredBrush);

  static void deleteColoredPens(CBrush* coloredBrush);

  static void paintKickSegmentsToCDC(
    CDC& dc, 
    int currentWidth,
    int currentHeight,
    int numberOfSegments,
    int indexOfHighlightedSegment,
    int indexOfSelectedSegment,
    int frameSize);

  static void paintCoordinateSystemToCDC(CDC& dc, double scale);

  static void paintCoordinateSystemForKickSelectionToCDC(CDC& dc, double scale);

  static void paintCoordinateSystemToCDC(
    CDC& dc, 
    double scale,
    double maxX,
    double minX,
    double maxY,
    double minY,
    double spacing
    );

  static void paintRobotToCDC(CDC& dc, double scale);
  static void paintFrameForScalingToCDC(CDC& dc, double scale);
  static void paintBallPerceptToCDC(
    BallPercept& ballPercept, 
    bool bearingBased, 
    bool ballAsCross,
    bool lineToBall,
    CDC& dc, 
    double scale);
  static void paintBallModelToCDC(
    BallModel& ballModel, 
    RobotPose& robotPose, 
    bool ballAsCross,
    bool lineToBall,
    CDC& dc, 
    double scale);
  static void paintLandmarksPerceptToCDC(LandmarksPercept& landmarksPercept, CDC& dc, double scale);
  static void paintObstaclesPerceptToCDC(ObstaclesPercept& obstaclesPercept, CDC& dc, double scale);
  static void paintObstaclesModelToCDC(ObstaclesModel& obstaclesModel, CDC& dc, double scale);
  static void paintPSDPerceptToCDC(PSDPercept& psdPercept, CDC& dc, double scale);
  static void paintLinesPerceptToCDC(LinesPercept& linesPercept, CDC& dc, double scale);
  static void paintCameraMatrixToCDC(const CameraMatrix& cameraMatrix, const CameraInfo& cameraInfo, CDC& dc, double scale);
  static void paintImageProjectionOnGroundToCDC(const CameraMatrix& cameraMatrix, const CameraInfo& cameraInfo, const Image& image, CDC& dc, double scale);

  /** Defines how to paint a value*/
  class PaintStyle
  {
  public:
    /** Default constructor */
    PaintStyle() : shape(vcross), penColor(RGB(0,0,0)), brushColor(RGB(0,0,0)), penWidth(1), scale(1) {}

    /** The shape of a node */
    enum Shape { circle, filledCircle, square, filledSquare, cross, vcross, horizontalLine } shape;
 
    /** The color of a node */
    COLORREF penColor;

    /** The color of a node */
    COLORREF brushColor;

    /** The width of the pen*/
    int penWidth;

    /** 
    * A factor that is multiplied to the value for that it fits into the 
    * range from [-100, 100]
    */
    double scale;
  };
};
#endif // __PaintMethodsWin32_h__


/*
 * Change log :
 * 
 * $Log: PaintMethodsWin32.h,v $
 * Revision 1.6  2004/03/25 21:21:49  juengel
 * Implemented paintBallModelToCDC.
 *
 * Revision 1.5  2004/03/16 14:00:23  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.6  2004/03/10 22:02:46  juengel
 * Made frame size variable.
 *
 * Revision 1.5  2004/03/10 14:52:57  juengel
 * Improved kick segment drawing.
 *
 * Revision 1.4  2004/03/09 18:42:18  juengel
 * Moved PaintStyle to PaintMethodsWin32
 *
 * Revision 1.3  2004/03/09 13:41:41  juengel
 * Added parameters to paintCoordinateSystem.
 *
 * Revision 1.2  2004/03/09 01:15:14  juengel
 * Visualization of kick sectors.
 *
 * Revision 1.4  2004/02/07 15:48:24  juengel
 * Added buffer edit box to RadarViewer.
 * Added some different modes of view for the BallPercept
 *
 * Revision 1.3  2004/02/03 13:19:49  spranger
 * renamed all references to  class BallPosition to BallModel
 *
 * Revision 1.2  2004/02/02 10:09:49  juengel
 * Redesign of RadarViewer.
 *
 * Revision 1.1  2003/10/07 10:11:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:28:24  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:27  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/03/31 03:50:39  osterhues
 * changed ColorTable64 to ColorTable
 *
 * Revision 1.5  2002/12/18 16:49:30  dueffert
 * doxygen docu corrected
 *
 * Revision 1.4  2002/12/18 16:22:56  dueffert
 * doxygen docu corrected
 *
 * Revision 1.3  2002/11/12 10:49:02  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.2  2002/10/10 11:46:12  juengel
 * Scalable version for paintDebugDrawingToCDC(...) added.
 *
 * Revision 1.1  2002/09/10 15:49:12  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.1.1.1  2002/05/10 12:40:27  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.2  2002/02/11 00:53:33  loetzsch
 * ::Paint Methoden in ::paint umbenannt
 *
 * Revision 1.1  2002/01/04 14:36:00  juengel
 * no message
 *
 */
