/** 
* @file DrawingMethods.h
* Declaration of class DrawingMethods.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/

#ifndef __DrawingMethods_h__
#define __DrawingMethods_h__

#include "Visualization/DebugDrawing.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/PlayersPercept.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Perception/PSDPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/SpecialPercept.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Representations/Cognition/ObstaclesModel.h"
#include "Representations/Perception/CollisionPercept.h"
#include "Tools/Player.h"

/**
* @class DrawingMethods
*
* Defines static methods to paint debug drawings.
*/
class DrawingMethods 
{
public:

  /** 
  * Paints the field lines to a debug drawing.
  * One unit is one millimeter.
  */
  static void paintFieldLines(DebugDrawing& fieldLinesDrawing);
  
  /** 
  * Paints the field to a debug drawing.
  * The lawn is green, the goals are yellow and blue.
  * One unit is one millimeter.
  */
  static void paintFieldPolygons(
    DebugDrawing& fieldPolygonsDrawing,
    Player::teamColor ownColor
    );
  
  /**
  * Paints the special percept to a debug drawing for the image viewer
  */
  static void paintSpecialPerceptForImageView(
    DebugDrawing& perceptsDrawing, 
    const SpecialPercept& specialPercept,
    const CameraMatrix& cameraMatrix);

  /**
  * Paints a BallPercept to an image drawing
  */
  static void paintBallPerceptForImageView(
     DebugDrawing& perceptsDrawing, 
     const BallPercept& ballPercept,
     const CameraMatrix& cameraMatrix,
     const CameraInfo& cameraInfo
     );

  /**
  * Paints a LandmarksPercept to an image drawing
  */
  static void paintLandmarksPerceptForImageView(
     DebugDrawing& perceptsDrawing, 
     const LandmarksPercept& landmarksPercept,
     const CameraMatrix& cameraMatrix,
     const CameraInfo& cameraInfo
     );

  /**
  * Paints a LinesPercept to an image drawing
  */
  static void paintLinesPerceptForImageView(
     DebugDrawing& perceptsDrawing, 
     const LinesPercept& linesPercept,
     const CameraMatrix& cameraMatrix,
     const CameraInfo& cameraInfo
     );

  /**
  * Paints an ObstaclesPercept to an image drawing
  */
  static void paintObstaclesPerceptForImageView(
     DebugDrawing& perceptsDrawing, 
     const ObstaclesPercept& obstaclesPercept,
     const CameraMatrix& cameraMatrix,
     const CameraInfo& cameraInfo
     );

  /**
  * Paints the "free part of goal" percept to an image drawing
  */
  static void paintFreePartOfGoalPerceptForImageView(
     DebugDrawing& perceptsDrawing, 
     const ObstaclesPercept& obstaclesPercept,
     const CameraMatrix& cameraMatrix,
     const CameraInfo& cameraInfo
     );


  /**
  * Paints a ObstaclesModel to an image drawing
  */
  static void paintObstaclesModelForImageView(
     DebugDrawing& obstaclesDrawing, 
     const ObstaclesModel& obstaclesModel,
     const CameraMatrix& cameraMatrix,
     const CameraInfo& cameraInfo
     );

  /**
  * Paints the "free part of goal" model to an image drawing
  */
  static void paintFreePartOfGoalModelForImageView(
     DebugDrawing& freePartOfGoalDrawing, 
     const ObstaclesModel& obstaclesModel,
     const CameraMatrix& cameraMatrix,
     const CameraInfo& cameraInfo
     );


  /**
  * Paints a PSDPercept to an image drawing
  */
  static void paintPSDPerceptForImageView(
     DebugDrawing& perceptsDrawing, 
     const PSDPercept& psdPercept,
     const CameraMatrix& cameraMatrix,
     const CameraInfo& cameraInfo
     );

  /**
  * Paints a percept collection to a debug drawing for the radar viewer
  */
  static void paintPerceptCollectionForRadarView(
    DebugDrawing& perceptsDrawing, 
    const LandmarksPercept& landmarksPercept,
    const BallPercept& ballPercept,
    const PlayersPercept& playersPercept,
    const ObstaclesPercept& obstaclesPercept,
    const LinesPercept& linesPercept);
  
  /**
  * Paints a percept collection to a debug drawing for the field view
  */
  static void paintPerceptCollectionForFieldView(
    DebugDrawing& perceptsDrawing, 
    const LandmarksPercept& landmarksPercept,
    const BallPercept& ballPercept,
    const PlayersPercept& playersPercept,
    const ObstaclesPercept& obstaclesPercept,
    const LinesPercept& linesPercept,
    const RobotPose& origin,
    const Player::teamColor teamColor);
  
  /**
  * Paints a Playerspercept to a debug drawing
  */
  static void paintPlayersPerceptForFieldView(DebugDrawing& playersDrawing,
    const PlayersPercept& playersPercept,
    const Pose2D& origin);
  
  /**
  * Paints a BallPercept to a debug drawing
  */
  static void paintBallPerceptForFieldView(DebugDrawing& ballDrawing,
    const BallPercept& ballPercept,
    const Pose2D& origin);
  
  /**
  * Paints a LinesPercept to a debug drawing
  */
  static void paintLinesPerceptForFieldView(DebugDrawing& linesDrawing,
    const LinesPercept& linesPercept,
    const Pose2D& origin);

  /**
  * Paints a ObstaclesPercept to a debug drawing
  */
  static void paintObstaclesPerceptForFieldView(DebugDrawing& obstaclesDrawing,
    const ObstaclesPercept& obstaclesPercept,
    const Pose2D& origin);

  /**
  * Paints a ObstaclesModel to a debug drawing for the field view
  */
  static void paintObstaclesModelForFieldView(DebugDrawing& obstaclesDrawing,
    const ObstaclesModel& obstaclesModel,
    const Pose2D& origin);
  
  /**
  * Paints a CollisionPercept to a debug drawing for the field view
  */
  static void paintCollisionPerceptForFieldView(
    DebugDrawing& collisionPerceptDrawing,
    const CollisionPercept& collisionPercept,
    const Pose2D& origin
    );
  
  /**
  * Paints a ObstaclesModel to a debug drawing for the radar view
  */
  static void paintObstaclesModelForRadarView(DebugDrawing& obstaclesDrawing,
    const ObstaclesModel& obstaclesModel);
  
  /**
  * Paints a LandmarksPercept to a debug drawing
  */
  static void paintLandmarksPerceptForFieldView(DebugDrawing& landmarksDrawing,
    const LandmarksPercept& landmarksPercept,
    const Pose2D& origin);
  
  /** 
  * Paints a world state to a debug drawing.
  */
  static void paintWorldState(
    DebugDrawing& debugDrawing,
    const RobotPose& robotPose,
    const BallModel& ballPosition,
    const PlayerPoseCollection& playerPoseCollection,
    Player::teamColor ownColor,
    unsigned long timestamp,
    bool oracled = false);
  
  static void paintAngleBoundaryToImage(
    DebugDrawing& perceptsDrawing, 
    const CameraMatrix& cameraMatrix,
    const CameraInfo& cameraInfo,
    const ConditionalBoundary boundary);
  
  /**
  * Paints a special percept to a debug drawing for the field view
  */
  static void paintSpecialPerceptForFieldView(
    DebugDrawing& specialPerceptDrawing, 
    const SpecialPercept& specialPercept,
    const RobotPose& origin,
    Player::teamColor teamColor);
  
private:
  /** 
  * Paints a robot to a debug drawing.
  */
  static void drawRobot(
    DebugDrawing& debugDrawing,
    Pose2D pose,
    const DebugDrawing::Color& color,
    const DebugDrawing::Color& headColor
    );
  
};
#endif // __DrawingMethods_h__


/*
* Change log :
* 
* $Log: DrawingMethods.h,v $
* Revision 1.7  2004/04/07 13:00:44  risler
* ddd checkin after go04 - second part
*
* Revision 1.2  2004/03/29 15:19:03  Marc
* Intruduced the Black and White Image
* Normal Images (not Jpeg) images were now send as Color Image with BW
*
* Revision 1.6  2004/02/06 10:16:23  juengel
* percepts for image view are painted in different debug drawings now
*
* Revision 1.5  2004/02/03 13:19:49  spranger
* renamed all references to  class BallPosition to BallModel
*
* Revision 1.4  2003/12/15 11:46:58  juengel
* Introduced CameraInfo
*
* Revision 1.3  2003/12/09 18:11:45  loetzsch
* Added parameter timestamp to
* Drawingmethods::paintWorldStateToDebugDrawing
*
* Revision 1.2  2003/12/07 18:59:27  loetzsch
* changed methods for field view
*
* Revision 1.1  2003/10/07 10:11:08  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/09/26 21:23:20  loetzsch
* renamed class JointState to CollisionPercept
*
* Revision 1.3  2003/09/26 15:28:24  juengel
* Renamed DataTypes to representations.
*
* Revision 1.2  2003/09/25 11:21:59  juengel
* Removed BlobCollection.
*
* Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.12  2003/06/13 14:52:37  juengel
* Added paintJointStateForFieldView.
*
* Revision 1.11  2003/05/26 08:22:44  juengel
* Added paintObstaclesModelForImageView
*
* Revision 1.10  2003/05/16 14:07:48  risler
* added radar view obstacles drawing
*
* Revision 1.9  2003/04/05 16:49:03  juengel
* Added specialPercept to DrawingMethods.
*
* Revision 1.8  2003/03/29 21:08:03  juengel
* Added paintObstaclesAndPSDPerceptForImageView.
*
* Revision 1.7  2003/03/29 13:44:37  roefer
* Ball percept visualization in image
*
* Revision 1.6  2003/03/28 14:32:45  juengel
* Added visualization of psdPercept.
*
* Revision 1.5  2003/03/11 11:27:49  juengel
* Added obstaclesPercept to drawingMethods.
*
* Revision 1.4  2002/12/16 12:05:07  dueffert
* doxygen bug fixed
*
* Revision 1.3  2002/12/14 19:45:54  roefer
* paintLinesPerceptToImageView added
*
* Revision 1.2  2002/12/14 17:56:08  roefer
* Names of drawing functions changed
*
* Revision 1.1  2002/11/12 10:52:52  juengel
* New debug drawing macros - second step
* -moved /Tools/Debugging/PaintMethods.h and . cpp
*  to /Visualization/DrawingMethods.h and .cpp
* -moved DebugDrawing.h and .cpp from /Tools/Debugging/
*  to /Visualization
*
* Revision 1.4  2002/11/11 11:27:10  juengel
* First Step: New debug drawing macros.
*
* Revision 1.3  2002/09/22 18:40:52  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/17 23:55:23  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.1  2002/09/10 15:53:58  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.3  2002/07/23 13:48:28  loetzsch
* - new streaming classes
* - removed many #include statements
* - exchanged StaticQueue by MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.2  2002/06/08 11:43:59  mkunz
* Special Percept on Field
*
* Revision 1.1.1.1  2002/05/10 12:40:33  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.14  2002/04/16 16:13:14  dueffert
* no message
*
* Revision 1.2  2002/04/12 07:10:38  juengel
* Player Specialist added.
*
* Revision 1.13  2002/04/08 19:53:14  juengel
* Drawing of percept collections in images added.
*
* Revision 1.12  2002/03/24 17:47:02  juengel
* LinesPercept defined and LinesPerceptor added.
*
* Revision 1.11  2002/03/23 22:25:44  roefer
* LandmarksPercepts now in player coordinates
*
* Revision 1.10  2002/03/06 10:20:55  juengel
* Drawing of field polygons depends on team color.
*
* Revision 1.9  2002/02/19 14:26:34  roefer
* drawArc removed
*
* Revision 1.8  2002/02/18 12:05:18  loetzsch
* Percept visualization continued
*
* Revision 1.7  2002/02/13 22:43:02  roefer
* First working versions of DefaultLandmarksPerceptor and MonteCarloSelfLocator
*
* Revision 1.6  2002/02/11 00:55:38  loetzsch
* ::Paint Methoden in ::paint umbenannt,
* BallPerzept- Visualisierung fertig
*
* Revision 1.5  2002/02/10 12:45:15  loetzsch
* radar viewer and percept visualization continued
*
* Revision 1.4  2002/02/05 03:37:57  loetzsch
* added a method for painting percept collections.
*
* replaced direct member access in WorldState
* and PerceptCollection members by
* inline const VALUE& get...() const   and
* inline void set...(const Value&) methods.
*
* Revision 1.3  2002/02/03 16:11:57  juengel
* wordStateOracle drawing added
*
* Revision 1.2  2002/02/03 14:37:58  juengel
* Drawing of the world state removed from Berlin2001BehaviorControl.
* Drawing method for world states added to DrawingMethods.
* Drawing of the world state added to the Processes with BehaviorControl.
*
* Revision 1.1  2002/01/04 14:36:00  juengel
* no message
*
*/
