/**
* @file SimRobotDocument.cpp
*
* Declaration of class CSimRobotDocument
*
* @author Thomas Rfer
* @author Martin Ltzsch
*/

#include "StdAfx.h"

#include "RobotControl.h"
#include "SimRobotDocument.h"
#include "SimRob95/SimRobot/sim3DWorld.h"
#include "Bars/SimulatorObjectViewerDlgBar.h"

CONNECTION* pStart = 0;

CONNECTION::CONNECTION(char* pScene)
{
  m_pScene = pScene;
  m_pNext = pStart;
  pStart = this;
}

void WinExit(const char* pMessage)
{
  AfxMessageBox(pMessage,MB_OK | MB_ICONSTOP);
  exit(1);
}


CSimRobotDocument::CSimRobotDocument()
{
  m_pSim = 0;
  m_pController = 0;
  m_bRunning = FALSE;
  m_bStep = 0;
  m_sStatusMessage = "";
  pSimulatorObjectViewerDlgBar = 0;
}

CSimRobotDocument::~CSimRobotDocument()
{
  if (m_pController)
  {
    m_pController->OnDestroy();
    delete m_pController;
    m_pController = 0;
  }
  if (m_pSim)
  {
    delete m_pSim;
    m_pSim = 0;
  }
}



char ErrorMessage[200];
int ErrorLine,
    ErrorColumn;

void CSimRobotDocument::Compile()
{
// check if document has a name
  if(m_filePath == "")
  {
    return;
  }
// set directory for relative paths in texture load commands
  CString sPathName = m_filePath.Left(m_filePath.ReverseFind('\\'));
  SetCurrentDirectory(sPathName);
  CString sTempName = m_filePath.Left(m_filePath.ReverseFind('.')) + ".tmp";
  GenerateSceneFile(m_filePath,sTempName);
// compile
  *ErrorMessage = 0;
  sim3DTakeLastError();
  sim3DErrorOutputOn();
  m_pSim = new SIMULATION((const char*) sTempName);
  sim3DErrorOutputOff();
// check for an error
  if(*ErrorMessage)
  {
    delete m_pSim;
    m_pSim = 0;
    m_sStatusMessage = ErrorMessage;
    MessageBeep(MB_ICONHAND);
    sim3DTakeLastError();
  }
  else
  {
// search for controller
    STRING sScene = m_pSim->GetObject("")->GetName();
    CONNECTION* pThis = pStart;
    while (pThis && !(sScene == pThis->m_pScene))
	    pThis = pThis->m_pNext;
    if(pThis)
    {
// create controller
      try
      {
        m_pController = pThis->CreateController();
      }
      catch(XCONTROLLER x)
      {
      }
    }
  }
}

void CSimRobotDocument::GenerateSceneFile(const CString& from,const CString& to)
{
  CStdioFile fromFile(from,CFile::modeRead | CFile::typeText),
             toFile(to,CFile::modeCreate | CFile::modeWrite | CFile::typeText);
  CString line;
  CRobotControlSimulatedRobots& robots = getSimulatedRobots();

  while(fromFile.ReadString(line))
  {
    if(line.Left(3) == "//R") // active robots
    {
      int index = line[3] - '1';
      if(robots.getState(index) == CRobotControlSimulatedRobots::activeWithImages
        || robots.getState(index) == CRobotControlSimulatedRobots::activeWithoutImages)
        line = line.Mid(4);
      else
        continue;
    }
    else if(line.Left(3) == "//E") // extras
    {
      int index = line[3] - '1';
      if(robots.getState(index) == CRobotControlSimulatedRobots::passive) // if robot is passive
        line = line.Mid(4);
      else
        continue;
    }
    toFile.WriteString(line);
    toFile.WriteString("\n");
  }
}

STRING CSimRobotDocument::GetSelectedObject()
{
  return "";
}

void CSimRobotDocument::NewSection(const STRING& sSection)
{
  int nCount;
  if(!m_map.Lookup(sSection,(void*&)nCount))
    nCount = 1;
  else
    nCount++;
  m_map.SetAt(sSection,(void*&)nCount);
  char buf[10];
  sprintf(buf,"%d",nCount);
  m_sSection = sSection + buf;
}

void CSimRobotDocument::WriteString(const STRING& sName,const STRING& sValue)
{
  AfxGetApp()->WriteProfileString(m_sSection,sName,sValue);
}

void CSimRobotDocument::WriteInt(const STRING& sName,int nValue)
{
  char buf[30];
  sprintf(buf,"%d",nValue);
  WriteString(sName,buf);
}

STRING CSimRobotDocument::ReadString(const STRING& sName)
{
  CString buf = AfxGetApp()->GetProfileString(m_sSection,sName);
  return (const char*) buf;
}

int CSimRobotDocument::ReadInt(const STRING& sName)
{
  int n;
  if(sscanf(ReadString(sName)," %d",&n) > 0)
    return n;
  else
    return 0;
}


BOOL CSimRobotDocument::IsRunning()
{
  return m_bRunning;
}

void CSimRobotDocument::OnSimReset() 
{
  BOOL bWasRunning = m_bRunning;
  m_bRunning = FALSE;
  if(m_pController)
  {
    m_pController->OnDestroy();
    delete m_pController;
    m_pController = 0;
  }
  if(m_pSim)
  {
    delete m_pSim;
    m_pSim = 0;
  }
  Compile();
  if(m_pSim)
  {
    m_bRunning = bWasRunning;
  }

  sim3DTakeLastError();
}


void CSimRobotDocument::OnSimStart() 
{
  if(m_bRunning)
    m_bRunning = FALSE;
  else
  {
    if(!m_pSim)
      OnSimReset();
    if(m_pSim)
      m_bRunning = TRUE;
  }
}


void CSimRobotDocument::OnSimStep() 
{
  if(m_bRunning)
    return;
  if(!m_pSim)
    OnSimReset();
  if(m_pSim)
    m_bStep = TRUE;
}


void CSimRobotDocument::OnIdle()
{
  if(m_bStep || m_bRunning)
  {
    m_bStep = FALSE;
    sim3DTakeLastError();
    try
    {
      STRING sMessage;
      if(m_pController && !m_pController->SetActors(sMessage))
      {
        m_bRunning = FALSE;
        AfxGetMainWnd()->PostMessage(WM_USER);
      }
      m_pSim->DoTimeStep();
      pSimulatorObjectViewerDlgBar->Repaint();

      m_sStatusMessage = (const char *) sMessage;
    }
    catch(XCONTROLLER& x)
    {
//###      switch(x.m_xType)
//###      {
//###        case XCONTROLLER::InvalidObject:
//###          m_sStatusMessage.LoadString(IDS_INVALIDOBJECT);
//###          break;
//###        case XCONTROLLER::InvalidSensorPort:
//###          m_sStatusMessage.LoadString(IDS_INVALIDSENSOR);
//###          break;
//###        case XCONTROLLER::InvalidActorPort:
//###          m_sStatusMessage.LoadString(IDS_INVALIDACTOR);
//###          break;
//###      }
      char buf[200];
      sprintf(buf,m_sStatusMessage,(const char*) x.m_sName);
//###      m_sStatusMessage = buf;
//###      m_pController->OnDestroy();
//###      delete m_pController;
//###      m_pController = 0;
      m_bRunning = FALSE;
//###      MessageBeep(MB_ICONHAND);
    }
    if(m_bRunning)
      AfxGetMainWnd()->PostMessage(WM_USER);
  }
//###  ((CFrame*) AfxGetMainWnd())->SetMessageText(m_sStatusMessage);
}

void CSimRobotDocument::UpdateFrameCounts()
{
}

void CSimRobotDocument::OnMouseClick(const CLICKINFO& ci)
{
  if(m_pController)
    m_pController->OnMouseClick(ci);
}

void CSimRobotDocument::OnMouseDoubleClick(const CLICKINFO& ci)
{
  if(m_pController)
    m_pController->OnMouseDoubleClick(ci);
}

void CSimRobotDocument::OnBeginDrag(const CLICKINFO& ci)
{
  if(m_pController)
    m_pController->OnBeginDrag(ci);
}

void CSimRobotDocument::OnDrag(const CLICKINFO& ci)
{
  if(m_pController)
    m_pController->OnDrag(ci);
}

void CSimRobotDocument::OnEndDrag(const CLICKINFO& ci)
{
  if(m_pController)
    m_pController->OnEndDrag(ci);
}

/* 
* Change Log:
*
* $Log: SimRobotDocument.cpp,v $
* Revision 1.3  2003/12/06 06:31:20  loetzsch
* no message
*
* Revision 1.2  2003/11/30 01:53:21  loetzsch
* prepared RobotControl port to Visual C++ .Net
*
* Revision 1.1  2003/10/07 10:11:08  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/05/11 23:36:18  dueffert
* Depend now works with RobotControl too
*
* Revision 1.5  2003/03/23 19:14:21  loetzsch
* finished GUI for 8 simulated robots
* - added the passive state
* - added the state[8] variable to CRobotControlSimulatedRobots
* - Repaint of object viewer after reset
* - Resend debug key table after reset
* - don't stop simulation when selected robot changes
*
* Revision 1.4  2003/03/23 14:36:08  roefer
* GenerateSceneFile added
*
* Revision 1.3  2003/03/22 22:37:52  loetzsch
* finished GUI and message routing for 8 simulated robots
* almost all functionallity for simulated robots now is encapsulated in class
*   CRobotControlSimulatedRobots
*
* Revision 1.2  2002/11/20 13:37:31  dueffert
* missing or wrong doxygen file corrected
*
* Revision 1.1  2002/09/10 15:49:12  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.3  2002/08/07 10:50:21  loetzsch
* - removed direct access to the dialogs
* - faster starting of RobotControl application and main window
* - general clean up
*
*/