/**
* @file Controller.cpp
*
* Implementation of class VIEW, CLICKINFO, CONTROLLER
*/
#include "StdAfx.h"

#include "SimRobotDocument.h"
#include "../RobotControlSimulatedRobots.h"
#include <stdlib.h>
#include "Controller.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////////
// class VIEW

VIEW::VIEW (const STRING& sName,INTEGER xSize,INTEGER ySize,INTEGER numClasses)
 : SIM3DMultiSensor(sName)
{
  SizeX = xSize;
  SizeY = ySize;
  KnowClasses = TRUE;
  for (int i = 1; i < numClasses; i++)
    Classes.AddElement (i);
  ResizeBuffer ();
}

void VIEW::CalcSensor()
{
  CalcView();
}

//////////////////////////////////////////////////////////////////////////
// class CLICKINFO

CLICKINFO::CLICKINFO(SIM3DOBJECT obj,const VECTOR& v,int x,int y,int nZoom,double dDist)
:m_v(v / double(nZoom)),
  m_m(TurnYZMatrix ((double) y)
      * TurnXYMatrix ((double) x)
      * MATRIX(VECTOR (nZoom,0,0),VECTOR (0,nZoom,0),VECTOR (0,0,nZoom))
      ,VECTOR (0,0,0)),
  m_mY(TurnYZMatrix ((double) y)
      * MATRIX(VECTOR (nZoom,0,0),VECTOR (0,nZoom,0),VECTOR (0,0,nZoom))
      ,VECTOR (0,0,0))
{
  m_obj = obj; 
  m_dZoom = nZoom,
  m_dDist = dDist;
  m_x = x;
  m_y = y;
}

double CLICKINFO::DistanceTo(SIM3DOBJECT obj) const
{
  VECTOR v = obj->Transform.Offset;
  while(obj->Owner && obj->Owner != m_obj)
  {
    obj = obj->Owner;
    v = obj->Transform * v;
  }
  if(obj->Owner == m_obj)
  {
    v = m_m * v;
    v = v * ((m_dDist - 1) / (m_dDist - v.z) / m_dZoom);
    v.z = 0;
    return abs(v - m_v);
  }
  else
    return HUGE_VAL;
}

VECTOR CLICKINFO::SceneToScreenY(VECTOR v) const
{
  v = m_mY * v;
  v = v * ((m_dDist - 1) / (m_dDist - v.z) / m_dZoom);
  v.z = 0;
  return v;
}

//////////////////////////////////////////////////////////////////////////
// class CONTROLLER

CONTROLLER::CONTROLLER()
{
  CSimRobotDocument* pSimRobotDocument = getSimulatedRobots().getSimRobotDocument();
  m_pSim = pSimRobotDocument->GetSimulation();
}

SENSORPORT CONTROLLER::AddView(VIEW* pView)
{
  ((SIM3DGroup*) GetObject(""))->AddObject((SIM3DElement*) pView);
  return GetSensorPort(ObjectName((SIM3DElement*) pView)+STRING(".VALUE"));
}

//////////////////////////////////////////////////////////////////////////
// Clean all error flags

void CONTROLLER::Clean()
{
  sim3DTakeLastError();
}

//////////////////////////////////////////////////////////////////////////
// Check WasOk for many types and throw XCONTROLLER on error

void CONTROLLER::Check()
{
  if(!m_pSim->WasOk())
    throw XCONTROLLER(XCONTROLLER::InvalidObject);
}

INTEGER CONTROLLER::Check(INTEGER i)
{
  if(!m_pSim->WasOk())
    throw XCONTROLLER(XCONTROLLER::InvalidObject);
  return i;
}

STRING CONTROLLER::Check(const STRING& s)
{
  if(!m_pSim->WasOk())
    throw XCONTROLLER(XCONTROLLER::InvalidObject);
  return s;
}

SIM3DOBJECT CONTROLLER::Check(SIM3DOBJECT s)
{
  if(!m_pSim->WasOk())
    throw XCONTROLLER(XCONTROLLER::InvalidObject);
  return s;
}

PSHORTREAL CONTROLLER::Check(PSHORTREAL p)
{
  if(!m_pSim->WasOk())
    throw XCONTROLLER(XCONTROLLER::InvalidObject);
  return p;
}

MOVEMATRIX CONTROLLER::Check(const MOVEMATRIX& m)
{
  if(!m_pSim->WasOk())
    throw XCONTROLLER(XCONTROLLER::InvalidObject);
  return m;
}

//////////////////////////////////////////////////////////////////////////
// Check WasOk for SIMULATION API and throw XCONTROLLER on error

ACTORPORT CONTROLLER::GetActorPort(const STRING& portname)
{
  Clean();
  ACTORPORT ap = m_pSim->GetActorPort(portname);
  if(!m_pSim->WasOk())
    throw XCONTROLLER(XCONTROLLER::InvalidActorPort,portname);
  return ap;
}

SENSORPORT CONTROLLER::GetSensorPort(const STRING& portname)
{
  Clean();
  SENSORPORT sp = m_pSim->GetSensorPort(portname);
  if(!m_pSim->WasOk())
    throw XCONTROLLER(XCONTROLLER::InvalidSensorPort,portname);
  return sp;
}

INTEGER CONTROLLER::GetSensorDim(SENSORPORT sp)
{
  Clean();
  return Check(m_pSim->GetSensorDim(sp));
}

INTEGER CONTROLLER::GetSensorDimSize(SENSORPORT sp,INTEGER d)
{
  Clean();
  return Check(m_pSim->GetSensorDimSize(sp,d));
}

void CONTROLLER::GetSensorValue(SENSORPORT sp,PSHORTREAL value)
{
  Clean();
  m_pSim->GetSensorValue(sp,value);
  Check();
}

void CONTROLLER::SetActorValue(ACTORPORT ap, REAL value)
{
  Clean();
  m_pSim->SetActorValue(ap,value);
  Check();
}

PSHORTREAL CONTROLLER::NewSensorBuffer(SENSORPORT sp)
{
  Clean();
  return Check(m_pSim->NewSensorBuffer(sp));
}

void CONTROLLER::DeleteSensorBuffer(SENSORPORT sp,PSHORTREAL buffer)
{
  Clean();
  m_pSim->DeleteSensorBuffer(sp,buffer);
  Check();
}

INTEGER CONTROLLER::SensorPortCount(SIM3DOBJECT obj)
{
  Clean();
  return Check(m_pSim->SensorPortCount(obj));
}

INTEGER CONTROLLER::ActorPortCount(SIM3DOBJECT obj)
{
  Clean();
  return Check(m_pSim->ActorPortCount(obj));
}

STRING CONTROLLER::SensorPortName(SIM3DOBJECT obj,INTEGER n)
{
  Clean();
  return Check(m_pSim->SensorPortName(obj,n));
}

STRING CONTROLLER::ActorPortName(SIM3DOBJECT obj,INTEGER n)
{
  Clean();
  return Check(m_pSim->ActorPortName(obj,n));
}

SIM3DOBJECT CONTROLLER::GetObject(const STRING& name)
{
  Clean();
  return Check(m_pSim->GetObject(name));
}

STRING CONTROLLER::ObjectName(SIM3DOBJECT obj)
{
  Clean();
  return Check(m_pSim->ObjectName(obj));
}

STRING CONTROLLER::ObjectRemark(SIM3DOBJECT obj)
{
  Clean();
  return Check(m_pSim->ObjectRemark(obj));
}

STRING CONTROLLER::ObjectClass(SIM3DOBJECT obj)
{
  Clean();
  return Check(m_pSim->ObjectClass(obj));
}

INTEGER CONTROLLER::SubObjectCount(SIM3DOBJECT obj)
{
  Clean();
  return Check(m_pSim->SubObjectCount(obj));
}

SIM3DOBJECT CONTROLLER::SubObject(SIM3DOBJECT obj,INTEGER n)
{
  Clean();
  return Check(m_pSim->SubObject(obj,n));
}

INTEGER CONTROLLER::MacroCount()
{
  Clean();
  return Check(m_pSim->MacroCount());
}

SIM3DOBJECT CONTROLLER::GetMacro(INTEGER n)
{
  Clean();
  return Check(m_pSim->GetMacro(n));
}

SIM3DOBJECT CONTROLLER::GetMacro(const STRING& s)
{
  Clean();
  return Check(m_pSim->GetMacro(s));
}

SIM3DOBJECT CONTROLLER::AddMacroAs(SIM3DOBJECT obj,const STRING& asname)
{
  Clean();
  return Check(m_pSim->AddMacroAs(obj,asname));
}

void CONTROLLER::RemoveObject(SIM3DOBJECT obj)
{
  Clean();
  m_pSim->RemoveObject(obj);
  Check();
}

MOVEMATRIX CONTROLLER::GetLocation(SIM3DOBJECT obj)
{
  Clean();
  return Check(m_pSim->GetLocation(obj));
}

void CONTROLLER::SetLocation(SIM3DOBJECT obj,const MOVEMATRIX& m)
{
  Clean();
  m_pSim->SetLocation(obj,m);
  Check();
}

/*
 * Change log:
 *
 * $Log: Controller.cpp,v $
 * Revision 1.2  2004/04/26 15:58:59  thomas
 * added new project RobotRemote based on ATHAiboControl
 *
 * Revision 1.1  2003/10/07 10:11:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/05/11 23:36:18  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.5  2003/03/28 11:37:31  timrie
 * Added changelog
 *
 */
