/**
* @file RobotControlSimulatedRobots.h
*
* Definition of class CRobotControlSimulatedRobots
*
* @author Martin Ltzsch
* @author Thomas Rfer
*/

#ifndef __RobotControlSimulatedRobots_h_
#define __RobotControlSimulatedRobots_h_

#include "SimRobot/SimRobotDocument.h"
#include "Platform/Win32/ForRobotControl/OVirtualRobotComm.h"

/** 
* @class CRobotControlSimulatedRobots
*
* Encapsulates the simulated robots, their interfaces to RobotControl and the simulator.
* 
* @author Martin Ltzsch
* @author Thomas Rfer
*/
class CRobotControlSimulatedRobots
{
public:
  /** Possible states of a simulated robot */
  enum State {disabled, passive, activeWithoutImages, activeWithImages};

  /** Default constructor */
  CRobotControlSimulatedRobots();

  /** Destructor */
  ~CRobotControlSimulatedRobots();

  /** Creates the simulated robots */
  void create();

  /** Manages the communication with the simulated robots */
  void onIdle();

  /** Should be called for resetting the simulation */
  void OnSimReset();

  /** Should be called for starting/stopping the simulation */
  void OnSimStart();

  /** Should be called for performing 1 simulation step */
  void OnSimStep();

  /** 
  * If true, then the configuration of the simulated robots changed and the simulator 
  * needs to be restarted. No communication with the simulated robots should occur then
  */
  bool getDoRestart() const;

  /** Sets the state and the design of a simulated robot */
  void setStateAndDesign(int robot, State newState, RobotDesign::Design newDesign);

  /** Returns the state of a simulated robot */
  CRobotControlSimulatedRobots::State getState(int robot) const;

  /** Selects a simulated robot. If -1, then no robot is selected. */
  void setSelectedRobot(int robot);

  /** returns the design of a robot */
  RobotDesign::Design getDesign(int robot) const;

  /** Returns the selected simulated robot. If -1, then no robot is selected. */
  int getSelectedRobot() const;

  /** If oracled world state shall be sent to the simulated robots */
  bool getSendOracle() const;

  /** Sets the sendOracle flag */
  void setSendOracle(bool doSend);

  /** Returns the SimRobot document */
  CSimRobotDocument* getSimRobotDocument() const;

  /** pointers to the simulated robots */
  OVirtualRobotComm* robots[8];

  /** the design of each simuated robot */
  RobotDesign::Design design[8];

  /** 
  * Returns whether the variable simulatedRobotsAreDisabled is true. 
  * Returns not whether one of the robots is active
  */ 
  bool getSimulatedRobotsAreDisabled() const;

  /** 
  * Sets the variable simulatedRobotsAreDisabled.
  * Does not disable or deactivate any robots.
  */
  void setSimulatedRobotsAreDisabled(bool disabled);

  
private:
  /** 
  * A flag that globally disables all simulated robots. If true,
  * the robots are nevertheless constructed but not triggered.
  * This is used to bypass messages from physical robots directly
  * to the queue to gui without sending them through the simulated 
  * robots.
  */
  bool simulatedRobotsAreDisabled;

  /** The states of the simulated robots */
  State state[8];

  /** 
  * Stops the simulator and the communication with the simulated robots
  * because a change in the configuration occurred 
  */
  void stop();

  /** Selects the first active robot or -1. Used when the selected robot becomes disabled */
  void selectFirstActiveRobot();

  /** A pointer to the SimRobot document (contains the simulation) */
  CSimRobotDocument* pSimRobotDocument;

  /** 
  * If true, then the configuration of the simulated robots changed and the simulator 
  * needs to be restarted.
  */
  bool doRestart;

  /** The selected simulated robot. If -1, then no robot is selected. */
  int selectedRobot;

  /** determines, if world models computed by the oracle should be sent to the simulated robots */ 
  bool sendOracle;
  
 };

/** Returns a reference to the simulated robots instance inside RobotControl */
CRobotControlSimulatedRobots& getSimulatedRobots();

#endif //__RobotControlSimulatedRobots_h_

/*
* Change Log:
*
* $Log: RobotControlSimulatedRobots.h,v $
* Revision 1.3  2004/01/02 16:03:34  loetzsch
* added support for different robot designs to CRobotControlSimulatedRobots
* and to the tool bar
*
* Revision 1.2  2003/12/12 17:12:23  loetzsch
* With the button "Disable simulated robots" all simulated robots can be
* disabled. But this does not deactivate any simulated robots.
*
* If the simulated robots are disabled, messages from physical robots or a
* log file are directly sent to the queue "toGUI". (One does not need to
* deactivate all simulated robots anymore if one wants to view images from a log file
* without sending them to the selected simulated robot.)
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/30 10:51:12  dueffert
* typos fixed
*
* Revision 1.2  2003/09/26 11:40:12  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/03/23 19:14:21  loetzsch
* finished GUI for 8 simulated robots
* - added the passive state
* - added the state[8] variable to CRobotControlSimulatedRobots
* - Repaint of object viewer after reset
* - Resend debug key table after reset
* - don't stop simulation when selected robot changes
*
* Revision 1.5  2003/03/22 22:37:52  loetzsch
* finished GUI and message routing for 8 simulated robots
* almost all functionallity for simulated robots now is encapsulated in class
*   CRobotControlSimulatedRobots
*
* Revision 1.4  2003/03/22 16:39:33  loetzsch
* continued GUI for simulated robots
*
* Revision 1.3  2003/03/21 14:25:38  loetzsch
* continued support for 8 simulated robots
*
* Revision 1.2  2003/03/20 23:42:17  loetzsch
* changed cast of AfxGetApp() to CRobotControlApp*
* to getRobotControlApp()
*
* Revision 1.1  2003/03/20 20:36:22  loetzsch
* renamed SimulatorToolbar to Simulated Robots toolbar
* added class CRobotControlSimulatedRobots
* started implementing the GUI for 8 simulated robots
*
*/
