/**
* @file RobotControlMessageHandler.cpp
*
* Implementation of the main MessageHandler derivates in the RobotControl application.
*
* @author Martin Ltzsch
*/

#include "StdAfx.h"

#define USE_GETMESSAGEIDNAME

#include "RobotControlMessageHandler.h"
#include "RobotControlQueues.h"
#include "RobotControlSimulatedRobots.h"
#include "RobotControlMainFrame.h"

#define HAVE_BOOLEAN
#include "Modules/ImageProcessor/ImageProcessorTools/FastSUSANNoiseReduction.h"
#include "Representations/Perception/JPEGImage.h"
#include "Representations/Perception/LowResImage.h"

bool CMessageHandlerForQueueFromPhysicalRobots::handleMessage(InMessage& message)
{
  // notify the log player queue
  getQueues().logPlayer.handleMessage(message);
  
  // if no simulated robot is activated or if all simulated robots are disabled 
  // in the "SimulatedRobots" toolbar, all messages are directly sent to the GUI
  if (getSimulatedRobots().getSelectedRobot() == -1
    || getSimulatedRobots().getSimulatedRobotsAreDisabled())
  {
    message >> getQueues().toGUI;
    return true;
  }
  else
  {
    // sent some messages to the selected simulated robot, all others to the GUI
    switch(message.getMessageID())
    {
    case idJPEGImage:
      if(getQueues().logPlayer.isSmoothingEnabled())    
      {
        // strange behavior: if all images are created on the stack, RobotControl will crash
        Image image;
        CameraMatrix cameraMatrix;
        JPEGImage jpegImage;
        message.bin >> jpegImage >> cameraMatrix;

        Image tempImage;
        jpegImage.toImage(tempImage);
        FastSUSANNoiseReduction noiseFilter(16);
        noiseFilter.getFilteredImage(tempImage, image);

        for(int y = 0; y < image.cameraInfo.resolutionHeight; ++y)
          for(int c = 3; c < 6; ++c)
            memset(&image.image[y][c][0], 128, image.cameraInfo.resolutionWidth);

        MessageQueue m;
        m.out.bin << SEND_IMAGE(image, cameraMatrix);
        m.out.finishMessage(idImage);
        m.handleAllMessages(*this);
      }
      else 
      {
        message >> getQueues().toSimulated.selectedRobot;
      }
      return true;
    case idLowResImage:
      {
        Image image;
        CameraMatrix cameraMatrix;

        message.bin >> LowResImage(image) >> cameraMatrix;

        getQueues().toSimulated.selectedRobot.out.bin << SEND_IMAGE(image, cameraMatrix);
        getQueues().toSimulated.selectedRobot.out.finishMessage(idImage);
      }
      return true;
    case idImage:
    case idWorldState:
    case idPercepts:
    case idSpecialPercept:
    case idOdometryData:
      message >> getQueues().toSimulated.selectedRobot;
      return true;
    default:
      message >> getQueues().toGUI;
      return true;
    }
  }
}

bool CMessageHandlerForQueueFromSimulatedRobots::handleMessage(InMessage& message)
{
  // until now, all messages are directly copied to the queue to gui
  message >> getQueues().toGUI;
  return true;
}

bool CMessageHandlerForQueueToGUI::handleMessage(InMessage& message)
{
  MessageID messageID = message.getMessageID();
  
  // If more than one simulated physical or simulated robot is connected,
  // messages from multiple robots arrive in this queue.
  // 
  // Dialogs that can display data for only one robot should
  // process the message only if 
  // 
  // getQueues().isFromSelectedOrUndefinedRobot(message)
  //
  // is true
  //
  switch(messageID)
  {
  case idText:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_MESSAGEVIEWER,message);
    }
    return true;
    
  case idDebugImage:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_COLORTABLE64,message);
      message.resetReadPosition();
    }
  case idDebugDrawingFinished:
  case idDebugDrawing:
  case idDebugDrawing2:
  case idDebugColorClassImage:
  case idOracledWorldState:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.debugDrawingManager.messageHandler.handleMessage(message);
    }
    return true;
    
  case idJPEGImage:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      Image tempImage, image;
      JPEGImage jpegImage;
      CameraMatrix cameraMatrix;
      message.bin >> jpegImage >> cameraMatrix;
      if(getQueues().logPlayer.isSmoothingEnabled())    
      {
        jpegImage.toImage(tempImage);
        FastSUSANNoiseReduction noiseFilter(16);
        noiseFilter.getFilteredImage(tempImage, image);      
      }
      else 
      {
        jpegImage.toImage(image);
      }
      MessageQueue m;
      m.out.bin << SEND_IMAGE(image, cameraMatrix);
      m.out.finishMessage(idImage);
      m.handleAllMessages(*this);
      return true;
    }
    
  case idImage:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      /*{
        Image image;
        message.bin >> image;
        OutBinaryFile("green.img") << image;
        message.resetReadPosition();
      }*/
      mainFrame.debugDrawingManager.messageHandler.handleMessage(message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_COLORTABLE64,message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_COLOR_SPACE,message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_HSICOLORTABLETOOL,message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_RADAR_VIEWER_3D,message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_RADAR_VIEWER,message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_TSLCOLORTABLETOOL,message);
    }
    return true;
    
    
  case idParametersFromGridImageProcessor2:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_COLOR_SPACE,message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_GRID_IMAGE_PROCESSOR_2,message);
    }
    return true;
    
  case idColorTable64:
  case idColorTableReferenceColor:
  case idColorTableCuboids:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_COLOR_SPACE,message);
    }
    return true;
    
  case idSensorData:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_JOINT_VIEWER,message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_MOFTESTER,message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_RADAR_VIEWER_3D,message);
    }
    return true;
    
  case idJointData:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_JOINT_VIEWER,message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_RADAR_VIEWER_3D,message);
    }
    return true;
    
  case idStopwatch:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_TIME_DIAGRAM,message);
    }
    return true;
    
  case idHistogram:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_HISTOGRAM,message);
    }
    return true;
    
  case idWorldState:
  case idPercepts:
      if (getQueues().isFromSelectedOrUndefinedRobot(message))
      {
          mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_PERCEPTSCOUNTER,message);
      }
      return true;
    

  case idSpecialPercept:
    {
      mainFrame.debugDrawingManager.messageHandler.handleMessage(message);
      message.resetReadPosition();
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_WORLDSTATE_ANALYSIS,message);
      
      if (getQueues().isFromSelectedOrUndefinedRobot(message))
      {
        message.resetReadPosition();
        mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_RADAR_VIEWER_3D,message);
        message.resetReadPosition();
        mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_RADAR_VIEWER,message);
        message.resetReadPosition();
        mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_VALUE_HISTORY,message);
        message.resetReadPosition();
        mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_POTENTIAL_FIELD_VIEWER,message);
      }
      return true;
    }
    
  case idExecutedMotionRequest:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_VALUE_HISTORY,message);
    }
    return true;
    
  case idXabsl2DebugMessage:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_XABSL2,message);
    }
    return true;
    
  case idInvKinWalkingParameters:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_INVKINPARAMETERS,message);
    }
    return true;
    
  case idUDParameters:
    if (getQueues().isFromSelectedOrUndefinedRobot(message))
    {
      mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_UDPARAMETERSET,message);
    }
    return true;
    
  case idSingleBeliefs:
	case idChoosenOption:
	case idChoosenTacticEntry:
  case idTimeWeights:
    mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_OPTIONRATING,message);
    return true;

  case idKalmanData:
    mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_KALMAN_SETTINGS,message);
    return true;

  case idMotionRatingBehaviorMessage:
    mainFrame.handleMessageInDialog(IDD_DIALOG_BAR_MOTIONOPTIMIZER,message);
    return true;

  default:
    getQueues().toGUI.out.text << "Unhandled debug message in queue to GUI ("
      << getMessageIDName(message.getMessageID()) << ")";
    getQueues().toGUI.out.finishMessage(idText);
    return false;
    }
    
    return false;
}


/*
* Change Log:
*
* $Log: RobotControlMessageHandler.cpp,v $
* Revision 1.28  2004/06/04 13:45:56  hodie
* New Dialog to count percepts.
* Still a lot a more to implement but working.
*
* Revision 1.27  2004/04/17 21:50:50  roefer
* New color corrector (BB/DDD mix) used in BB2004ImageProcessor and Simulator
* Displaying corrected images as debug image
*
* Revision 1.26  2004/04/16 17:17:49  roefer
* Correcting stack overflows
*
* Revision 1.25  2004/04/09 11:35:53  roefer
* Bremen Byters German Open check-in
*
* Revision 1.24  2004/04/07 13:00:47  risler
* ddd checkin after go04 - second part
*
* Revision 1.6  2004/04/07 11:44:05  risler
* added sending low res images
* added Image::setCameraInfo
*
* Revision 1.5  2004/04/06 17:10:25  risler
* renamed idOldImage to idLowResImage
*
* Revision 1.4  2004/04/06 13:19:39  risler
* cleaned up and improved high resolution image support
*
* Revision 1.3  2004/03/29 20:44:41  risler
* bugfix old logfile support
*
* Revision 1.2  2004/03/29 16:33:10  risler
* old logfile support
*
* Revision 1.1.1.1  2004/03/29 08:28:43  Administrator
* initial transfer from tamara
*
* Revision 1.23  2004/03/22 10:19:32  tim
* added potential field viewer
*
* Revision 1.22  2004/03/17 17:24:47  risler
* ColorTable64Dlg can segment debug images
*
* Revision 1.21  2004/03/17 16:18:50  thomas
* added preversion of motion optimisation with behaviour, selflocator, headcontrol and robotcontrol dialog
*
* Revision 1.20  2004/03/11 14:27:32  uhrig
* KalmanBallLocator was added
*
* Revision 1.19  2004/03/09 15:44:55  tim
* changed color correction
*
* Revision 1.18  2004/02/29 17:29:54  dueffert
* (evolved) UDParameters now get to its dialog
*
* Revision 1.17  2004/02/23 12:40:39  roefer
* Minor change
*
* Revision 1.16  2004/02/21 20:08:13  roefer
* Minor change
*
* Revision 1.15  2004/02/21 18:36:13  roefer
* Color corrector added
*
* Revision 1.14  2004/02/16 12:26:40  nistico
* Added noise reduction functionality for jpeg images in log file player
*
* Revision 1.13  2004/02/02 10:10:54  juengel
* Redesign of RadarViewer.
*
* Revision 1.12  2004/01/28 21:54:11  kerdels
* fixed aging...
*
* Revision 1.11  2004/01/06 00:01:57  kerdels
* Aging im TacticChooser hinzugefgt,
* OptionRatingDlgBar um choosenTacticEntry erweitert
*
* Revision 1.10  2003/12/28 23:00:12  kerdels
* tactic chooser berarbeitet
*
* Revision 1.9  2003/12/28 19:03:29  kerdels
* option rating dialog added
*
* Revision 1.8  2003/12/13 11:31:14  loetzsch
* bug fix in the handling of idJointData
*
* Revision 1.7  2003/12/12 17:30:15  loetzsch
* changed a hack:
* The WorldStateAnalysisDlgBar is not invoked for all messages, but only
* for world states and percepts.
*
* Revision 1.6  2003/12/12 17:12:23  loetzsch
* With the button "Disable simulated robots" all simulated robots can be
* disabled. But this does not deactivate any simulated robots.
*
* If the simulated robots are disabled, messages from physical robots or a
* log file are directly sent to the queue "toGUI". (One does not need to
* deactivate all simulated robots anymore if one wants to view images from a log file
* without sending them to the selected simulated robot.)
*
* Revision 1.5  2003/12/12 10:16:30  hamerla
* added Simulator showings to WorldstateAnalysis
*
* Revision 1.4  2003/12/09 19:49:24  loetzsch
* Renamed some of the main queues of RobotControl.
*
* Added possibility to send messages to specific simulated or physical robots.
*
* Revision 1.3  2003/12/07 19:02:31  loetzsch
* In queueToGUI now messages from all robots instead of from the
* selected ones are  handled.
*
* The message handling for the debug drawing manager is now done in a
* separate class
*
* Revision 1.2  2003/12/06 16:44:45  loetzsch
* improved message handling
*
* Revision 1.1  2003/12/06 06:31:19  loetzsch
* no message
*
*/
