/**
* @file ToolBarEx.cpp
*
* Implementation of the CToolBarEx class.
*
* Copyright (c) 2001 by Nikolay Denisov. All rights reserved.
*
* This code is free for personal and commercial use, providing this 
* notice remains intact in the source files and all eventual changes are
* clearly marked with comments.
*
* You must obtain the author's consent before you can include this code
* in a software library.
*
* No warrantee of any kind, express or implied, is included with this
* software; use at your own risk, responsibility for damages (if any) to
* anyone resulting from the use of this software rests entirely with the
* user.
*
* Please email bug reports, bug fixes, enhancements, requests and
* comments to: nick@actor.ru
*/

#include "StdAfx.h"

#include "SizableReBar.h"
#include "ToolBarEx.h"
#include "WinAppEx.h"




LPCTSTR             CToolBarEx::m_lpszStateInfoEntry = _T("ToolbarStateInfo (v1.01)");
CToolBarEx*         CToolBarEx::m_pToolBar           = 0;
HHOOK               CToolBarEx::m_hCBTHook           = 0;

IMPLEMENT_DYNAMIC( CToolBarEx, CToolBar )

CToolBarEx::CToolBarEx()
{
}

CToolBarEx::~CToolBarEx()
{
}

bool CToolBarEx::Create( CWnd* pParentWnd, UINT nID)
{
  if ( !CreateEx( pParentWnd,
    TBSTYLE_FLAT | TBSTYLE_TOOLTIPS,
    WS_CHILD | WS_VISIBLE | CBRS_ALIGN_TOP | CBRS_FLYBY , CRect( 0, 0, 0, 0 ), nID ) )
  {
    return false;
  }
  
  GetToolBarCtrl().SetExtendedStyle(
    TBSTYLE_EX_DRAWDDARROWS | TBSTYLE_EX_HIDECLIPPEDBUTTONS );
  
  ModifyStyle( 0, CCS_ADJUSTABLE );
  
  return true;
}

/////////////////////////////////////////////////////////////////////////////
// Attributes

void CToolBarEx::SetText()
{
  
}



void CToolBarEx::SetIcons()
{
  ASSERT( ::IsWindow( m_hWnd ) );
  
  
  // Set image list(s) and icon size for selected icon options
  CToolBarCtrl& tbCtrl = GetToolBarCtrl();
  tbCtrl.SetImageList( 0 );
  tbCtrl.SetHotImageList( 0 );
  
  VERIFY( tbCtrl.SetBitmapSize( szImageSmall) );
  
  UINT nIDCold = m_nIDSmallCold ;
  UINT nIDHot  = m_nIDSmallHot;
  ASSERT( nIDCold != ( UINT )-1 );    // at least there must be "cold" imagelist
  
  m_imageListCold.DeleteImageList();
  VERIFY( m_imageListCold.Attach( ImageList_LoadImage( AfxGetResourceHandle(),
    MAKEINTRESOURCE( nIDCold ), szImageSmall.cx, 0, m_clrMask,
    IMAGE_BITMAP, LR_CREATEDIBSECTION ) ) );
  tbCtrl.SetImageList( &m_imageListCold );
  
  if ( nIDHot != ( UINT )-1 ) // "hot" imagelist is optional
  {
    m_imageListHot.DeleteImageList();
    VERIFY( m_imageListHot.Attach( ImageList_LoadImage( AfxGetResourceHandle(),
      MAKEINTRESOURCE( nIDHot ), szImageSmall.cx, 0, m_clrMask,
      IMAGE_BITMAP, LR_CREATEDIBSECTION ) ) );
    tbCtrl.SetHotImageList( &m_imageListHot );
  }
  
  // If requested, reflect changes immediately
  
  ReloadButtons();
  UpdateParentBandInfo();
  
}



/////////////////////////////////////////////////////////////////////////////
// Operations

void CToolBarEx::SetBitmaps( UINT nIDButtons, UINT nIDButtonsCold, COLORREF clrMask /*=RGB(255,0,255)*/ )
{
  m_nIDSmallCold = nIDButtonsCold;
  m_nIDSmallHot  = nIDButtons;
  
  m_clrMask      = clrMask;
  
  SetIcons();        // apply new options
}

void CToolBarEx::SetButtons( int nNumButtons, TBBUTTONEX* lpButtons)
{
  // Delete old buttons
  CToolBarCtrl& tbCtrl = GetToolBarCtrl();
  while ( tbCtrl.DeleteButton( 0 ) );
  
  // Load buttons from provided array
  m_aButtons.RemoveAll();
  for ( int nIndex = 0; nIndex < nNumButtons; nIndex++ )
  {
    if ( lpButtons[ nIndex ].bInitiallyVisible )
    {
      VERIFY( tbCtrl.AddButtons( 1, &lpButtons[ nIndex ].tbinfo ) );
    }
    
    m_aButtons.Add( lpButtons[ nIndex ] );
  }
  
  // save the original button layout
  m_aButtonsSaved.RemoveAll();
  for ( nIndex = 0; nIndex < nNumButtons; nIndex++ )
  {
    m_aButtonsSaved.Add( lpButtons[ nIndex ] );
  }

  ASSERT( ::IsWindow( m_hWnd ) );
  ASSERT( GetStyle() & TBSTYLE_TOOLTIPS );
  ASSERT( !( GetBarStyle() & CBRS_TOOLTIPS ) );
  
  ModifyStyle(0 , TBSTYLE_LIST);
  
  DWORD dwStyleEx = tbCtrl.GetExtendedStyle();
  tbCtrl.SetExtendedStyle(dwStyleEx |  TBSTYLE_EX_MIXEDBUTTONS );
  VERIFY( tbCtrl.SetMaxTextRows(1));
  
  // Modify all (even currently hidden ones) buttons in internal cache
  for ( nIndex = 0; nIndex <= m_aButtons.GetUpperBound(); nIndex++ )
  {
    TBBUTTON& tbinfo = m_aButtons[ nIndex ].tbinfo;
    if ( !( tbinfo.fsStyle & TBSTYLE_SEP ) )
    {
      CString strButtonText;
      GetButtonText( tbinfo.idCommand, strButtonText );
      CString strToAdd( strButtonText, strButtonText.GetLength() + 1 );
      tbinfo.iString = tbCtrl.AddStrings( strToAdd );
      
      tbinfo.fsStyle |= ( TBSTYLE_AUTOSIZE |
        ( HasButtonText( tbinfo.idCommand ) ? BTNS_SHOWTEXT : 0 ) );
    }
  }
  
  ReloadButtons();
  UpdateParentBandInfo();
}

void CToolBarEx::LoadState( LPCTSTR lpszProfileName )
{
  CString strSubKey;
  strSubKey.Format( _T("Software\\%s\\%s\\%s"),
    AfxGetApp()->m_pszRegistryKey,
    AfxGetApp()->m_pszProfileName,
    lpszProfileName );
  GetToolBarCtrl().RestoreState( HKEY_CURRENT_USER,
    strSubKey, m_lpszStateInfoEntry );
  
  // It was found out that TB_SAVERESTORE causes TBN_BEGINADJUST
  // and TBN_ENDADJUST to be sent correspondingly at the beggining
  // and the end of save/restore process.  So, the following
  // call is redundant and therefore was commented out.
  
  //    UpdateParentBandInfo();
}

void CToolBarEx::SaveState( LPCTSTR lpszProfileName )
{
  CString strSubKey;
  strSubKey.Format( _T("Software\\%s\\%s\\%s"),
    AfxGetApp()->m_pszRegistryKey,
    AfxGetApp()->m_pszProfileName,
    lpszProfileName );
  GetToolBarCtrl().SaveState( HKEY_CURRENT_USER,
    strSubKey, m_lpszStateInfoEntry );
}

/////////////////////////////////////////////////////////////////////////////
// Overrides

BOOL CToolBarEx::OnNotify( WPARAM wParam, LPARAM lParam, LRESULT* pResult )
{
  NMHDR* pNMHDR = ( NMHDR* )lParam;
  if ( ( pNMHDR->code == TTN_NEEDTEXTA ) ||
    ( pNMHDR->code == TTN_NEEDTEXTW ) )
  {
    // If button doesn't have a tip, ignore notification
    *pResult = HasButtonTip( pNMHDR->idFrom ) ? Default() : 0;
    return TRUE;
  }
  
  return CToolBar::OnNotify( wParam, lParam, pResult );
}

LRESULT CToolBarEx::DoCustomDraw( NMHDR* pNMHDR, CWnd* pToolBar )
{
  LPNMTBCUSTOMDRAW lpNMCustomDraw = ( LPNMTBCUSTOMDRAW )pNMHDR;
  
  switch ( lpNMCustomDraw->nmcd.dwDrawStage )
  {
  case CDDS_PREPAINT:
    return CDRF_NOTIFYITEMDRAW;
    
  case CDDS_ITEMPREPAINT:
    {
      UINT nID = (UINT)lpNMCustomDraw->nmcd.dwItemSpec;
      bool bHot = ( lpNMCustomDraw->nmcd.uItemState & CDIS_HOT ) != 0;
      if ( pToolBar->SendMessage( TB_ISBUTTONCHECKED, nID ) &&
        pToolBar->SendMessage( TB_ISBUTTONENABLED, nID ) && bHot )
      {
        // I personally don't like when background of checked item
        // which is currently hot is drawn using dither brush
        lpNMCustomDraw->hbrMonoDither = 0;
      }
      // fall through...
    }
  default:
    return CDRF_DODEFAULT;
  }
}

bool CToolBarEx::HasButtonText( UINT /*nID*/ )
{
  return true;
}

bool CToolBarEx::HasButtonTip( UINT nID )
{
  return !HasButtonText( nID );
}

void CToolBarEx::GetButtonText( UINT nID, CString& strText )
{
  CString strFull;
  VERIFY( strFull.LoadString( nID ) );
  VERIFY( AfxExtractSubString( strText, strFull, 1, _T('\n') ) );
}

void CToolBarEx::GetButtonTip( UINT nID, CString& strTip )
{
  GetButtonText( nID, strTip );
}

void CToolBarEx::Init()
{
  ASSERT( false );    // must be overridden
}

bool CToolBarEx::IsTextOptionAvailable( ) const
{
  return true;
}

bool CToolBarEx::IsIconOptionAvailable( ) const
{
  return true;
}

/////////////////////////////////////////////////////////////////////////////
// Implementation

CReBarCtrl& CToolBarEx::GetParentReBarCtrl() const
{
  return STATIC_DOWNCAST( CReBar, GetParent() )->GetReBarCtrl();
}

int CToolBarEx::GetParentBandIndex() const
{
  int nBand = GetParentReBarCtrl().IDToIndex( ( UINT )GetDlgCtrlID() );
  ASSERT( nBand != -1 );
  return nBand;
}

void CToolBarEx::ReloadButtons()
{
  // Reload buttons from internal cache
  CToolBarCtrl& tbCtrl = GetToolBarCtrl();
  for ( int nIndex = 0, nButtons = tbCtrl.GetButtonCount(); nIndex < nButtons; nIndex++ )
  {
    TBBUTTON tbinfo;
    VERIFY( tbCtrl.GetButton( 0, &tbinfo ) );
    VERIFY( GetButtonInfo( tbinfo.idCommand, tbinfo ) );
    VERIFY( tbCtrl.DeleteButton( 0 ) );
    VERIFY( tbCtrl.AddButtons( 1, &tbinfo ) );
  }
}

void CToolBarEx::UpdateParentBandInfo()
{
  CToolBarCtrl& tbCtrl = GetToolBarCtrl();
  tbCtrl.AutoSize();
  
  // Calculate desired height and ideal width of the bar
  CRect rcItem;
  int cyChild = 0;
  int cxIdeal = 0;
  for ( int nIndex = 0, nButtons = tbCtrl.GetButtonCount(); nIndex < nButtons; nIndex++ )
  {
    if ( tbCtrl.GetItemRect( nIndex, rcItem ) )
    {
      cxIdeal += rcItem.Width();
      cyChild = max( cyChild, rcItem.Height() );
    }
  }
  
  // Modify parent band info accordingly
  REBARBANDINFO rbbi;
  rbbi.cbSize     = sizeof( rbbi );
  rbbi.fMask      = RBBIM_CHILDSIZE | RBBIM_IDEALSIZE;
  rbbi.cxIdeal    = cxIdeal;
  rbbi.cxMinChild = 0;
  rbbi.cyMinChild = cyChild;
  VERIFY( GetParentReBarCtrl().SetBandInfo( GetParentBandIndex(), &rbbi ) );
  
  Invalidate();   // visual feedback
}

bool CToolBarEx::GetButtonInfo( UINT nID, TBBUTTON& tbinfo )
{
  if ( tbinfo.fsStyle & TBSTYLE_SEP )
  {
    return true;
  }
  
  for ( int nIndex = 0; nIndex <= m_aButtons.GetUpperBound(); nIndex++ )
  {
    if ( ( UINT )m_aButtons[ nIndex ].tbinfo.idCommand == nID )
    {
      tbinfo = m_aButtons[ nIndex ].tbinfo;
      return true;
    }
  }
  
  return false;
}

/////////////////////////////////////////////////////////////////////////
// CToolBarEx message handlers

BEGIN_MESSAGE_MAP(CToolBarEx, CToolBar)
//{{AFX_MSG_MAP(CToolBarEx)
//}}AFX_MSG_MAP

ON_MESSAGE( WM_REBAR_CHEVRONPUSHED, OnReBarChevronPushed )

ON_NOTIFY_REFLECT( NM_CUSTOMDRAW, OnCustomDraw )
ON_NOTIFY_REFLECT( TBN_GETINFOTIP, OnGetInfoTip )

// Toolbar customization
ON_NOTIFY_REFLECT( TBN_BEGINADJUST, OnBeginAdjust )
ON_NOTIFY_REFLECT( TBN_ENDADJUST, OnEndAdjust )
ON_NOTIFY_REFLECT( TBN_QUERYINSERT, OnQueryInsert )
ON_NOTIFY_REFLECT( TBN_QUERYDELETE, OnQueryDelete )
ON_NOTIFY_REFLECT( TBN_GETBUTTONINFO, OnGetButtonInfo )
ON_NOTIFY_REFLECT( TBN_TOOLBARCHANGE, OnToolBarChange )

// Saving and restoring toolbar
ON_NOTIFY_REFLECT( TBN_SAVE, OnSave )
ON_NOTIFY_REFLECT( TBN_RESTORE, OnRestore )
END_MESSAGE_MAP()



LRESULT CToolBarEx::OnReBarChevronPushed( WPARAM wParam, LPARAM /*lParam*/ )
{
  CRect rcChevron( ( LPCRECT )wParam );
  
  CToolBarPopup menu( this );
  menu.ShowPopup( TPM_LEFTALIGN | TPM_VERTICAL |
    ( CWinAppEx::GetInstance()->IsWin98_2K() ? TPM_VERPOSANIMATION : 0 ),
    CPoint( rcChevron.left, rcChevron.bottom ), rcChevron );
  
  return 0L;
}

void CToolBarEx::OnCustomDraw( NMHDR* pNMHDR, LRESULT* pResult )
{
  *pResult = DoCustomDraw( pNMHDR, this );
}

void CToolBarEx::OnGetInfoTip( NMHDR* pNMHDR, LRESULT* pResult )
{
  NMTBGETINFOTIP* lptbgit = ( NMTBGETINFOTIP* )pNMHDR;
  CString strTip;
  GetButtonTip( lptbgit->iItem, strTip );
  _tcsncpy( lptbgit->pszText, strTip, lptbgit->cchTextMax );
  
  *pResult = 0;
}

LRESULT CALLBACK CToolBarEx::CBTProc( int nCode, WPARAM wParam, LPARAM lParam )
{
  ASSERT( m_pToolBar != 0 );
  return ::CallNextHookEx( m_hCBTHook, nCode, wParam, lParam );
}

void CToolBarEx::OnBeginAdjust( NMHDR* /*pNMHDR*/, LRESULT* pResult )
{
  m_pToolBar = this;
  m_hCBTHook = ::SetWindowsHookEx( WH_CBT, CBTProc, 0, ::GetCurrentThreadId() );
  ASSERT( m_hCBTHook != 0 );
  
  *pResult = 0;
}

void CToolBarEx::OnEndAdjust( NMHDR* /*pNMHDR*/, LRESULT* pResult )
{
  VERIFY( ::UnhookWindowsHookEx( m_hCBTHook ) );
  m_hCBTHook = 0;
  m_pToolBar = 0;
  
  UpdateParentBandInfo();
  
  *pResult = 0;
}

void CToolBarEx::OnQueryInsert( NMHDR* /*pNMHDR*/, LRESULT* pResult )
{
  *pResult = TRUE;    // otherwise Customize dialog will not be shown
}

void CToolBarEx::OnQueryDelete( NMHDR* /*pNMHDR*/, LRESULT* pResult )
{
  *pResult = TRUE;    // why not? :)
}


void CToolBarEx::OnGetButtonInfo( NMHDR* pNMHDR, LRESULT* pResult )
{
  NMTOOLBAR* lpnmtb = ( NMTOOLBAR* )pNMHDR;
  
  int nItem = lpnmtb->iItem;
  if ( ( 0 <= nItem ) && ( nItem <= m_aButtons.GetUpperBound() ) )
  {
    // Copy button info from internal cache
    lpnmtb->tbButton = m_aButtons[ nItem ].tbinfo;
    if ( !( lpnmtb->tbButton.fsStyle & TBSTYLE_SEP ) )
    {
      CString strText;
      GetButtonText( lpnmtb->tbButton.idCommand, strText );
      _tcsncpy( lpnmtb->pszText, strText, lpnmtb->cchText );
    }
    
    *pResult = TRUE;
  }
  else
  {
    *pResult = FALSE;
  }
}

void CToolBarEx::OnToolBarChange( NMHDR* /*pNMHDR*/, LRESULT* pResult )
{
  UpdateParentBandInfo();
  
  *pResult = 0;
}

void CToolBarEx::OnSave( NMHDR* pNMHDR, LRESULT* pResult )
{
  NMTBSAVE* lpnmtb = ( NMTBSAVE* )pNMHDR;
  if ( lpnmtb->iItem == -1 )
  {
    lpnmtb->cbData  += sizeof( DWORD ) * 2;
    lpnmtb->pData    = ( LPDWORD )::GlobalAlloc( GMEM_FIXED, lpnmtb->cbData );
    lpnmtb->pCurrent = lpnmtb->pData;
    
  }
  
  *pResult = 0;
}

void CToolBarEx::OnRestore( NMHDR* pNMHDR, LRESULT* pResult )
{
  NMTBRESTORE* lpnmtb = ( NMTBRESTORE* )pNMHDR;
  if ( lpnmtb->iItem == -1 )
  {
    lpnmtb->cButtons = ( lpnmtb->cbData - sizeof( DWORD ) * 2 ) / lpnmtb->cbBytesPerRecord;
    lpnmtb->pCurrent = lpnmtb->pData;
    
  }
  else
  {
    VERIFY( GetButtonInfo( lpnmtb->tbButton.idCommand, lpnmtb->tbButton ) );
  }
  
  *pResult = 0;
}


bool CToolBarEx::SetBarText( LPCTSTR text )
{
 	REBARBANDINFO rbBand;
  rbBand.cbSize     = sizeof( rbBand );
  rbBand.fMask      = RBBIM_TEXT;
  
  rbBand.lpText = const_cast<LPTSTR>(text);
  
  if (!GetParentReBarCtrl().SetBandInfo(GetParentBandIndex(),&rbBand))
    return false;
  
  return true;
  
}

void CToolBarEx::AddString(CStatic* pStatic, UINT nID, CString text, int width)
{
  int index = 0;
  while(GetItemID(index)!=nID) index++;

  CRect rect;
  SetButtonInfo(index, nID, TBBS_SEPARATOR, width);

  GetItemRect(index, &rect);

  rect.top += 4;

  pStatic->Create(text, SS_RIGHT, rect, this, nID);
  pStatic->SetFont(this->GetFont(),true);  
  pStatic->ShowWindow(SW_SHOW);
}


void CToolBarEx::AddSlider(CSliderCtrl* pSlider, UINT nID, int width)
{
  int index = 0;
  while(GetItemID(index)!=nID) index++;
  
  CRect rect;
  SetButtonInfo(index, nID, TBBS_SEPARATOR , width);
  
  GetItemRect(index, &rect);
  pSlider->Create(TBS_HORZ | TBS_BOTTOM,
    rect, this, nID);

  pSlider->ShowWindow(SW_SHOW);
}


void CToolBarEx::AddCombo(CComboBox* pComboBox, UINT nID, int width)
{
  int index = 0;
  while(GetItemID(index)!=nID) index++;
  
  CRect rect;
  SetButtonInfo(index, nID, TBBS_SEPARATOR , width);
  
  GetItemRect(index, &rect);
  
  rect.top +=1;
  rect.bottom += 500;
  
  pComboBox->Create(WS_BORDER | WS_CHILD|WS_VISIBLE | CBS_AUTOHSCROLL | WS_VSCROLL | CBS_DROPDOWNLIST | CBS_NOINTEGRALHEIGHT 
    , rect, this, nID);
 
  pComboBox->SetFont(this->GetFont(),true);
  
  pComboBox->ShowWindow(SW_SHOW);
}


void CToolBarEx::AddEdit(CEdit* pEdit, UINT nID, int width, CComboBox* pDummyCombo)
{
  int index = 0;
  while(GetItemID(index)!=nID) index++;
  
  int textWidth = 10;
  SetButtonInfo(index, nID, TBBS_SEPARATOR | TBSTATE_HIDDEN , textWidth + width + 1);
  
  CRect rectText;
  CRect rectFrame;
  GetItemRect(index, &rectText);
  GetItemRect(index, &rectFrame);
  


  rectFrame.top +=1;
  rectFrame.left +=4;
  rectFrame.bottom -=2;
  rectFrame.right -=2;

  pDummyCombo->Create(WS_CHILD|WS_VISIBLE | CBS_AUTOHSCROLL | CBS_SIMPLE | CBS_NOINTEGRALHEIGHT 
    , rectFrame, this, 0);
  pDummyCombo->SetFont(this->GetFont());
  pDummyCombo->EnableWindow(false);
  pDummyCombo->ShowWindow(SW_SHOW);

  rectFrame.top +=2;
  rectFrame.left +=2;
  rectFrame.bottom -=0;
  rectFrame.right -=2;
  pEdit->Create(WS_CHILDWINDOW | WS_VISIBLE | CBS_AUTOHSCROLL | ES_LEFT | ES_NOHIDESEL, rectFrame, this, nID);
  pEdit->ModifyStyleEx(0,WS_EX_NOPARENTNOTIFY | WS_EX_CLIENTEDGE | WS_EX_LEFT);

  pEdit->SetMargins(2,2);
  pEdit->SetFont(this->GetFont(),true);
  
  pEdit->ShowWindow(SW_SHOW);
}



HHOOK           CToolBarPopup::m_hKeyboardHook  = 0;
CToolBarPopup*  CToolBarPopup::m_pPopup         = 0;

IMPLEMENT_DYNAMIC( CToolBarPopup, CWnd )

CToolBarPopup::CToolBarPopup( CToolBarEx* pToolBar )
{
  ASSERT_VALID( pToolBar );
  
  m_pToolBar    = pToolBar;
  m_bOverTbCtrl = false;
  m_bTextLabels = false;
}

CToolBarPopup::~CToolBarPopup()
{
}

/////////////////////////////////////////////////////////////////////////////
// Operations

bool CToolBarPopup::ShowPopup( UINT nFlags, CPoint pt, CRect& rcExclude )
{
  CString strWndClass = AfxRegisterWndClass( CS_SAVEBITS,
    ::LoadCursor( 0, IDC_ARROW ), ::GetSysColorBrush( COLOR_MENU ), 0 );
  
  if ( !CreateEx( WS_EX_PALETTEWINDOW, strWndClass, 0,
    WS_POPUP | WS_CLIPSIBLINGS | WS_CLIPCHILDREN | WS_DLGFRAME,
    CRect( 0, 0, 0, 0 ), 0, 0 ) )
  {
    return false;
  }
  
  SetPosition( nFlags, pt, rcExclude );
  SetCapture();
  SendMessage( WM_SETCURSOR, ( WPARAM )m_hWnd, MAKELPARAM( HTCLIENT, 0 ) );
  
  m_pPopup = this;
  m_hKeyboardHook = ::SetWindowsHookEx( WH_KEYBOARD, KeyboardProc, 0, ::GetCurrentThreadId() );
  ASSERT( m_hKeyboardHook != 0 );
  
  // Emulate menu loop
  RunModalLoop( MLF_NOKICKIDLE );
  
  VERIFY( ::UnhookWindowsHookEx( m_hKeyboardHook ) );
  m_hKeyboardHook = 0;
  m_pPopup = 0;
  
  DestroyWindow();
  
  return true;
}

/////////////////////////////////////////////////////////////////////////////
// Implementation

void CToolBarPopup::SetPosition( UINT nFlags, CPoint pt, CRect& rcExclude )
{
  // The main purpose of this functions is to find proper position of
  // the popup window, so that we neither exceed screen dimensions nor
  // intersect rcExclude rectangle.  The function also takes into account
  // control flags specified by the user.
  
  // Calc initial position
  CRect rc;
  m_tbCtrl.GetWindowRect( rc );
  
  CalcWindowRect( rc, CWnd::adjustBorder );
  rc.OffsetRect( -rc.TopLeft() );
  rc.OffsetRect(
    ( nFlags & TPM_CENTERALIGN ) ? ( pt.x - rc.Width() / 2 ) :
  ( nFlags & TPM_RIGHTALIGN  ) ? ( pt.x - rc.Width() ) : pt.x, 0 );
  rc.OffsetRect( 0,
    ( nFlags & TPM_VCENTERALIGN ) ? ( pt.y - rc.Height() / 2 ) :
  ( nFlags & TPM_BOTTOMALIGN  ) ? ( pt.y - rc.Height() ) : pt.y );
  
  // Make sure we don't exceed screen dimensions
  CRect rcDesktop;
  GetDesktopWindow()->GetWindowRect( rcDesktop );
  
  rc.OffsetRect(
    min( rcDesktop.right  - rc.right,  0 ),
    min( rcDesktop.bottom - rc.bottom, 0 ) );
  rc.OffsetRect(
    max( rcDesktop.left   - rc.left,   0 ),
    max( rcDesktop.top    - rc.top,    0 ) );
  
  // Make sure we don't intersect rcExclude rectangle
  CRect rcTemp;
  if ( !rcExclude.IsRectEmpty() && rcTemp.IntersectRect( rc, rcExclude ) )
  {
    if ( nFlags & TPM_VERTICAL )
    {
      CRect rcUp( rc );
      int nUp = rc.bottom - rcExclude.top;
      rcUp.OffsetRect( 0, -nUp );
      
      CRect rcDown( rc );
      int nDown = rcExclude.bottom - rc.top;
      rcDown.OffsetRect( 0, nDown );
      
      bool bUp = false;
      if ( ( rcUp.top >= rcDesktop.top ) && ( rcDown.bottom <= rcDesktop.bottom ) )
      {
        bUp = ( nUp < nDown );
      }
      else if ( rcUp.top >= rcDesktop.top )
      {
        bUp = true;
      }
      else if ( rcDown.bottom <= rcDesktop.bottom )
      {
        bUp = false;
      }
      else
      {
        ASSERT( false );    // this case is not supported yet
      }
      
      rc = bUp ? rcUp : rcDown;
      nFlags &= ~( TPM_HORPOSANIMATION | TPM_HORNEGANIMATION | TPM_VERPOSANIMATION | TPM_VERNEGANIMATION );
      nFlags |=  ( bUp ? TPM_VERNEGANIMATION : TPM_VERPOSANIMATION );
    }
    else
    {
      CRect rcLeft( rc );
      int nLeft = rc.right - rcExclude.left;
      rcLeft.OffsetRect( -nLeft, 0 );
      
      CRect rcRight( rc );
      int nRight = rcExclude.right - rc.left;
      rcRight.OffsetRect( nRight, 0 );
      
      bool bLeft = false;
      if ( ( rcLeft.left >= rcDesktop.top ) && ( rcRight.right <= rcDesktop.right ) )
      {
        bLeft = ( nLeft < nRight );
      }
      else if ( rcLeft.left >= rcDesktop.left )
      {
        bLeft = true;
      }
      else if ( rcRight.right <= rcDesktop.right )
      {
        bLeft = false;
      }
      else
      {
        ASSERT( false );    // this case is not supported yet
      }
      
      rc = bLeft ? rcLeft : rcRight;
      nFlags &= ~( TPM_HORPOSANIMATION | TPM_HORNEGANIMATION | TPM_VERPOSANIMATION | TPM_VERNEGANIMATION );
      nFlags |=  ( bLeft ? TPM_HORNEGANIMATION : TPM_HORPOSANIMATION );
    }
  }
  
  Show( nFlags, rc );
}

void CToolBarPopup::Show( UINT nFlags, const CRect& rc )
{
  // On W98/W2K systems the menu animation feature is supported.
  // This function tries to mimic this feature conformably to
  // toolbar popup window.
  
  // Position window on the screen
  SetWindowPos( &wndTopMost, rc.left, rc.top, rc.Width(), rc.Height(),
    SWP_NOACTIVATE );
  
  CWinAppEx* pApp = CWinAppEx::GetInstance();
  if ( pApp->IsWin98_2K() && !( nFlags & TPM_NOANIMATION ) ) // W98/W2K specific (menu animation)
  {
    if ( pApp->GetMenuAnimation() )
    {
      DWORD dwAnimationFlags = AW_SLIDE |
        ( ( nFlags & TPM_HORPOSANIMATION ) ? AW_HOR_POSITIVE : 0 ) |
        ( ( nFlags & TPM_HORNEGANIMATION ) ? AW_HOR_NEGATIVE : 0 ) |
        ( ( nFlags & TPM_VERPOSANIMATION ) ? AW_VER_POSITIVE : 0 ) |
        ( ( nFlags & TPM_VERNEGANIMATION ) ? AW_VER_NEGATIVE : 0 );
      if ( dwAnimationFlags == AW_SLIDE )
      {
        // If none of TPM_*ANIMATION flags is set, set default animation
        dwAnimationFlags |= AW_HOR_POSITIVE | AW_VER_POSITIVE;
      }
      
      if ( pApp->IsWin2K() && pApp->GetMenuFade() )   // W2K specific (fade effect)
      {
        dwAnimationFlags = AW_BLEND;
      }
      
      VERIFY( ::AnimateWindowWin50( m_hWnd, 200, dwAnimationFlags ) );
      return;
    }
  }
  
  // The animation feature is whether turned off or unsupported
  // on this system.  Therefore, just show window without activation.
  ShowWindow( SW_SHOWNOACTIVATE );
}

void CToolBarPopup::OnKeyDown( UINT nChar )
{
  switch ( nChar )
  {
  case VK_ESCAPE: // dismiss menu
  case VK_MENU:
    SendMessage( WM_TB_ENDMODALLOOP );
    break;
    
  case VK_UP:     // select next/prev button
  case VK_DOWN:
    {
      int nLastItem = m_tbCtrl.GetButtonCount() - 1;
      int nHotItem  = m_tbCtrl.GetHotItem();
      
      for ( int nIndex = 0; nIndex <= nLastItem; nIndex++ )
      {
        if ( nHotItem >= 0 )
        {
          nHotItem = ( nChar == VK_UP ) ?
            ( nHotItem ? nHotItem - 1 : nLastItem ) :
          ( nHotItem == nLastItem ? 0 : nHotItem + 1 );
        }
        else
        {
          nHotItem = ( nChar == VK_UP ) ?
nLastItem : 0;
        }
        
        m_tbCtrl.SetHotItem( nHotItem );
        if ( m_tbCtrl.GetHotItem() == nHotItem )
        {
          break;
        }
      }
      break;
    }
  case VK_RETURN: // issue command associated with selected button
    {
      int nHotItem = m_tbCtrl.GetHotItem();
      if ( nHotItem >= 0 )
      {
        TBBUTTON tbinfo;
        VERIFY( m_tbCtrl.GetButton( nHotItem, &tbinfo ) );
        if ( tbinfo.idCommand != 0 )
        {
          SendMessage( WM_COMMAND, tbinfo.idCommand );
          break;
        }
      }
      
      SendMessage( WM_TB_ENDMODALLOOP );
      break;
    }
  default:
    break;
  }
}

LRESULT CALLBACK CToolBarPopup::KeyboardProc( int code, WPARAM wParam, LPARAM lParam )
{
  ASSERT( m_pPopup != 0 );
  
  if ( code == HC_ACTION )
  {
    CWnd* pCapture = GetCapture();
    if ( ( pCapture == m_pPopup ) ||
      ( pCapture == &m_pPopup->m_tbCtrl ) )
    {
      if ( !( HIWORD( lParam ) & KF_UP ) )
      {
        m_pPopup->OnKeyDown( wParam );
      }
      
      return 1;
    }
  }
  
  return ::CallNextHookEx( m_hKeyboardHook, code, wParam, lParam );
}

/////////////////////////////////////////////////////////////////////////////
// Overrides

BOOL CToolBarPopup::OnCommand( WPARAM wParam, LPARAM /*lParam*/ )
{
  // Dismiss menu
  SendMessage( WM_TB_ENDMODALLOOP );
  
  // Forward command to the original toolbar window
  m_pToolBar->PostMessage( WM_COMMAND, LOWORD( wParam ), 0 );
  
  return TRUE;    // command was processed
}

BOOL CToolBarPopup::OnNotify( WPARAM wParam, LPARAM lParam, LRESULT* pResult )
{
  NMHDR* pNMHDR = ( NMHDR* )lParam;
  if ( pNMHDR->hwndFrom == m_tbCtrl.m_hWnd )
  {
    // Handle certain notifications from embedded toolbar control
    switch ( pNMHDR->code )
    {
    case NM_RELEASEDCAPTURE:
      SetCapture();
      *pResult = 0;
      break;
      
    case NM_CUSTOMDRAW:
      *pResult = m_pToolBar->DoCustomDraw( pNMHDR, &m_tbCtrl );
      break;
      
    case TBN_GETINFOTIP:
      if ( !m_bTextLabels )
      {
        NMTBGETINFOTIP* lptbgit = ( NMTBGETINFOTIP* )pNMHDR;
        CString strTip;
        m_pToolBar->GetButtonTip( lptbgit->iItem, strTip );
        _tcsncpy( lptbgit->pszText, strTip, lptbgit->cchTextMax );
      }
      *pResult = 0;
      break;
      
    default:    // forward message to the parent of the original toolbar
      *pResult = m_pToolBar->GetParent()->SendMessage( WM_NOTIFY, wParam, lParam );
      break;
    }
    
    return TRUE;
  }
  
  return CWnd::OnNotify( wParam, lParam, pResult );
}

LRESULT CToolBarPopup::WindowProc( UINT message, WPARAM wParam, LPARAM lParam )
{
  if ( WM_MOUSEFIRST <= message && message <= WM_MOUSELAST )
  {
    DWORD dwPos = ::GetMessagePos();
    CPoint ptScreen( LOWORD( dwPos ), HIWORD( dwPos ) );
    CWnd* pWnd = WindowFromPoint( ptScreen );
    if ( pWnd != 0 )
    {
      CPoint ptClient( ptScreen );
      pWnd->ScreenToClient( &ptClient );
      
      switch ( message )
      {
      case WM_MOUSEMOVE:
        {
          // Check if hot item should be changed
          bool bOverTbCtrl = ( pWnd == &m_tbCtrl );
          if ( bOverTbCtrl )
          {
            int nHit = m_tbCtrl.HitTest( &ptClient );
            m_tbCtrl.SetHotItem( nHit );
            
            // Let tooltip control process mouse event
            CToolTipCtrl* pTtCtrl = m_tbCtrl.GetToolTips();
            if ( pTtCtrl != 0 )
            {
              MSG msg;
              msg.hwnd    = m_tbCtrl.m_hWnd;
              msg.message = WM_MOUSEMOVE;
              msg.wParam  = wParam;
              msg.lParam  = MAKELPARAM( ptClient.x, ptClient.y );
              msg.pt      = ptScreen;
              msg.time    = ::GetMessageTime();
              pTtCtrl->RelayEvent( &msg );
            }
          }
          else if ( m_bOverTbCtrl )
          {
            m_tbCtrl.SetHotItem( -1 );
          }
          
          m_bOverTbCtrl = bOverTbCtrl;
          return 0L;
        }
      case WM_LBUTTONDOWN:
        if ( pWnd != this )
        {
          // Dismiss menu if user has clicked outside the window
          if ( pWnd != &m_tbCtrl )
          {
            SendMessage( WM_TB_ENDMODALLOOP );
          }
          
          // Forward this mouse event to the window that was clicked
          LPARAM nPosition = MAKELPARAM( ptScreen.x, ptScreen.y );
          WPARAM nHitTest  = pWnd->SendMessage( WM_NCHITTEST, 0, nPosition );
          if ( nHitTest == HTCLIENT )
          {
            nPosition = MAKELPARAM( ptClient.x, ptClient.y );
            pWnd->PostMessage( WM_LBUTTONDOWN, wParam, nPosition );
          }
          else
          {
            pWnd->PostMessage( WM_NCLBUTTONDOWN, nHitTest, nPosition );
          }
        }
        return 0L;
        
      default:
        break;
      }
    }
  }
  
  return CWnd::WindowProc( message, wParam, lParam );
}


/////////////////////////////////////////////////////////////////////////
// CToolBarPopup message handlers

BEGIN_MESSAGE_MAP(CToolBarPopup, CWnd)
//{{AFX_MSG_MAP(CToolBarPopup)
ON_WM_CAPTURECHANGED()
ON_WM_CREATE()
ON_WM_MOUSEACTIVATE()
//}}AFX_MSG_MAP
ON_MESSAGE_VOID( WM_TB_ENDMODALLOOP, OnEndModalLoop )
END_MESSAGE_MAP()

void CToolBarPopup::OnEndModalLoop()
{
  EndModalLoop( 0 );
}

void CToolBarPopup::OnCaptureChanged( CWnd* pWnd )
{
  if ( ( pWnd != this ) && ( pWnd != &m_tbCtrl ) && ContinueModal() )
  {
    PostMessage( WM_TB_ENDMODALLOOP );  // dismiss menu
  }
  
  CWnd::OnCaptureChanged( pWnd );
}

int CToolBarPopup::OnCreate( LPCREATESTRUCT lpCreateStruct )
{
  // M.L. changed completely 
  if ( CWnd::OnCreate( lpCreateStruct ) == -1 )
  {
    return -1;
  }
  
  // Create embedded toolbar control
  if ( !m_tbCtrl.Create(
    TBSTYLE_FLAT | TBSTYLE_WRAPABLE | TBSTYLE_LIST | TBSTYLE_TOOLTIPS |
    CCS_NODIVIDER | CCS_NOPARENTALIGN | CCS_NORESIZE |
    WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN,
    CRect( 0, 0, 0, 0 ), this, m_pToolBar->GetDlgCtrlID() ) )
  {
    return -1;
  }
  
  m_tbCtrl.SetExtendedStyle( TBSTYLE_EX_DRAWDDARROWS | TBSTYLE_EX_MIXEDBUTTONS );
  VERIFY( m_tbCtrl.GetToolTips()->ModifyStyle( 0, TTS_ALWAYSTIP ) );
  
  m_bTextLabels = true;
  
  
  // Copy all required information from the original toolbar
  CToolBarCtrl& tbCtrl = m_pToolBar->GetToolBarCtrl();
  m_tbCtrl.SetImageList( tbCtrl.GetImageList() );
  m_tbCtrl.SetHotImageList( tbCtrl.GetHotImageList() );
  m_tbCtrl.SetDisabledImageList( tbCtrl.GetDisabledImageList() );
  
  CRect rcItem, rcClient;
  tbCtrl.GetClientRect( rcClient );
  
  TBBUTTON tbinfo;
  int nMaxWidth = 0;
  int nButtons = tbCtrl.GetButtonCount();
  int nIndex;
  for ( nIndex = 0; nIndex < nButtons; nIndex++ )
  {
    tbCtrl.GetItemRect( nIndex, rcItem );
    
    if ( rcItem.right > rcClient.right )
    {
      tbCtrl.GetButton( nIndex, &tbinfo );

      CString strButtonText;
      if (!( tbinfo.fsStyle & TBSTYLE_SEP ))
      {
        m_pToolBar->GetButtonText( tbinfo.idCommand, strButtonText );
        m_tbCtrl.CheckButton(tbinfo.idCommand, m_pToolBar->GetToolBarCtrl().IsButtonChecked(tbinfo.idCommand));
        CString strToAdd( strButtonText, strButtonText.GetLength() + 1 );
        tbinfo.iString  = m_tbCtrl.AddStrings( strToAdd );
        tbinfo.fsStyle |= TBSTYLE_AUTOSIZE | BTNS_SHOWTEXT;
        VERIFY( m_tbCtrl.AddButtons( 1, &tbinfo ) );
        VERIFY( m_tbCtrl.GetItemRect( m_tbCtrl.CommandToIndex( tbinfo.idCommand ), rcItem ) );
        nMaxWidth = max( nMaxWidth, rcItem.Width() );
      }
      else
      {
        // check if there is a control for that id
        HWND hwnd;
        m_pToolBar->GetDlgItem(tbinfo.idCommand,&hwnd);
        if (hwnd != 0)
        {
          // there is a control, lets check the class name
          CString className = m_pToolBar->GetDlgItem(tbinfo.idCommand)
            ->GetRuntimeClass()->m_lpszClassName;
          if (className == "CComboBox")
          {
            CComboBox* pCombo = (CComboBox*)m_pToolBar->GetDlgItem(tbinfo.idCommand);

            if ((pCombo->GetStyle() & CBS_DROPDOWNLIST) == CBS_DROPDOWNLIST)
            {
              // copy the buttons from the saved array (contains buttons for other controls
              memcpy(&tbinfo, &m_pToolBar->m_aButtonsSaved.ElementAt(nIndex).tbinfo,sizeof(tbinfo));
              
              m_pToolBar->GetButtonText( tbinfo.idCommand, strButtonText );
              
              CString temp;
              if (pCombo->GetCount() > 0)
              {
                pCombo->GetLBText(pCombo->GetCurSel(),temp);
              }
              else
              {
                temp = "empty";
              }
              strButtonText += ": " +temp;
            }
            else 
            {
            };
          }
          else if(className=="CEdit")
          {
            CEdit* pEdit = (CEdit*)m_pToolBar->GetDlgItem(tbinfo.idCommand);

            // copy the buttons from the saved array (contains buttons for other controls
            memcpy(&tbinfo, &m_pToolBar->m_aButtonsSaved.ElementAt(nIndex).tbinfo,sizeof(tbinfo));
            
            m_pToolBar->GetButtonText( tbinfo.idCommand, strButtonText );
            
            CString temp;
            pEdit->GetWindowText(temp);
            strButtonText += ": " +temp;
          }
          else if (className=="CSliderCtrl")
          {
            CSliderCtrl* pSlider = (CSliderCtrl*)m_pToolBar->GetDlgItem(tbinfo.idCommand);
            // copy the buttons from the saved array (contains buttons for other controls
            memcpy(&tbinfo, &m_pToolBar->m_aButtonsSaved.ElementAt(nIndex).tbinfo,sizeof(tbinfo));
            
            m_pToolBar->GetButtonText( tbinfo.idCommand, strButtonText );
            
            CString temp;
            temp.Format("%i",pSlider->GetPos());

            strButtonText += ": " + temp;
          }
          else if (className=="CStatic")
          {
            CStatic* pStatic = (CStatic*)m_pToolBar->GetDlgItem(tbinfo.idCommand);
            // copy the buttons from the saved array (contains buttons for other controls
            memcpy(&tbinfo, &m_pToolBar->m_aButtonsSaved.ElementAt(nIndex).tbinfo,sizeof(tbinfo));

            m_pToolBar->GetButtonText( tbinfo.idCommand, strButtonText );
            CString temp;
            pStatic->GetWindowText(temp);
            strButtonText += temp;

          }
          else
          {
            AfxMessageBox(className);
          }
          CString strToAdd( strButtonText, strButtonText.GetLength() + 1 );
          tbinfo.iString  = m_tbCtrl.AddStrings( strToAdd );
          tbinfo.fsStyle |= TBSTYLE_AUTOSIZE | BTNS_SHOWTEXT;
          VERIFY( m_tbCtrl.AddButtons( 1, &tbinfo ) );
          VERIFY( m_tbCtrl.GetItemRect( m_tbCtrl.CommandToIndex( tbinfo.idCommand ), rcItem ) );
          nMaxWidth = max( nMaxWidth, rcItem.Width() );
        }
      }
    }
  }
  
  nButtons = m_tbCtrl.GetButtonCount();
  if ( nButtons == 0 )
  {
    ASSERT( false );    // this should never happen
    return -1;
  }
  
  if ( m_bTextLabels )
  {
    TBBUTTONINFO tbbi;
    tbbi.cbSize = sizeof( tbbi );
    tbbi.dwMask = TBIF_SIZE | TBIF_STYLE;
    for ( nIndex = 0; nIndex < nButtons; nIndex++ )
    {
      VERIFY( m_tbCtrl.GetButton( nIndex, &tbinfo ) );
      tbbi.cx      = ( WORD )nMaxWidth;
      tbbi.fsStyle = ( BYTE )( tbinfo.fsStyle & ~TBSTYLE_AUTOSIZE );
      VERIFY( m_tbCtrl.SetButtonInfo( tbinfo.idCommand, &tbbi ) );
    }
  }
  
  m_tbCtrl.AutoSize();
  
  // Calc toolbar size
  if ( nButtons > 1 )
  {
    m_tbCtrl.SetRows( nButtons, m_bTextLabels, rcClient );
  }
  else
  {
    VERIFY( m_tbCtrl.GetItemRect( 0, rcClient ) );
  }
  
  m_tbCtrl.MoveWindow( rcClient );
  
  return 0;
}

int CToolBarPopup::OnMouseActivate( CWnd* /*pDesktopWnd*/, UINT /*nHitTest*/, UINT /*message*/ )
{
  return MA_NOACTIVATEANDEAT; // just in case
}

void CToolBarEx::SetImage(UINT nID,int iImage)
{
  TBBUTTONINFO tbbi;
  tbbi.cbSize = sizeof(tbbi);
  tbbi.dwMask = TBIF_COMMAND | TBIF_IMAGE | TBIF_SIZE ;

  CToolBarCtrl& tbCtrl = GetToolBarCtrl();
  tbCtrl.GetButtonInfo(nID,&tbbi);
  tbbi.iImage = iImage;
  tbCtrl.SetButtonInfo(nID,&tbbi);
}

/*
 * Change log :
 * 
 * $Log: ToolBarEx.cpp,v $
 * Revision 1.4  2004/02/26 16:30:56  loetzsch
 * improved support for CStatics on tool bars
 *
 * Revision 1.3  2004/02/23 13:27:25  jhoffman
 * - changes to Main-menu are shown ... this currently results in one additional menu (!) somebody please fix this!
 * - added GenericModalBinary-dialog: can be used to create a modal dialog where you can specify the button labels and the descriptive text (like APX...ModalDialog but you can specify what is written on the buttons)
 * - CameraToolBar: mode, shutter, gain extracted from dropdown
 * - ToolBarEx: added AddString method to write const. strings on a toolbar
 *
 * Revision 1.2  2003/11/30 01:53:21  loetzsch
 * prepared RobotControl port to Visual C++ .Net
 *
 * Revision 1.1  2003/10/07 10:10:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.5  2003/05/12 14:22:45  risler
 * added scrollbar to toolbar combo boxes
 *
 * Revision 1.4  2003/05/11 23:29:26  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.3  2003/03/20 20:35:15  loetzsch
 * function setBitmap added
 *
 * Revision 1.2  2003/03/19 23:47:16  loetzsch
 * Added support for sliders on toolbars
 *
 * Revision 1.1  2002/09/10 15:49:10  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.1  2002/09/01 17:19:20  loetzsch
 * tidied up the MfcTools/ directory. Created directories
 * MfcTools/DockingControlBars and MfcTools/IEStyleToolbars.
 *
 * Revision 1.1.1.1  2002/05/10 12:40:26  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.5  2002/01/30 17:32:46  loetzsch
 * kommt nun auch mit leeren Comboboxen klar
 *
 * Revision 1.4  2001/12/10 17:47:10  risler
 * change log added
 *
 */
