/**
* @file MenuBar.cpp
* Implementation of the CMenuBar class.
*
* Copyright (c) 2001 by Nikolay Denisov. All rights reserved.
*
* This code is free for personal and commercial use, providing this 
* notice remains intact in the source files and all eventual changes are
* clearly marked with comments.
*
* You must obtain the author's consent before you can include this code
* in a software library.
*
* No warrantee of any kind, express or implied, is included with this
* software; use at your own risk, responsibility for damages (if any) to
* anyone resulting from the use of this software rests entirely with the
* user.
*
* Please email bug reports, bug fixes, enhancements, requests and
* comments to: nick@actor.ru
*/

#include "StdAfx.h"

#include "MenuBar.h"
#include "SizableReBar.h"
#include "WinAppEx.h"

#pragma warning(disable: 4310)

CMenuBarButton::CMenuBarButton()
{
  m_bHidden = true;
  m_bPushed = false;
  
  SetMDIChild( 0 );
  SetMenuBarRect( 0 );
}

bool CMenuBarButton::HitTest( CPoint pt ) const
{
  return ( IsVisible() && GetButtonRect().PtInRect( pt ) );
}

void CMenuBarButton::SetMDIChild( HWND hWndMDIChild )
{
  m_hWndMDIChild = hWndMDIChild;
}

void CMenuBarButton::SetMenuBarRect( LPCRECT lpRect )
{
  if ( lpRect != 0 )
  {
    m_rcMenuBar = *lpRect;
  }
  else
  {
    m_rcMenuBar.SetRectEmpty();
  }
}

DWORD CMenuBarButton::GetMDIChildStyle() const
{
  return ( DWORD )::GetWindowLong( m_hWndMDIChild, GWL_STYLE );
}

bool CMenuBarButton::IsEnabled() const
{
  return true;
}

bool CMenuBarButton::IsVisible() const
{
  return ( !m_bHidden && ::IsWindow( m_hWndMDIChild ) && ( GetMDIChildStyle() & WS_SYSMENU ) );
}

bool CMenuBarButton::IsPushed() const
{
  return m_bPushed;
}

bool CMenuBarButton::HideButton( bool bHide )
{
  bool bWasHidden = m_bHidden;
  m_bHidden = bHide;
  return bWasHidden;
}

bool CMenuBarButton::PushButton( bool bPush )
{
  bool bWasPushed = m_bPushed;
  m_bPushed = bPush;
  return bWasPushed;
}

CSize CMenuBarButton::GetButtonSize()
{
  // Sys-menu icon and caption buttons are all inside
  // a rectangle of the following size:
  const NONCLIENTMETRICS& info = CWinAppEx::GetInstance()->GetNonClientMetrics();
  return CSize(
    info.iMenuWidth,     //::GetSystemMetrics( SM_CXMENUSIZE ),
    info.iMenuHeight );  //::GetSystemMetrics( SM_CYMENUSIZE ) );
}

void CMenuBarButton::DrawButton( CDC* pDC )
{
  if ( IsVisible() )
  {
    UINT nState;
    switch ( GetSysCommandID() )
    {
    case SC_CLOSE:
      nState = DFCS_CAPTIONCLOSE;
      break;
    case SC_MINIMIZE:
      nState = DFCS_CAPTIONMIN;
      break;
    case SC_MAXIMIZE:
      nState = DFCS_CAPTIONMAX;
      break;
    case SC_RESTORE:
      nState = DFCS_CAPTIONRESTORE;
      break;
    default:
      ASSERT( false );
      return;
    }
    CRect rc = GetButtonRect();
    VERIFY( pDC->DrawFrameControl( rc, DFC_CAPTION, nState |
      ( IsPushed() ? DFCS_PUSHED : 0 ) |
      ( IsEnabled() ? 0 : DFCS_INACTIVE ) ) );
  }
}

/////////////////////////////////////////////////////////////////////////////
// CMenuBarButtonMin

CRect CMenuBarButtonMin::GetButtonRect() const
{
  // Minimize box has 2 pixel border on all sides but right
  CSize szButton = GetButtonSize();
  szButton.cx -= 2;
  szButton.cy -= 4;
  
  CPoint ptTopLeft;
  ptTopLeft.x = m_rcMenuBar.right - ( szButton.cx + 2 ) * 3 + 2;
  ptTopLeft.y = m_rcMenuBar.top + ( m_rcMenuBar.Height() - szButton.cy ) / 2;
  
  return CRect( ptTopLeft, szButton );
}

UINT CMenuBarButtonMin::GetSysCommandID() const
{
  return ::IsIconic( m_hWndMDIChild ) ? SC_RESTORE : SC_MINIMIZE;
}

bool CMenuBarButtonMin::IsEnabled() const
{
  return ( ( GetMDIChildStyle() & WS_MINIMIZEBOX ) != 0 );
}

bool CMenuBarButtonMin::IsVisible() const
{
  return CMenuBarButton::IsVisible() &&
    ( ( GetMDIChildStyle() & ( WS_MINIMIZEBOX | WS_MAXIMIZEBOX ) ) != 0 );
}

/////////////////////////////////////////////////////////////////////////////
// CMenuBarButtonMax

CRect CMenuBarButtonMax::GetButtonRect() const
{
  // Max box has a 2 pixel border on all sides but left, which is zero
  CSize szButton = GetButtonSize();
  szButton.cx -= 2;
  szButton.cy -= 4;
  
  CPoint ptTopLeft;
  ptTopLeft.x = m_rcMenuBar.right - ( szButton.cx + 2 ) * 2;
  ptTopLeft.y = m_rcMenuBar.top + ( m_rcMenuBar.Height() - szButton.cy ) / 2;
  
  return CRect( ptTopLeft, szButton );
}

UINT CMenuBarButtonMax::GetSysCommandID() const
{
  return ::IsZoomed( m_hWndMDIChild ) ? SC_RESTORE : SC_MAXIMIZE;
}

bool CMenuBarButtonMax::IsEnabled() const
{
  return ( ( GetMDIChildStyle() & WS_MAXIMIZEBOX ) != 0 );
}

bool CMenuBarButtonMax::IsVisible() const
{
  return CMenuBarButton::IsVisible() &&
    ( ( GetMDIChildStyle() & ( WS_MINIMIZEBOX | WS_MAXIMIZEBOX ) ) != 0 );
}

/////////////////////////////////////////////////////////////////////////////
// CMenuBarButtonClose

CRect CMenuBarButtonClose::GetButtonRect() const
{
  // Close box has a 2 pixel border on all sides but left, which is zero
  CSize szButton = GetButtonSize();
  szButton.cx -= 2;
  szButton.cy -= 4;
  
  CPoint ptTopLeft;
  ptTopLeft.x = m_rcMenuBar.right - ( szButton.cx + 2 );
  ptTopLeft.y = m_rcMenuBar.top + ( m_rcMenuBar.Height() - szButton.cy ) / 2;
  
  return CRect( ptTopLeft, szButton );
}

UINT CMenuBarButtonClose::GetSysCommandID() const
{
  return SC_CLOSE;
}

/////////////////////////////////////////////////////////////////////////////
// CMenuBar

#define IDBUTTON_FIRST      65000
#define IDBUTTON_SYSMENU    65000
#define IDBUTTON_LAST       65100

// Static member variables
HHOOK     CMenuBar::m_hMsgHook = 0;
CMenuBar* CMenuBar::m_pMenuBar = 0;

IMPLEMENT_DYNAMIC( CMenuBar, CToolBar )

CMenuBar::CMenuBar()
{
  m_hWndMDIChild   = 0;
  m_hWndOldFocus   = 0;
  m_hMenu          = 0;
  m_hMenuTracking  = 0;
  m_bItemTracking  = false;
  m_bItemDropped   = false;
  m_bButtonCapture = false;
  m_bFrameActive   = true;
  
  // Create menu bar buttons
  //m_aMenuBarButtons.Add( new CMenuBarButtonMin   );
  //m_aMenuBarButtons.Add( new CMenuBarButtonMax   );
  //m_aMenuBarButtons.Add( new CMenuBarButtonClose );
}

CMenuBar::~CMenuBar()
{
  if ( m_fontMenu.GetSafeHandle() != 0 )
  {
    VERIFY( m_fontMenu.DeleteObject() );
  }
  
  for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
  {
    delete m_aMenuBarButtons[ nIndex ];
  }
  
  m_aMenuBarButtons.RemoveAll();
}

/////////////////////////////////////////////////////////////////////////////
// Operations

bool CMenuBar::SetMenu( HMENU hMenu )
{
  // Delete old buttons
  while ( m_nCount > 0 )
  {
    VERIFY( GetToolBarCtrl().DeleteButton( --m_nCount ) );
  }
  
  CMenu* pMenu = CMenu::FromHandle( hMenu );
  if ( pMenu != 0 )
  {
    // Allocate space for buttons
    UINT nItems = pMenu->GetMenuItemCount();
    VERIFY( SetButtons( 0, nItems + 1 ) );
    
    // Add sys-menu button which is the leftmost
    SetButtonInfo( 0, IDBUTTON_SYSMENU,
      TBBS_BUTTON | TBBS_DROPDOWN | TBBS_NOPREFIX, 0 );
    
    // Add all other buttons
    for ( UINT nIndex = 0; nIndex < nItems; nIndex++ )
    {
      UINT nID    = pMenu->GetMenuItemID( nIndex );
      UINT nStyle = TBBS_BUTTON | TBBS_AUTOSIZE | TBBS_DROPDOWN;
      
      switch ( nID )
      {
      case -1:
        nID = IDBUTTON_SYSMENU + 1 + nIndex;
        break;
      case 0:
        nStyle = TBBS_SEPARATOR;
        break;
      default:
        nStyle ^= TBBS_DROPDOWN;
        //this assert would kill my About in menu for no useful reason -> leave it out
        //ASSERT( false );    // not supported yet
        break;
      }
      
      CString strMenu;
      pMenu->GetMenuString( nIndex, strMenu, MF_BYPOSITION );
      
      SetButtonInfo( nIndex + 1, nID, nStyle, 0 /*-2=I_IMAGENONE*/ );
      VERIFY( SetButtonText( nIndex + 1, strMenu ) );
    }
  }
  
  UpdateMenuBar();
  
  m_hMenu = hMenu;
  return true;
}

HMENU CMenuBar::GetMenu() const
{
  return m_hMenu;
}

/////////////////////////////////////////////////////////////////////////////
// Implementation

CReBarCtrl& CMenuBar::GetParentReBarCtrl() const
{
  return STATIC_DOWNCAST( CReBar, GetParent() )->GetReBarCtrl();
}

int CMenuBar::GetParentBandIndex() const
{
  int nBand = GetParentReBarCtrl().IDToIndex( ( UINT )GetDlgCtrlID() );
  ASSERT( nBand != -1 );
  return nBand;
}

void CMenuBar::SetButtonWidth( UINT nID, int nWidth )
{
  TBBUTTONINFO tbbi;
  tbbi.cbSize = sizeof( tbbi );
  tbbi.dwMask = TBIF_SIZE;
  tbbi.cx     = ( WORD )nWidth;
  
  VERIFY( GetToolBarCtrl().SetButtonInfo( nID, &tbbi ) );
}

void CMenuBar::UpdateMenuBar()
{
  // Set new font
  if ( m_fontMenu.GetSafeHandle() != 0 )
  {
    VERIFY( m_fontMenu.DeleteObject() );
  }
  
  VERIFY( m_fontMenu.CreateFontIndirect(
    &CWinAppEx::GetInstance()->GetNonClientMetrics().lfMenuFont ) );
  SetFont( &m_fontMenu, TRUE );
  
  // Calc row height
  int cyMenu = ::GetSystemMetrics( SM_CYMENU );
  int cyRow  = cyMenu + HIWORD( SendMessage( TB_GETPADDING ) );
  
  CToolBarCtrl& tbCtrl = GetToolBarCtrl();
  VERIFY( tbCtrl.SetBitmapSize( CSize( 0, cyMenu ) ) );
  
  // Adjust appearance of sys-menu button
//  bool bSysMenu = ( m_hWndMDIChild != 0 );
  bool bSysMenu = false;
  int cxButton = bSysMenu ? CMenuBarButton::GetButtonSize().cx : 0;
  if ( GetCount() > 0 )
  {
    VERIFY( tbCtrl.HideButton( IDBUTTON_SYSMENU, !bSysMenu ) );
    if ( bSysMenu )
    {
      // Adjust sys-menu button width
      SetButtonWidth( IDBUTTON_SYSMENU, cxButton );
    }
  }
  
  // Calc minimal and ideal width of the menu bar
  int cxIdeal = cxButton * 3;
  for ( int nIndex = 0; nIndex < GetCount(); nIndex++ )
  {
    CRect rcItem;
    if ( tbCtrl.GetItemRect( nIndex, rcItem ) )
    {
      cxIdeal += rcItem.Width();
    }
  }
  
  // Our parent must be a re-bar control by design.
  // So, minimal and ideal width as well as minimal height of
  // the parent band should also be adjusted accordingly.
  REBARBANDINFO rbbi;
  rbbi.cbSize     = sizeof( rbbi );
  rbbi.fMask      = RBBIM_CHILDSIZE | RBBIM_IDEALSIZE;
  rbbi.cxMinChild = cxButton;
  rbbi.cyMinChild = cyRow;
  rbbi.cxIdeal    = cxIdeal;
  VERIFY( GetParentReBarCtrl().SetBandInfo( GetParentBandIndex(), &rbbi ) );
  
  CRect rcMenuBar;
  GetClientRect( rcMenuBar );
  RepositionSysButtons( rcMenuBar );
}

void CMenuBar::RepositionSysButtons( CRect rcMenuBar )
{
  CRect rcItem;
  GetToolBarCtrl().GetItemRect( GetCount() - 1, rcItem );
  
  int cxAvailable = ( rcMenuBar.right - rcItem.right );
  int cxNeeded    = ( CMenuBarButton::GetButtonSize().cx * 3 );
  if ( cxAvailable < cxNeeded )
  {
    // Without this sys-buttons would overlap menu items
    rcMenuBar.right = rcItem.right + cxNeeded;
  }
  
  for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
  {
    CMenuBarButton* pButton = m_aMenuBarButtons[ nIndex ];
    pButton->SetMenuBarRect( rcMenuBar );
    pButton->HideButton( m_hWndMDIChild == 0 );
  }
  
  Invalidate();
}

void CMenuBar::EnterTrackingMode( int nItem )
{
  if ( !m_bItemTracking )
  {
    m_bItemTracking = true;
    m_hWndOldFocus  = 0;
    
    // Gain focus
    if ( GetFocus() != this )
    {
      m_hWndOldFocus = SetFocus()->GetSafeHwnd();
    }
    
    // Capture mouse
    if ( GetCapture() != this )
    {
      SetCapture();
      SendMessage( WM_SETCURSOR, ( WPARAM )m_hWnd, MAKELPARAM( HTCLIENT, 0 ) );
    }
    
    GetToolBarCtrl().SetHotItem( nItem );
  }
}

void CMenuBar::TrackChevronMenu( CRect& rcChevron, int nItem )
{
  ExitTrackingMode();
  
  CMenu* pMenu = CMenu::FromHandle( m_hMenu );
  if ( pMenu != 0 )
  {
    CWinAppEx* pApp = CWinAppEx::GetInstance();
    
    // Create new popup menu
    CMenu menu;
    VERIFY( menu.CreatePopupMenu() );
    
    TCHAR szBuffer[ MAX_PATH ];
    MENUITEMINFO mii;
    mii.cbSize = sizeof( mii );
    
    // Populate popup menu by adding clipped items
    UINT nItems = pMenu->GetMenuItemCount();
    for ( UINT nIndex = 0; nIndex < nItems; nIndex++ )
    {
      if ( IsItemClipped( nIndex + 1 ) )
      {
        mii.fMask      = MIIM_ID | MIIM_STATE | MIIM_SUBMENU | MIIM_TYPE;
        mii.dwTypeData = szBuffer;
        mii.cch        = sizeof( szBuffer ) / sizeof( szBuffer[ 0 ] );
        VERIFY( pMenu->GetMenuItemInfo( nIndex, &mii, TRUE ) );
        
        UINT nMenuItems = menu.GetMenuItemCount();
        if ( ( nMenuItems > 0 ) || !( mii.fType & MFT_SEPARATOR ) )
        {
          VERIFY( ::InsertMenuItem( menu.m_hMenu, nMenuItems, TRUE, &mii ) );
        }
      }
    }
    
    // Populate popup menu by adding menu items for clipped sys-buttons
    if ( m_hWndMDIChild != 0 )
    {
      CMenu* pSysMenu = CMenu::FromHandle( ::GetSystemMenu( m_hWndMDIChild, FALSE ) );
      ASSERT( pSysMenu != 0 );
      
      if ( menu.GetMenuItemCount() > 0 )
      {
        VERIFY( menu.AppendMenu( MF_SEPARATOR ) );
      }
      
      CRect rcClient;
      GetClientRect( rcClient );
      
      for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
      {
        CMenuBarButton* pButton = m_aMenuBarButtons[ nIndex ];
        if ( pButton->GetButtonRect().right > rcClient.right )
        {
          UINT nID = pButton->GetSysCommandID();
          if ( pApp->IsWin98_2K() )
          {
            MENUITEMINFO_WIN50 mii;
            mii.cbSize     = sizeof( mii );
            mii.fMask      = MIIM_ID | MIIM_STATE | MIIM_FTYPE | MIIM_STRING | MIIM_BITMAP;
            mii.dwTypeData = szBuffer;
            mii.cch        = sizeof( szBuffer ) / sizeof( szBuffer[ 0 ] );
            VERIFY( pSysMenu->GetMenuItemInfo( nID, &mii ) );
            
            mii.fState    &= ~MFS_DEFAULT;
            VERIFY( ::InsertMenuItem( menu.m_hMenu, menu.GetMenuItemCount(), TRUE, &mii ) );
          }
          else
          {
            mii.fMask      = MIIM_STATE | MIIM_TYPE;
            mii.dwTypeData = szBuffer;
            mii.cch        = sizeof( szBuffer ) / sizeof( szBuffer[ 0 ] );
            VERIFY( pSysMenu->GetMenuItemInfo( nID, &mii ) );
            
            mii.fState    &= ~MFS_DEFAULT;
            VERIFY( menu.AppendMenu( mii.fState | MF_STRING, nID, szBuffer ) );
          }
        }
      }
    }
    
    // Expand sub-menu item, if requested
    if ( nItem >= 0 )
    {
      CString strText;
      GetButtonText( nItem, strText );
      
      for ( int nPos = 0; strText[ nPos ]; nPos++ )
      {
        if ( strText[ nPos ] == _T('&') )
        {
          nPos++;
          
          if ( strText[ nPos ] && ( strText[ nPos ] != _T('&') ) )
          {
            // Simulate hot key press
            ::keybd_event( strText[ nPos ], 0, 0, 0 );
            ::keybd_event( strText[ nPos ], 0, KEYEVENTF_KEYUP, 0 );
            break;
          }
        }
      }
    }
    
    // Now, just track menu
    TPMPARAMS tpm;
    tpm.cbSize    = sizeof( tpm );
    tpm.rcExclude = rcChevron;
    
    CWinAppEx::TrackPopupMenuEx( menu.m_hMenu,
      TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_VERTICAL | ( pApp->IsWin98_2K() ? TPM_VERPOSANIMATION : 0 ),
      rcChevron.left, rcChevron.bottom, GetParentFrame(), &tpm );
    
    // Clean-up
    while ( menu.RemoveMenu( 0, MF_BYPOSITION ) );
    VERIFY( menu.DestroyMenu() );
    }
}

void CMenuBar::TrackPopupMenu()
{
  ASSERT( m_hMenu != 0 );
  
  CWinAppEx* pApp = CWinAppEx::GetInstance();
  UINT nMenuAnimation = TPM_VERPOSANIMATION;
  for ( int nItem = m_nItem; m_bContinue; nItem = m_nItem )
  {
    // Get popup menu to be tracked
    m_hMenuTracking = ( nItem == 0 ) ?
      ::GetSystemMenu( m_hWndMDIChild, FALSE ) :
    ::GetSubMenu( m_hMenu, nItem - 1 );
    if ( m_hMenuTracking == 0 )
    {
      //this assert would kill my About in menu for no useful reason -> leave it out
      //ASSERT( false );
      break;
    }
    
    CToolBarCtrl& tbCtrl = GetToolBarCtrl();
    tbCtrl.PressButton( GetItemID( nItem ), TRUE );
    
    if ( m_bSelectFirst )
    {
      // Select first menu item
      ::keybd_event( VK_DOWN, 0, 0, 0 );
      ::keybd_event( VK_DOWN, 0, KEYEVENTF_KEYUP, 0 );
    }
    
    CRect rc;
    tbCtrl.GetItemRect( nItem, rc );
    ClientToScreen( rc );
    
    TPMPARAMS tpm;
    tpm.cbSize    = sizeof( tpm );
    tpm.rcExclude = rc;
    
    m_bPrimaryMenu = false;
    m_bSubmenuItem = false;
    m_bContinue    = false;
    
    // Install message hook
    m_bItemDropped = true;
    m_pMenuBar     = this;
    m_hMsgHook     = ::SetWindowsHookEx( WH_MSGFILTER, MessageProc, 0, ::GetCurrentThreadId() );
    ASSERT( m_hMsgHook != 0 );
    
    CWinAppEx::TrackPopupMenuEx( m_hMenuTracking,
      TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_VERTICAL | ( pApp->IsWin98_2K() ? nMenuAnimation : 0 ),
      rc.left, rc.bottom, GetParentFrame(), &tpm );
    
    // Uninstall message hook
    VERIFY( ::UnhookWindowsHookEx( m_hMsgHook ) );
    m_hMsgHook      = 0;
    m_pMenuBar      = 0;
    m_hMenuTracking = 0;
    m_bItemDropped  = false;
    
    tbCtrl.PressButton( GetItemID( nItem ), FALSE );
    
    if ( pApp->IsWin2K() )
    {
      // W2K: use animation only for the first menu
      nMenuAnimation = TPM_NOANIMATION;
    }
  }
}

void CMenuBar::ContinueTracking( bool bSelectFirst )
{
  GetParentFrame()->PostMessage( WM_CANCELMODE );   // close currently tracked menu
  
  m_bSelectFirst = bSelectFirst;
  m_bContinue    = true;
}

void CMenuBar::ExitTrackingMode()
{
  if ( m_bItemTracking )
  {
    GetToolBarCtrl().SetHotItem( -1 );
    
    // Restore focus
    if ( ( GetFocus() == this ) && ::IsWindow( m_hWndOldFocus ) )
    {
      ::SetFocus( m_hWndOldFocus );
    }
    
    // Release capture
    if ( GetCapture() == this )
    {
      VERIFY( ReleaseCapture() );
    }
    
    m_bItemTracking = false;
    m_hWndOldFocus  = 0;
  }
}

void CMenuBar::ShowChevronMenu( int nItem )
{
  if ( m_bItemDropped )
  {
    GetParentFrame()->PostMessage( WM_CANCELMODE );   // close currently tracked menu
  }
  
  GetParentReBarCtrl().PostMessage( RB_PUSHCHEVRON, GetParentBandIndex(), nItem + 1 );
}

bool CMenuBar::IsItemClipped( int nItem ) const
{
  CRect rcClient;
  GetClientRect( rcClient );
  
  CRect rcItem;
  GetToolBarCtrl().GetItemRect( nItem, rcItem );
  
  return ( rcItem.right > rcClient.right );
}

bool CMenuBar::IsOverChevron( CPoint pt ) const
{
  CPoint ptScreen( pt );
  ClientToScreen( &ptScreen );
  
  CReBarCtrl& rbCtrl = GetParentReBarCtrl();
  
  // Is mouse over parent re-bar control?
  if ( WindowFromPoint( ptScreen ) == &rbCtrl )
  {
    MapWindowPoints( &rbCtrl, &pt, 1 );
    
    RBHITTESTINFO rbht;
    rbht.pt = pt;
    
    // Is mouse over chevron of the parent band?
    return ( ( rbCtrl.HitTest( &rbht ) != -1 ) &&
      ( rbht.flags == RBHT_CHEVRON ) &&
      ( rbht.iBand == GetParentBandIndex() ) );
  }
  
  return false;
}

/////////////////////////////////////////////////////////////////////////////
// Overrides

void CMenuBar::OnUpdateCmdUI( CFrameWnd* pTarget, BOOL bDisableIfNoHndler )
{
  CMDIFrameWnd* pMDIFrame = DYNAMIC_DOWNCAST( CMDIFrameWnd, GetParentFrame() );
  if ( pMDIFrame != 0 )
  {
    // Check active MDI child
    BOOL bMaximized;
    CMDIChildWnd* pActive = pMDIFrame->MDIGetActive( &bMaximized );
    
    bool bSysMenu = ( pActive != 0 ) && ( pActive->GetStyle() & WS_SYSMENU ) && bMaximized;
    HWND hWndMDIChild = bSysMenu ? pActive->GetSafeHwnd() : 0;
    
    // Is it new MDI child?
    if ( hWndMDIChild != m_hWndMDIChild )
    {
      m_hWndMDIChild = hWndMDIChild;
      
      for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
      {
        m_aMenuBarButtons[ nIndex ]->SetMDIChild( m_hWndMDIChild );
      }
      
      UpdateMenuBar();
    }
  }
  
  CWinAppEx* pApp = CWinAppEx::GetInstance();
  if ( pApp->IsWin2K() && !m_bItemTracking )
  {
    BOOL bAltDown = ( ::GetKeyState( VK_MENU ) & 0x8000 );
    BOOL bF10Down = ( ::GetKeyState( VK_F10  ) & 0x8000 );
    DWORD dwDTFlags = ( pApp->GetMenuUnderlines() || bAltDown || bF10Down ) ? 0 : DT_HIDEPREFIX;
    if ( GetToolBarCtrl().SetDrawTextFlags( DT_HIDEPREFIX, dwDTFlags ) != dwDTFlags )
    {
      Invalidate();
    }
  }
  
  CToolBar::OnUpdateCmdUI( pTarget, FALSE);
}

int CMenuBar::OnToolHitTest( CPoint point, TOOLINFO* pTI ) const
{
  UINT nTipID = ( UINT )-1;
  
  if ( CWinAppEx::GetInstance()->IsWin98_2K() ) // W98/W2K specific
  {
    // If mouse is over any of the sys-buttons, show corresponding tooltip
    for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
    {
      CMenuBarButton* pButton = m_aMenuBarButtons[ nIndex ];
      if ( pButton->HitTest( point ) )
      {
        switch ( pButton->GetSysCommandID() )
        {
        case SC_CLOSE:
          nTipID = IDS_TOOLTIP_CLOSE;
          break;
        case SC_MINIMIZE:
          nTipID = IDS_TOOLTIP_MINIMIZE;
          break;
        case SC_MAXIMIZE:
          nTipID = IDS_TOOLTIP_MAXIMIZE;
          break;
        case SC_RESTORE:
          nTipID = IDS_TOOLTIP_RESTORE;
          break;
        }
        
        if ( ( nTipID != ( UINT )-1 ) && ( pTI != 0 ) )
        {
          pTI->uFlags   = TTF_TRANSPARENT;
          pTI->rect     = pButton->GetButtonRect();
          pTI->hwnd     = m_hWnd;
          pTI->uId      = nTipID;
          pTI->lpszText = LPSTR_TEXTCALLBACK;
        }
        
        break;
      }
    }
  }
  
  return nTipID;  // CToolBar::OnToolHitTest( point, pTI );
}

/////////////////////////////////////////////////////////////////////////
// CMenuBar message handlers

BEGIN_MESSAGE_MAP(CMenuBar, CToolBar)
//{{AFX_MSG_MAP(CMenuBar)
ON_WM_RBUTTONDOWN()
ON_WM_LBUTTONDOWN()
ON_WM_LBUTTONUP()
ON_WM_MOUSEMOVE()
ON_WM_LBUTTONDBLCLK()
ON_WM_CAPTURECHANGED()
ON_WM_KEYDOWN()
ON_WM_SYSKEYDOWN()
ON_WM_KILLFOCUS()
ON_WM_GETDLGCODE()
ON_WM_SETTINGCHANGE()
//}}AFX_MSG_MAP

ON_MESSAGE( WM_MB_SHOWPOPUPMENU, OnShowPopupMenu )
ON_MESSAGE( WM_REBAR_CHILDSIZE, OnReBarChildSize )
ON_MESSAGE( WM_REBAR_CHEVRONPUSHED, OnReBarChevronPushed )

ON_UPDATE_COMMAND_UI_REFLECT( OnUpdateCommandUI )

// Toolbar control notifications
ON_NOTIFY_REFLECT( TBN_DROPDOWN, OnDropDown )
ON_NOTIFY_REFLECT( TBN_HOTITEMCHANGE, OnHotItemChange )
ON_NOTIFY_REFLECT( NM_CUSTOMDRAW, OnCustomDraw )
END_MESSAGE_MAP()

bool CMenuBar::OnButtonDown( UINT nFlags, CPoint pt, bool bLeft )
{
  if ( m_bItemTracking && !m_bItemDropped )
  {
    ClientToScreen( &pt );
    CWnd* pWnd = WindowFromPoint( pt );
    if ( pWnd != this )
    {
      // It's a good idea to allow user exit tracking mode
      // by clicking mouse anywhere outside the menu bar.
      ExitTrackingMode();
      
      // It would be also nice to forward this event to
      // the window the user has just clicked.
      if ( pWnd != 0 )
      {
        LPARAM nPosition = MAKELPARAM( pt.x, pt.y );
        WPARAM nHitTest  = pWnd->SendMessage( WM_NCHITTEST, 0, nPosition );
        
        // Is over client area of the window?
        if ( nHitTest == HTCLIENT )
        {
          pWnd->ScreenToClient( &pt );
          LPARAM nPosition = MAKELPARAM( pt.x, pt.y );
          pWnd->PostMessage( bLeft ? WM_LBUTTONDOWN : WM_RBUTTONDOWN,
            nFlags, nPosition );
        }
        else
        {
          pWnd->PostMessage( bLeft ? WM_NCLBUTTONDOWN : WM_NCRBUTTONDOWN,
            nHitTest, nPosition );
        }
      }
      
      return false;
    }
  }
  
  return true;
}

void CMenuBar::OnRButtonDown( UINT nFlags, CPoint point )
{
  if ( OnButtonDown( nFlags, point, false ) )
  {
    CToolBar::OnRButtonDown( nFlags, point );
  }
}

void CMenuBar::OnLButtonDown( UINT nFlags, CPoint point )
{
  if ( OnButtonDown( nFlags, point, true ) )
  {
    // Was it over any of the sys-buttons?
    for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
    {
      CMenuBarButton* pButton = m_aMenuBarButtons[ nIndex ];
      if ( pButton->IsEnabled() && pButton->HitTest( point ) )
      {
        ExitTrackingMode(); // just in case
        
        pButton->PushButton( true );
        RedrawWindow( pButton->GetButtonRect(), 0,
          RDW_INVALIDATE | RDW_ERASE );   // visual feedback
        
        SetCapture();
        m_bButtonCapture = true;
        return;
      }
    }
    
    CToolBar::OnLButtonDown( nFlags, point );
  }
}

void CMenuBar::OnLButtonUp( UINT nFlags, CPoint point )
{
  if ( m_bButtonCapture )
  {
    // Was it over any of the sys-buttons?
    for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
    {
      CMenuBarButton* pButton = m_aMenuBarButtons[ nIndex ];
      if ( pButton->IsEnabled() && pButton->HitTest( point ) )
      {
        pButton->PushButton( false );
        RedrawWindow( pButton->GetButtonRect(), 0,
          RDW_INVALIDATE | RDW_ERASE );   // visual feedback
        
        // Send corresponding sys-command
        ::PostMessage( m_hWndMDIChild, WM_SYSCOMMAND, pButton->GetSysCommandID(), 0 );
        break;
      }
    }
    
    VERIFY( ReleaseCapture() );
    m_bButtonCapture = false;
    return;
  }
  
  CToolBar::OnLButtonUp( nFlags, point );
}

void CMenuBar::OnMouseMove( UINT nFlags, CPoint point )
{
  if ( m_bItemTracking )
  {
    if ( m_bItemDropped && ( m_ptMouseLast == point ) )
    {
      return; // mouse has not actually moved
    }
    
    if ( IsOverChevron( point ) )
    {
      ShowChevronMenu( -1 );
      return;
    }
    
    CPoint ptScreen( point );
    ClientToScreen( &ptScreen );
    
    if ( WindowFromPoint( ptScreen ) != this )
    {
      return;
    }
  }
  
  m_ptMouseLast = point;
  
  if ( !m_bItemTracking && m_bButtonCapture )
  {
    // Are we over any of the sys-buttons?
    for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
    {
      CMenuBarButton* pButton = m_aMenuBarButtons[ nIndex ];
      bool bPush = pButton->IsEnabled() && pButton->HitTest( point );
      if ( bPush != pButton->PushButton( bPush ) )
      {
        RedrawWindow( pButton->GetButtonRect(), 0,
          RDW_INVALIDATE | RDW_ERASE );   // visual feedback
      }
    }
    
    return;
  }
  
  CToolBar::OnMouseMove( nFlags, point );
}

void CMenuBar::OnLButtonDblClk( UINT nFlags, CPoint point )
{
  CRect rcSysMenu;
  GetItemRect( 0, rcSysMenu );
  
  // If we are over sys-menu button, send Close command
  if ( rcSysMenu.PtInRect( point ) )
  {
    ::PostMessage( m_hWndMDIChild, WM_SYSCOMMAND, SC_CLOSE, 0 );
    return;
  }
  
  CToolBar::OnLButtonDblClk( nFlags, point );
}

void CMenuBar::OnCaptureChanged( CWnd* pWnd )
{
  if ( m_bButtonCapture && ( pWnd != this ) )
  {
    for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
    {
      CMenuBarButton* pButton = m_aMenuBarButtons[ nIndex ];
      if ( pButton->PushButton( false ) )
      {
        RedrawWindow( pButton->GetButtonRect(), 0,
          RDW_INVALIDATE | RDW_ERASE );
      }
    }
    
    m_bButtonCapture = false;
  }
  
  CToolBar::OnCaptureChanged( pWnd );
}

void CMenuBar::OnKeyDown( UINT nChar, UINT nRepCnt, UINT nFlags )
{
  if ( m_bItemTracking )
  {
    switch ( nChar )
    {
    case VK_SPACE:
      ExitTrackingMode();
      GetParentFrame()->PostMessage( WM_SYSCOMMAND, SC_KEYMENU, 32 );
      return;
    case VK_ESCAPE:
      ExitTrackingMode();
      return;
    default:
      break;
    }
  }
  
  CToolBar::OnKeyDown( nChar, nRepCnt, nFlags );
}

void CMenuBar::OnSysKeyDown( UINT /*nChar*/, UINT /*nRepCnt*/, UINT /*nFlags*/ )
{
  ExitTrackingMode();
  
  //    CToolBar::OnSysKeyDown( nChar, nRepCnt, nFlags );
}

void CMenuBar::OnKillFocus( CWnd* pNewWnd )
{
  CToolBar::OnKillFocus( pNewWnd );
  
  if ( m_bItemTracking )
  {
    ExitTrackingMode();
  }
}

UINT CMenuBar::OnGetDlgCode() 
{
  return ( CToolBar::OnGetDlgCode() | ( m_bItemTracking ? DLGC_WANTALLKEYS : 0 ) );
}

void CMenuBar::OnSettingChange( UINT uFlags, LPCTSTR lpszSection )
{
  CToolBar::OnSettingChange( uFlags, lpszSection );
  
  UpdateMenuBar();
}

LRESULT CMenuBar::OnShowPopupMenu( WPARAM wParam, LPARAM lParam )
{
  int  nItem         = ( int )wParam;
  bool bSelectFirst  = ( LOWORD( lParam ) != 0 );
  bool bCheckClipped = ( HIWORD( lParam ) != 0 );
  
  // If item is clipped, show chevron menu
  if ( bCheckClipped && IsItemClipped( nItem ) )
  {
    ShowChevronMenu( nItem );
  }
  // otherwise, track "normal" popup menu
  else
  {
    EnterTrackingMode( nItem );
    
    m_nItem        = nItem;
    m_bSelectFirst = bSelectFirst;
    m_bContinue    = true;
    m_bEscape      = false;
    
    TrackPopupMenu();
    
    // If menu was closed by pressing Esc, stay in tracking mode
    if ( m_bEscape )
    {
      SetCapture();
      SendMessage( WM_SETCURSOR, ( WPARAM )m_hWnd, MAKELPARAM( HTCLIENT, 0 ) );
    }
    // otherwise, exit tracking mode
    else
    {
      ExitTrackingMode();
    }
  }
  
  return 0L;
}

LRESULT CMenuBar::OnReBarChildSize( WPARAM /*wParam*/, LPARAM lParam )
{
  //  CRect rcBand ( ( LPCRECT )wParam );
  CRect rcChild( ( LPCRECT )lParam );
  
  rcChild.OffsetRect( -rcChild.TopLeft() );
  RepositionSysButtons( rcChild );
  
  return 0L;
}

LRESULT CMenuBar::OnReBarChevronPushed( WPARAM wParam, LPARAM lParam )
{
  CRect rcChevron( ( LPCRECT )wParam );
  int nItem = ( int )lParam - 1;
  
  TrackChevronMenu( rcChevron, nItem );
  
  return 0L;
}

void CMenuBar::OnUpdateCommandUI( CCmdUI* pCmdUI )
{
  // Do not disable menu buttons
  if ( ( pCmdUI->m_nID < IDBUTTON_FIRST ) || ( pCmdUI->m_nID > IDBUTTON_LAST ) )
  {
    pCmdUI->ContinueRouting();
  }
}

void CMenuBar::OnDropDown( NMHDR* pNMHDR, LRESULT* pResult )
{
  NMTOOLBAR* pNMToolBar = ( NMTOOLBAR* )pNMHDR;
  
  if ( !m_bItemDropped )
  {
    int  nItem        = CommandToIndex( pNMToolBar->iItem );
    bool bLButtonDown = ( ::GetKeyState( VK_LBUTTON ) < 0 );
    
    PostMessage( WM_MB_SHOWPOPUPMENU, nItem,
      MAKELPARAM( bLButtonDown ? FALSE : TRUE, bLButtonDown ? FALSE : TRUE ) );
  }
  
  *pResult = TBDDRET_DEFAULT;
}

void CMenuBar::OnHotItemChange( NMHDR* pNMHDR, LRESULT* pResult )
{
  NMTBHOTITEM* pNMTBHotItem = ( NMTBHOTITEM* )pNMHDR;
  
  if ( m_bItemTracking)
  {
    // Is it really new hot item?
    int nItem = ( pNMTBHotItem->dwFlags & HICF_LEAVING ) ? -1 : CommandToIndex( ( UINT )pNMTBHotItem->idNew );
    if ( m_nItem != nItem )
    {
      // Was it changed not by SetHotItem()?
      if ( pNMTBHotItem->dwFlags & ( HICF_ACCELERATOR | HICF_ARROWKEYS | HICF_DUPACCEL | HICF_MOUSE ) )
      {
        // There should be always hot item in tracking mode
        if ( nItem == -1)
        {
          *pResult = 1;
          return;
        }
        
        // If item is clipped, show chevron menu
        if ( IsItemClipped( nItem ) )
        {
          ShowChevronMenu( nItem );
        }
        // If we are tracking popup menu, track new one, associated with hot item
        else if ( m_bItemDropped )
        {
          ContinueTracking( !( pNMTBHotItem->dwFlags & HICF_MOUSE ) );
        }
      }
      
      m_nItem = nItem;    // remember last hot item
    }
  }
  
  *pResult = 0;
}

void CMenuBar::OnCustomDraw( NMHDR* pNMHDR, LRESULT* pResult )
{
  LPNMTBCUSTOMDRAW lpNMCustomDraw = ( LPNMTBCUSTOMDRAW )pNMHDR;
  
  switch ( lpNMCustomDraw->nmcd.dwDrawStage )
  {
  case CDDS_PREERASE:
    *pResult = CDRF_NOTIFYPOSTERASE;
    break;
    
  case CDDS_POSTERASE:
    {
      CDC* pDC = CDC::FromHandle( lpNMCustomDraw->nmcd.hdc );
      
      // Draw the sys-buttons
      for ( int nIndex = 0; nIndex <= m_aMenuBarButtons.GetUpperBound(); nIndex++ )
      {
        m_aMenuBarButtons[ nIndex ]->DrawButton( pDC );
      }
      
      *pResult = CDRF_DODEFAULT;
      break;
    }
  case CDDS_PREPAINT:
    *pResult = CDRF_NOTIFYITEMDRAW;
    break;
    
  case CDDS_ITEMPREPAINT:
    {
      NMCUSTOMDRAW& nmcd = lpNMCustomDraw->nmcd;
      CDC* pDC = CDC::FromHandle( nmcd.hdc );
      UINT nID = nmcd.dwItemSpec;
      
      switch ( nID )
      {
      case IDBUTTON_SYSMENU:  // draw sys-menu button
        {
          HICON hIcon = ( HICON )::GetClassLong( m_hWndMDIChild, GCL_HICONSM );
          if ( hIcon != 0 )
          {
            // Icon has 2 pixel border on left, 1 pixel on top/bottom, 0 right
            CSize sz( CMenuBarButton::GetButtonSize() );
            sz.cx -= 2;
            sz.cy -= 2;
            
            CRect rc( &nmcd.rc );
            rc.left  += 2;
            rc.top   += ( rc.Height() - sz.cy ) / 2;
            rc.right  = rc.left + sz.cx;
            rc.bottom = rc.top  + sz.cy;
            /*
            CImageList imageList;
            VERIFY( imageList.Create( sz.cx, sz.cy, ILC_COLOR | ILC_MASK, 1, 0 ) );
            VERIFY( imageList.Draw( pDC, imageList.Add( hIcon ), rc.TopLeft(),
            ( nmcd.uItemState & CDIS_HOT ) ? ILD_BLEND25 : ILD_NORMAL ) );
            */
            VERIFY( ::DrawIconEx( pDC->GetSafeHdc(), rc.left, rc.top, hIcon,
              rc.Width(), rc.Height(), 0, 0, DI_NORMAL ) );
          }
          
          *pResult = CDRF_SKIPDEFAULT;
          break;
        }
      default:
        if ( CWinAppEx::GetInstance()->IsWin98_2K() && GetToolBarCtrl().IsButtonEnabled( nID ) )
        {
          // If main frame is not active, gray buttons
          lpNMCustomDraw->clrText = ::GetSysColor(
            m_bFrameActive ? COLOR_MENUTEXT : COLOR_3DSHADOW );
        }
        
        *pResult = CDRF_DODEFAULT;
        break;
      }
      break;
    }
  default:
    *pResult = CDRF_DODEFAULT;
    break;
  }
}

/////////////////////////////////////////////////////////////////////////////
// Implementation

void CMenuBar::HookMessageProc( UINT message, WPARAM wParam, LPARAM lParam )
{
  switch ( message )
  {
  case WM_MOUSEMOVE:
  case WM_LBUTTONDBLCLK:
    {
      CPoint pt( LOWORD( lParam ), HIWORD( lParam ) );
      ScreenToClient( &pt );
      SendMessage( message, wParam, MAKELPARAM( pt.x, pt.y ) );
      break;
    }
  case WM_KEYDOWN:
    {
      UINT nChar = ( UINT )wParam;
      bool bForward = false;  // by default
      
      switch ( nChar )
      {
      case VK_LEFT:
        bForward = m_bPrimaryMenu;
        break;
      case VK_RIGHT:
        bForward = !m_bSubmenuItem;
        break;
      case VK_ESCAPE:
        m_bEscape = m_bPrimaryMenu;
        break;
      }
      
      // Should we forward this message to the menu bar?
      if ( bForward )
      {
        SendMessage( message, wParam, lParam );
      }
      break;
    }
  default:
    break;
  }
}

LRESULT CALLBACK CMenuBar::MessageProc( int code, WPARAM wParam, LPARAM lParam )
{
  ASSERT( m_pMenuBar != 0 );
  
  if ( code == MSGF_MENU )
  {
    MSG* pMsg = ( MSG* )lParam;
    m_pMenuBar->HookMessageProc( pMsg->message, pMsg->wParam, pMsg->lParam );
  }
  
  return ::CallNextHookEx( m_hMsgHook, code, wParam, lParam );
}

bool CMenuBar::FrameOnSysCommand( UINT nID, LPARAM lParam )
{
  if ( IsVisible() && ( ( nID & 0xFFF0 ) == SC_KEYMENU ) && ( lParam == 0 ) )
  {
    if ( IsItemClipped( 1 ) )
    {
      ShowChevronMenu( -1 );
    }
    else
    {
      EnterTrackingMode( 1 );
    }
    
    return true;
  }
  
  return false;
}

bool CMenuBar::FrameOnMenuChar( UINT nChar, UINT nFlags, CMenu* /*pMenu*/ )
{
  if ( IsVisible() && ( nFlags & MF_SYSMENU ) )
  {
    UINT nID;
    if ( GetToolBarCtrl().MapAccelerator( ( TCHAR )nChar, &nID ) )
    {
      PostMessage( WM_MB_SHOWPOPUPMENU, CommandToIndex( nID ), MAKELPARAM( TRUE, TRUE ) );
      return true;
    }
  }
  
  return false;
}

void CMenuBar::FrameOnNcActivate( BOOL bActive )
{
  m_bFrameActive = ( bActive == TRUE );
  
  Invalidate();
  UpdateWindow();
}

void CMenuBar::FrameOnInitMenuPopup( CMenu* pPopupMenu, UINT nIndex, BOOL /*bSysMenu*/ )
{
  if ( m_hMenuTracking != 0 )
  {
    HMENU hMenu = pPopupMenu->GetSafeHmenu();
    
    // We should know if it is primary menu and whether or not selected
    // item is submenu item to handle Esc/Left/Right keys correctly
    m_bPrimaryMenu = ( hMenu == m_hMenuTracking );
    m_bSubmenuItem = ( ::GetSubMenu( hMenu, nIndex ) != 0 );
  }
}

void CMenuBar::FrameOnMenuSelect( UINT /*nItemID*/, UINT nFlags, HMENU hSysMenu )
{
  if ( m_hMenuTracking != 0 )
  {
    // We should know if it is primary menu and whether or not selected
    // item is submenu item to handle Esc/Left/Right keys correctly
    m_bPrimaryMenu = ( hSysMenu == m_hMenuTracking );
    m_bSubmenuItem = ( nFlags & MF_POPUP ) != 0;//( ::GetSubMenu( hSysMenu, nItemID ) != 0 );
  }
}

/*
 * Change log :
 * 
 * $Log: MenuBar.cpp,v $
 * Revision 1.2  2003/12/04 00:50:29  loetzsch
 * fixed VC7 related bug (bar was grayed)
 *
 * Revision 1.1  2003/10/07 10:10:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/08/09 14:53:10  dueffert
 * files and docu beautified
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.2  2003/05/11 23:29:26  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.1  2002/09/10 15:49:10  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.1  2002/09/01 17:19:20  loetzsch
 * tidied up the MfcTools/ directory. Created directories
 * MfcTools/DockingControlBars and MfcTools/IEStyleToolbars.
 *
 * Revision 1.2  2002/08/30 21:11:41  dueffert
 * AboutDlg added
 *
 * Revision 1.1.1.1  2002/05/10 12:40:25  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.2  2001/12/10 17:47:10  risler
 * change log added
 *
 */
