/**
* @file FrameWndEx.h
*
* Extended frame classes for MDI frames and children.
*
* Copyright (c) 2001 by Nikolay Denisov. All rights reserved.
*
* This code is free for personal and commercial use, providing this 
* notice remains intact in the source files and all eventual changes are
* clearly marked with comments.
*
* You must obtain the author's consent before you can include this code
* in a software library.
*
* No warrantee of any kind, express or implied, is included with this
* software; use at your own risk, responsibility for damages (if any) to
* anyone resulting from the use of this software rests entirely with the
* user.
*
* Please email bug reports, bug fixes, enhancements, requests and
* comments to: nick@actor.ru
*/

#ifndef __FRAMEWNDEX_H__
#define __FRAMEWNDEX_H__

#include "StdAfx.h"

#include "BmpMenuOwnerWnd.h"
#include "MenuBar.h"
#include "SizableReBar.h"
#include "WinAppEx.h"

// MenuBar window ID
#define IDW_MENUBAR                     0xE81A


/** Base for extended frame classes */
template< class TBase >
class CFrameWndBase : public CBmpMenuOwnerWnd< TBase >
{
// Constructors
public:
    CFrameWndBase();
    ~CFrameWndBase();

// Attributes
public:
    CSizableReBar   m_wndReBar;
    CMenuBar        m_wndMenuBar;

// Operations
public:
    void ShowWindow( int nCmdShow, LPCTSTR lpszProfileName );

    static LPCTSTR  m_lpszFrameWndPlacementEntry;
// Overrides
public:
    virtual BOOL LoadFrame( UINT nIDResource,
        DWORD dwDefaultStyle = WS_OVERLAPPEDWINDOW | FWS_ADDTOTITLE,
        CWnd* pParentWnd = 0, CCreateContext* pContext = 0 );

protected:

// Implementation data
protected:
    CString m_strProfileName;
    bool    m_bInPlace;

// Message map functions
protected:
    afx_msg int OnCreate( LPCREATESTRUCT lpCreateStruct );
    afx_msg void OnSysCommand( UINT nID, LPARAM lParam );
    afx_msg LRESULT OnMenuChar( UINT nChar, UINT nFlags, CMenu* pMenu );
    afx_msg BOOL OnNcActivate( BOOL bActive );
    afx_msg void OnSettingChange( UINT uFlags, LPCTSTR lpszSection );
    afx_msg void OnClose();
    afx_msg void OnInitMenuPopup( CMenu* pPopupMenu, UINT nIndex, BOOL bSysMenu );
    afx_msg void OnMenuSelect( UINT nItemID, UINT nFlags, HMENU hSysMenu );
};

/** An extended CFrameWnd class */
class CFrameWndEx : public CFrameWndBase< CFrameWnd >
{
    DECLARE_DYNCREATE( CFrameWndEx );

// Constructors
public:
    CFrameWndEx();
    ~CFrameWndEx();

// Overrides
protected:
    virtual void DelayUpdateFrameMenu( HMENU hMenuAlt );
    virtual void OnUpdateFrameMenu( HMENU hMenuAlt );

// Generated message map functions
protected:
    //{{AFX_MSG(CFrameWndEx)
    //}}AFX_MSG
    DECLARE_MESSAGE_MAP();
};

class CMDIFrameWndEx;

/** An extended CMDIClientWnd class */
class CMDIClientWndEx : public CWnd
{
    DECLARE_DYNCREATE( CMDIClientWndEx );

// Constructors
public:
    CMDIClientWndEx();
    ~CMDIClientWndEx();

// Implementation
protected:
    CMDIFrameWndEx* GetParentFrame() const;

// Implementation data
protected:
    HMENU m_hWindowMenu;

// Generated message map functions
protected:
    //{{AFX_MSG(CMDIClientWndEx)
    //}}AFX_MSG
    afx_msg LRESULT OnMDISetMenu( WPARAM wParam, LPARAM lParam );
    afx_msg LRESULT OnMDIRefreshMenu( WPARAM wParam, LPARAM lParam );
    DECLARE_MESSAGE_MAP();
};

/** An extended CMDIFrameWnd class */
class CMDIFrameWndEx : public CFrameWndBase< CMDIFrameWnd >
{
    DECLARE_DYNCREATE( CMDIFrameWndEx );

    friend CMDIClientWndEx;

// Constructors
public:
    CMDIFrameWndEx();
    ~CMDIFrameWndEx();

    void LoadBarState(LPCTSTR lpszProfileName);

// Overrides
protected:
    virtual void DelayUpdateFrameMenu( HMENU hMenuAlt );
    virtual void OnUpdateFrameTitle( BOOL bAddToTitle );
    
// Implementation data
protected:
    CMDIClientWndEx m_wndMDIClient;

// Generated message map functions
protected:
    //{{AFX_MSG(CMDIFrameWndEx)
    afx_msg int OnCreate( LPCREATESTRUCT lpCreateStruct );
    //}}AFX_MSG
    DECLARE_MESSAGE_MAP();
};

/** An extended CMDIChildWnd */
class CMDIChildWndEx : public CBmpMenuOwnerWnd< CMDIChildWnd >
{
    DECLARE_DYNCREATE( CMDIChildWndEx );

// Constructors
public:
    CMDIChildWndEx();
    ~CMDIChildWndEx();

// Generated message map functions
protected:
    //{{AFX_MSG(CMDIChildWndEx)
    //}}AFX_MSG
    DECLARE_MESSAGE_MAP();
};


template< class TBase >
LPCTSTR CFrameWndBase< TBase >::m_lpszFrameWndPlacementEntry = _T("FrameWndPlacement");

template< class TBase >
CFrameWndBase< TBase >::CFrameWndBase() : m_bInPlace( false )
{
}

template< class TBase >
CFrameWndBase< TBase >::~CFrameWndBase()
{
}

template< class TBase >
void CFrameWndBase< TBase >::ShowWindow( int nCmdShow, LPCTSTR lpszProfileName )
{
    m_strProfileName = lpszProfileName;
    if ( !m_strProfileName.IsEmpty() )
    {
        WINDOWPLACEMENT* lpwndpl;
        UINT nBytes;

        if ( AfxGetApp()->GetProfileBinary( m_strProfileName,
                m_lpszFrameWndPlacementEntry, ( LPBYTE* )&lpwndpl, &nBytes ) )
        {
            CRect rcDesktop;
            GetDesktopWindow()->GetWindowRect( rcDesktop );

            // Only restore if window intersects the screen
            CRect rcTemp;
            if ( IntersectRect( rcTemp, &lpwndpl->rcNormalPosition, rcDesktop ) )
            {
                nCmdShow = lpwndpl->showCmd;
                SetWindowPlacement( lpwndpl );
            }

            delete lpwndpl;
        }
    }

    CBmpMenuOwnerWnd< TBase >::ShowWindow( nCmdShow );
}

template< class TBase >
BOOL CFrameWndBase< TBase >::LoadFrame( UINT nIDResource, DWORD dwDefaultStyle,
                                        CWnd* pParentWnd, CCreateContext* pContext )
{
    if ( !TBase::LoadFrame( nIDResource, dwDefaultStyle, pParentWnd, pContext ) )
    {
        return FALSE;
    }

    VERIFY( m_wndMenuBar.SetMenu( ::GetMenu( m_hWnd ) ) );
    VERIFY( SetMenu( 0 ) );

    return TRUE;
}

template< class TBase >
int CFrameWndBase< TBase >::OnCreate( LPCREATESTRUCT lpCreateStruct )
{
    if ( CBmpMenuOwnerWnd< TBase >::OnCreate( lpCreateStruct ) == -1 )
    {
        return -1;
    }
    
    if ( !m_wndMenuBar.CreateEx( this,
            TBSTYLE_FLAT | TBSTYLE_LIST | TBSTYLE_CUSTOMERASE, // Toolbar styles
            WS_CHILD | WS_VISIBLE |                                             // Window styles
            CBRS_ALIGN_TOP | CBRS_TOOLTIPS | CBRS_FLYBY | CBRS_HIDE_INPLACE,    // Control bar styles
            CRect( 0, 0, 0, 0 ), IDW_MENUBAR ) )
    {
        return -1;
    }

    if ( !m_wndReBar.Create( this, AFX_IDW_CONTROLBAR_LAST - 1,
            WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | WS_CLIPCHILDREN |
            CBRS_TOP | CBRS_SIZE_DYNAMIC ) ||
         !m_wndReBar.AddBar( &m_wndMenuBar, 0, 0,
            RBBS_GRIPPERALWAYS | RBBS_FIXEDBMP ) )
    {
        return -1;
    }

    m_wndMenuBar.SetOwner( &m_wndReBar );
    m_wndMenuBar.UpdateMenuBar();

    return 0;
}

template< class TBase >
void CFrameWndBase< TBase >::OnSysCommand( UINT nID, LPARAM lParam )
{
    if ( !m_wndMenuBar.FrameOnSysCommand( nID, lParam ) )
    {
        CBmpMenuOwnerWnd< TBase >::OnSysCommand( nID, lParam );
    }
}

template< class TBase >
LRESULT CFrameWndBase< TBase >::OnMenuChar( UINT nChar, UINT nFlags, CMenu* pMenu )
{
    if ( m_wndMenuBar.FrameOnMenuChar( nChar, nFlags, pMenu ) )
    {
        return MAKELRESULT( 0, 1 );
    }

    return CBmpMenuOwnerWnd< TBase >::OnMenuChar( nChar, nFlags, pMenu );
}

template< class TBase >
BOOL CFrameWndBase< TBase >::OnNcActivate( BOOL bActive )
{
    CFrameWnd* pFrame = GetTopLevelFrame();
    ASSERT_VALID( pFrame );

    m_wndMenuBar.FrameOnNcActivate(
        ( bActive || ( pFrame->m_nFlags & WF_STAYACTIVE ) ) && pFrame->IsWindowEnabled() );

    return CBmpMenuOwnerWnd< TBase >::OnNcActivate( bActive );
}

template< class TBase >
void CFrameWndBase< TBase >::OnSettingChange( UINT uFlags, LPCTSTR lpszSection )
{
    CWinAppEx::GetInstance()->UpdateSystemParameters();    // reload cached metrics

    CBmpMenuOwnerWnd< TBase >::OnSettingChange( uFlags, lpszSection );
}

template< class TBase >
void CFrameWndBase< TBase >::OnClose()
{
    if ( m_lpfnCloseProc == 0 )
    {
        if ( !m_strProfileName.IsEmpty() )
        {
            WINDOWPLACEMENT wndpl;
            wndpl.length = sizeof( wndpl );

            if ( GetWindowPlacement( &wndpl ) )
            {
                AfxGetApp()->WriteProfileBinary( m_strProfileName,
                    m_lpszFrameWndPlacementEntry, ( LPBYTE )&wndpl, wndpl.length );
            }
        }
    }

    CBmpMenuOwnerWnd< TBase >::OnClose();
}

template< class TBase >
void CFrameWndBase< TBase >::OnInitMenuPopup( CMenu* pPopupMenu, UINT nIndex, BOOL bSysMenu )
{
    CBmpMenuOwnerWnd< TBase >::OnInitMenuPopup( pPopupMenu, nIndex, bSysMenu );

    m_wndMenuBar.FrameOnInitMenuPopup( pPopupMenu, nIndex, bSysMenu );
}

template< class TBase >
void CFrameWndBase< TBase >::OnMenuSelect( UINT nItemID, UINT nFlags, HMENU hSysMenu )
{
    CBmpMenuOwnerWnd< TBase >::OnMenuSelect( nItemID, nFlags, hSysMenu );

    m_wndMenuBar.FrameOnMenuSelect( nItemID, nFlags, hSysMenu );
}

/////////////////////////////////////////////////////////////////////////////
#endif  // !__FRAMEWNDEX_H__

/*
 * Change log :
 * 
 * $Log: FrameWndEx.h,v $
 * Revision 1.1  2003/10/07 10:10:08  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.2  2003/05/11 23:29:26  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.1  2002/09/10 15:49:10  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.1  2002/09/01 17:19:20  loetzsch
 * tidied up the MfcTools/ directory. Created directories
 * MfcTools/DockingControlBars and MfcTools/IEStyleToolbars.
 *
 * Revision 1.4  2002/08/30 18:12:43  dueffert
 * further docu bugs fixed
 *
 * Revision 1.3  2002/08/30 17:17:19  dueffert
 * doxygen parser bugs fixed, docu added
 *
 * Revision 1.2  2002/08/13 13:00:04  loetzsch
 * workaround for a bug in CFrameWnd
 * RobotControl doesn't crash any more if a tool bar or dialog bar was removed.
 *
 * Revision 1.1.1.1  2002/05/10 12:40:25  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.2  2001/12/10 17:47:10  risler
 * change log added
 *
 */
