/**
 *  @file HSIDataTypes.h
 *  Declaration of class HSIDataTypes.
 *  @author<A href=mailto:kspiess@informatik.uni-bremen.de>Kai Spiess</A>
 */
#ifndef __HSIDataTypes_h
#define __HSIDataTypes_h

#include "Tools/Streams/InOut.h"


/** This class represents an H, S and I value ranges in HSI color space. */
class HSIColorRange
{
public:
  /** The maximum value for H in this range */
  short int H_max;
  /** The minimum value for H in this range */
  short int H_min;
  /** The maximum value for S in this range */
  short int S_max;
  /** The minimum value for S in this range */
  short int S_min;
  /** The maximum value for I in this range */
  short int I_max;
  /** The minimum value for I in this range */
  short int I_min;
};


/** The number of color classes used for segmentation */
enum{NUMBER_OF_COLOR_CLASSES = 10};

/** This class represents an colortable for the HSI color space. */
class HSIColorTable
{
public:
  /** The initial values for the H, S, and I ranges are set. */
	void init();
  /** The color classes of the HSI colortable */
  HSIColorRange colorClasses[NUMBER_OF_COLOR_CLASSES];
};

/** This class represents a stucture needed to save undo informations when editing 
 *  an HSI color rang undo informations.
 */
class HSIColorRangeUndo
{
public:
  /** Sets the undo information for a HSI color range and the belonging color class.
   *  @param hsiColorRange The HSI color range before editing.
   *  @param colorClass The index of the color class to be edited.
   */
	void setHSIColorRangeUndo(HSIColorRange hsiColorRange, int colorClass);
  /** The edited HSI color range */
  HSIColorRange hsiColorRange;
  /** The edited color class */
  int colorClass;
  /** If this undo is set */
  bool set;
  /** Constructor */
  HSIColorRangeUndo();
};

/**
 * Streaming operator that reads a HSIColorRange from a stream.
 * @param stream The stream from which is read.
 * @param hsiColorRange The HSIColorRange object.
 * @return The stream.
 */ 
In& operator>>(In& stream,HSIColorRange& hsiColorRange);

/**
 * Streaming operator that writes a HSIColorRange to a stream.
 * @param stream The stream to write on.
 * @param hsiColorRange The HSIColorRange object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const HSIColorRange& hsiColorRange);

/**
 * Streaming operator that reads a HSIColorTable from a stream.
 * @param stream The stream from which is read.
 * @param hsiColorTable The HSIColorTable object.
 * @return The stream.
 */ 
In& operator>>(In& stream,HSIColorTable& hsiColorTable);
 
/**
 * Streaming operator that writes a HSIColorTable to a stream.
 * @param stream The stream to write on.
 * @param hsiColorTable The HSIColorTable object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const HSIColorTable& hsiColorTable);

/**
 * Streaming operator that reads a HSIColorRangeUndo from a stream.
 * @param stream The stream from which is read.
 * @param hsiColorRangeUndo The HSIColorRangeUndo object.
 * @return The stream.
 */ 
In& operator>>(In& stream,HSIColorRangeUndo& hsiColorRangeUndo);

/**
 * Streaming operator that writes a HSIColorRangeUndo to a stream.
 * @param stream The stream to write on.
 * @param hsiColorRangeUndo The HSIColorRangeUndo object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const HSIColorRangeUndo& hsiColorRangeUndo);

#endif //__HSIDataTypes_h

/*
 * Change log :
 * 
 * $Log: HSIDataTypes.h,v $
 * Revision 1.3  2004/05/12 20:38:14  roefer
 * Unused lines of code removed
 *
 * Revision 1.2  2004/01/17 17:07:41  roefer
 * HSI-Toolbar does not crash anymore, but image size is still 176x144
 *
 * Revision 1.1  2003/10/07 10:10:07  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:26  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.1  2002/09/10 15:49:09  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.2  2002/07/23 13:45:44  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - 5 instead of 3 debug queues in RobotControl
 * - new debug message handling
 * - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
 * - general clean up
 *
 * Revision 1.1.1.1  2002/05/10 12:40:25  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.2  2002/04/05 08:17:00  kspiess
 * gleichzeitiges Speichern von HSI und C64 Farbtabelle eingebaut;
 * Fehler beim Anklicken einzelner Pixel im Zoom-Dialog zur Verschiebung
 * der HSI-Grenzen behoben;
 * Fehler beim mehrstufigen Undo behoben
 *
 * Revision 1.1  2002/02/08 14:22:56  kspiess
 * DatenTypen und Kern fr das  HSIColorTableTool
 *
 *
 */
