/**
* @file WLANToolBar.cpp
* 
* Implementation of WLAN toolbar in the RobotControl project.
*
* @author Max Risler, Uwe Dffert
*/

#include "StdAfx.h"

#include "WLANToolBar.h"
#include "Dialogs/WLANConfigurationDialog.h"
#include "Tools/Player.h"
#include "Dialogs/GenericModalBinary.h"

#include "RobotControlQueues.h"
#include "RobotControlPhysicalRobots.h"

IMPLEMENT_DYNAMIC( CWLANToolBar, CToolBarEx )

CWLANToolBar::CWLANToolBar():numOfConfigs(0)
{
}

CWLANToolBar::~CWLANToolBar()
{
}

void CWLANToolBar::Init()
{
  static TBBUTTONEX tbButtons[] =
  {
    { {  0, IDC_WLAN_COMBO, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  0, IDC_WLAN_ADD, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  1, IDC_WLAN_EDIT, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  2, IDC_WLAN_DELETE, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  0, 0, 0, TBSTYLE_SEP, 0,0 }, true },
    { {  3, IDC_WLAN_CONNECT, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  4, IDC_WLAN_DISCONNECT, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  5, IDC_WLAN_RED1, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  6, IDC_WLAN_RED2, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  7, IDC_WLAN_RED3, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  8, IDC_WLAN_RED4, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { {  9, IDC_WLAN_BLUE1, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { { 10, IDC_WLAN_BLUE2, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { { 11, IDC_WLAN_BLUE3, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true },
    { { 12, IDC_WLAN_BLUE4, TBSTATE_ENABLED, TBSTYLE_BUTTON,    0, 0 }, true }
  };
  
  SetBitmaps( IDB_WLAN, IDB_WLAN_COLD);
  SetButtons( sizeof( tbButtons ) / sizeof( tbButtons[ 0 ] ), tbButtons);
  AddCombo(&m_wndWLANCombo,IDC_WLAN_COMBO,120);
  
  ReadConfigs();
  m_wndWLANCombo.SetCurSel(AfxGetApp()->GetProfileInt("WLAN","selectedConnection",0));
  changeConfiguration();
}

bool CWLANToolBar::HasButtonText( UINT nID )
{ 
  switch (nID)
  {
  case IDC_WLAN_EDIT:
  case IDC_WLAN_CONNECT:
  case IDC_WLAN_DISCONNECT:
    return true;
  default:
    return false;
  }
}

bool CWLANToolBar::handleCommand(UINT command)
{
  switch(command)
  {
  case IDC_WLAN_CONNECT:
    OnConnectAll(true);
    return true;
  case IDC_WLAN_DISCONNECT:
    OnConnectAll(false);
    return true;
  case IDC_WLAN_RED1:
    OnSelect(true,1); 
    return true;
  case IDC_WLAN_RED2:
    OnSelect(true,2); 
    return true;
  case IDC_WLAN_RED3:
    OnSelect(true,3); 
    return true;
  case IDC_WLAN_RED4:
    OnSelect(true,4); 
    return true;
  case IDC_WLAN_BLUE1:
    OnSelect(false,1); 
    return true;
  case IDC_WLAN_BLUE2:
    OnSelect(false,2); 
    return true;
  case IDC_WLAN_BLUE3:
    OnSelect(false,3); 
    return true;
  case IDC_WLAN_BLUE4:
    OnSelect(false,4); 
    return true;
  case IDC_WLAN_ADD:
    OnAdd(); 
    return true;
  case IDC_WLAN_EDIT:
    OnEdit(); 
    return true;
  case IDC_WLAN_DELETE:
    OnDelete(); 
    return true;
  default:
    return false;
  }
}

bool CWLANToolBar::handleSelChange(UINT nID)
{
  switch(nID)
  {
  case IDC_WLAN_COMBO:
    changeConfiguration();
    return true;
  default:
    return false;
  }
}

void CWLANToolBar::updateUI(CCmdUI* pCmdUI)
{
  switch(pCmdUI->m_nID)
  {
  case IDC_WLAN_COMBO:
  case IDC_WLAN_ADD:
  case IDC_WLAN_EDIT:
  case IDC_WLAN_DELETE:
  case IDC_WLAN_CONNECT:
    pCmdUI->Enable(!getPhysicalRobots().isSomeoneConnected());
    pCmdUI->SetCheck(false);
    return;
  case IDC_WLAN_DISCONNECT:
    pCmdUI->Enable(getPhysicalRobots().isSomeoneConnected());
    pCmdUI->SetCheck(false);
    return;
  case IDC_WLAN_RED1:
  case IDC_WLAN_RED2:
  case IDC_WLAN_RED3:
  case IDC_WLAN_RED4:
  case IDC_WLAN_BLUE1:
  case IDC_WLAN_BLUE2:
  case IDC_WLAN_BLUE3:
  case IDC_WLAN_BLUE4:
    {
      bool connected=getPhysicalRobots().isConnected(pCmdUI->m_nID - IDC_WLAN_RED1);
      pCmdUI->Enable(connected);
      pCmdUI->SetCheck(connected && (getPhysicalRobots().getSelectedRobot()==(int)(pCmdUI->m_nID - (int)IDC_WLAN_RED1)));
      return;
    }
  default:
    return;
  }
}

void CWLANToolBar::setImages()
{
  int cursel=m_wndWLANCombo.GetCurSel();
  for (int i=0;i<4;i++)
  {
    if (wlanConfig[cursel].useRed[i])
    {
      SetImage(IDC_WLAN_RED1+i,5+i);
    }
    else
    {
      SetImage(IDC_WLAN_RED1+i,13+i);
    }
    if (wlanConfig[cursel].useBlue[i])
    {
      SetImage(IDC_WLAN_BLUE1+i,9+i);
    }
    else
    {
      SetImage(IDC_WLAN_BLUE1+i,17+i);
    }
  }
}

void CWLANToolBar::OnSelect(bool red, int number)
{
  CRobotControlPhysicalRobots* phys = &getPhysicalRobots();
  int newSel=red?number-1:number+3;
  if (phys->getSelectedRobot()==newSel)
  {
    phys->setSelectedRobot(-1);
  }
  else
  {
    phys->setSelectedRobot(newSel);
  }
}

void CWLANToolBar::OnConnectAll(bool connect)
{
  CRobotControlPhysicalRobots* phys=&getPhysicalRobots();
  int cursel=m_wndWLANCombo.GetCurSel();
  if (connect)
  {
    phys->connect(wlanConfig[cursel]);
  }
  else
  {
    phys->disconnect();
  }
}

void CWLANToolBar::OnAdd()
{
  if (numOfConfigs==wlanToolBarMaxNumOfConfigs) return;
  CWLANConfigurationDialog newDialog;
  newDialog.setWLANConfiguration(&wlanConfig[numOfConfigs]);
  
  if (newDialog.DoModal()!=IDOK) return;
  
  m_wndWLANCombo.AddString(wlanConfig[numOfConfigs].name);
  m_wndWLANCombo.SetCurSel(numOfConfigs);
  
  numOfConfigs++;
  
  WriteConfigs();
  changeConfiguration();
}

void CWLANToolBar::changeConfiguration()
{
  int cursel=m_wndWLANCombo.GetCurSel();
  AfxGetApp()->WriteProfileInt("WLAN","selectedConnection",cursel);
  setImages();
}

void CWLANToolBar::OnEdit()
{
  if (numOfConfigs==0) return;
  CWLANConfigurationDialog newDialog;
  int index=m_wndWLANCombo.GetCurSel();
  if (index>=0)
  {
    newDialog.setWLANConfiguration(&wlanConfig[index]);
    if (newDialog.DoModal()!=IDOK) return;
    
    m_wndWLANCombo.DeleteString(index);
    m_wndWLANCombo.InsertString(index,wlanConfig[index].name);
    m_wndWLANCombo.SetCurSel(index);
    
    WriteConfigs(index);
    changeConfiguration();
  }
}

void CWLANToolBar::OnDelete()
{
  if (numOfConfigs==0) return;
  
  int index = m_wndWLANCombo.GetCurSel();
    
  CString message; 
  message.Format("Please onfirm permanent removal of WLAN setting: \n\n        \"%s\"\n\nDelete?", wlanConfig[index].name);
  CGenericModalBinary deleteDialog("Delete", "Cancel", message);
  if (IDCANCEL == deleteDialog.DoModal())
    return;

  numOfConfigs--;
  for (int i=index;i<numOfConfigs;i++)
  {
    wlanConfig[i]=wlanConfig[i+1];
  }
  m_wndWLANCombo.DeleteString(index);
  m_wndWLANCombo.SetCurSel((index>=numOfConfigs)?numOfConfigs-1:index);
  
  WriteConfigs();
  changeConfiguration();
}

void CWLANToolBar::ReadConfigs()
{
  numOfConfigs = AfxGetApp()->GetProfileInt("WLAN","numOfConnections",0);
  for (int i=0;i<numOfConfigs;i++)
  {
    wlanConfig[i].readFromProfile(i);
    m_wndWLANCombo.AddString(wlanConfig[i].name);
  }
  m_wndWLANCombo.SetCurSel(0);
}

void CWLANToolBar::WriteConfigs(int index)
{
  if (index<0)
  {
    AfxGetApp()->WriteProfileInt("WLAN","numOfConnections",numOfConfigs);
    for (int i=0;i<numOfConfigs;i++)
    {
      wlanConfig[i].writeToProfile(i);
    }
    wlanConfig[0].deleteProfile(numOfConfigs);
  }
  else
  {
    wlanConfig[index].writeToProfile(index);
  }
}


/*
* Change log :
* 
* $Log: WLANToolBar.cpp,v $
* Revision 1.5  2004/02/23 13:27:25  jhoffman
* - changes to Main-menu are shown ... this currently results in one additional menu (!) somebody please fix this!
* - added GenericModalBinary-dialog: can be used to create a modal dialog where you can specify the button labels and the descriptive text (like APX...ModalDialog but you can specify what is written on the buttons)
* - CameraToolBar: mode, shutter, gain extracted from dropdown
* - ToolBarEx: added AddString method to write const. strings on a toolbar
*
* Revision 1.4  2004/01/29 12:03:24  jhoffman
* UI changes:
* - Main menu entries renamed, reordered, added some new icons
* - delete-confirmation in settings and WLAN dialog
* - settings dialog uses icons, radio buttons added
* - WLAN dialog restructured
* - motion tester and head motion tested layout improved, stop button added
* - check boxes replaced by radio buttons where appropriate
*
* Revision 1.3  2003/12/14 14:24:34  loetzsch
* renamed WLANNewDialog to WLANConfigurationDialog
*
* Revision 1.2  2003/12/06 06:31:20  loetzsch
* no message
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.11  2003/05/11 23:46:36  dueffert
* Depend now works with RobotControl too
*
* Revision 1.10  2003/03/28 14:05:53  dueffert
* team wlan support improved
*
* Revision 1.9  2003/03/25 11:46:53  dueffert
* team wlan support improved
*
* Revision 1.8  2003/03/22 22:37:52  loetzsch
* finished GUI and message routing for 8 simulated robots
* almost all functionallity for simulated robots now is encapsulated in class
*   CRobotControlSimulatedRobots
*
* Revision 1.7  2003/03/20 23:42:17  loetzsch
* changed cast of AfxGetApp() to CRobotControlApp*
* to getRobotControlApp()
*
* Revision 1.6  2003/03/19 09:11:19  dueffert
* team wlan support improved
*
* Revision 1.5  2003/03/13 16:39:47  dueffert
* wlan toolbar can handle teams now
*
* Revision 1.4  2003/01/28 17:18:40  loetzsch
* encapsulated the WLan thread into an own class
*
* Revision 1.3  2002/11/20 13:37:30  dueffert
* missing or wrong doxygen file corrected
*
* Revision 1.2  2002/09/22 13:46:09  dueffert
* direct wlan removed (still in code as comments)
*
* Revision 1.1  2002/09/10 15:49:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.4  2002/07/23 13:43:37  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - empty constructors in bars / dialogs
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
* Revision 1.3  2002/06/20 23:38:47  dueffert
* player toolbar & wlan toolbar completed
*
* Revision 1.2  2002/06/08 12:57:40  dueffert
* further wlan config functionality added
*
* Revision 1.1.1.1  2002/05/10 12:40:21  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.11  2002/04/24 14:44:41  dueffert
* WLANToolbar saves actual setting in registry
*
* Revision 1.10  2002/04/23 17:45:15  loetzsch
* - splitted debugKeyTable into debugKeyTableForRobot and debugKeyTableForLocalProcesses
* - removed Modules Toolbar
* - removed access for dialogs and toolbars to solutionRequest.
* - changed access for dialogs and toolbars to debug queues
* - removed the instance of SolutionRequest in CRobotControlApp
* - only the log player, local processes and the robot put messages into queueToRobotControl
*
* Revision 1.9  2002/04/21 16:07:44  roefer
* Only "Auto" uses team color and player role to modify ip address/router port
*
* Revision 1.8  2002/04/16 15:45:44  dueffert
* BaseIP for team can be set in PlayersToolbar and is used in WLANToolbar
*
* Revision 1.2  2002/04/13 15:57:39  dueffert
* player - wlan synchronisation completed
*
* Revision 1.7  2002/02/27 12:10:37  dueffert
* minor direct wlan enhancements
*
* Revision 1.6  2002/02/21 15:58:26  dueffert
* direct wlan works! at the moment you can only send 3 pakets (<32K each) and receive any number of pakets (<32K each)
*
* Revision 1.5  2002/02/20 16:08:12  dueffert
* direct wlan connection added, not fully functional yet
*
* Revision 1.4  2002/02/16 20:11:56  dueffert
* direct wlan communication (without Router) added, integration not yet completed
*
* Revision 1.3  2002/01/25 14:54:11  risler
* no message
*
* Revision 1.2  2002/01/24 22:57:22  risler
* finished WLAN Toolbar
*
* Revision 1.1  2002/01/24 13:16:03  risler
* added WLAN Toolbar
*
*
*/
