/**
* @file ValueHistoryDlgBar.h
*
* Declaration of class CValueHistoryDlgBar
*
* @author Matthias Jngel
* @author Martin Ltzsch
*/

#ifndef __ValueHistoryDlgBar_h_
#define __ValueHistoryDlgBar_h_

#include "StdAfx.h"
#include "RobotControlDialogBar.h"
#include "Tools/RingBuffer.h"
#include "Visualization/PaintMethodsWin32.h"
/**
* @class CValueHistoryDlgBar 
* 
* A dialog bar to display certain values over time
*/
class CValueHistoryDlgBar : public CRobotControlDialogBar
{
public:
  /** IDs for values to display */
  enum valueID
  {
    cameraMatrixIsValid,
    ballPerceptX,
    ballPerceptY,      
    ballPerceptDistance,
    ballPerceptAngle,
    ballPerceptDistanceBearingBased,
    ballPerceptAngleBearingBased,
    ballPerceptDistanceSizeBased,
    ballPerceptAngleSizeBased,
    ballPositionDistance,
    ballPositionAngle,
    ballPositionSpeedAbs,
    ballPositionSpeedDir,
    specialPerceptRobotPoseX,
    specialPerceptRobotPoseY,
    specialPerceptRobotPoseAngle,
    psdPerceptToFarAway,
    psdPerceptX,
    psdPerceptY,
    psdPerceptZ,
    robotPoseX,
    robotPoseY,
    robotPoseAngle,
    robotPoseSpeed,
    walkSpeedX,
    walkSpeedY,
    walkSpeedRotation,
    numberOfValueIDs
  };

  /** returns the name for an id */
  static const char* getValueIDName(valueID id)
  {
    switch (id)
    {
    case cameraMatrixIsValid: return "cameraMatrixIsValid";
    case ballPerceptX: return "ballPercept.x";
    case ballPerceptY: return "ballPercept.y";
    case ballPerceptDistance: return "ballPercept.distance";
    case ballPerceptAngle: return "ballPercept.angle";
    case ballPerceptDistanceBearingBased: return "ballPercept.distanceBearingBased";
    case ballPerceptAngleBearingBased: return "ballPercept.angleBearingBased";
    case ballPerceptDistanceSizeBased: return "ballPercept.distanceSizeBased";
    case ballPerceptAngleSizeBased: return "ballPercept.angleSizeBased";
    case ballPositionDistance: return "ballPosition.distance";
    case ballPositionAngle: return "ballPosition.angle";
    case ballPositionSpeedAbs: return "ballPosition.speedAbs";
    case ballPositionSpeedDir: return "ballPosition.speedDir";
    case psdPerceptToFarAway: return "psdPerceptToFarAway";
    case psdPerceptX: return "psdPercept.x";
    case psdPerceptY: return "psdPercept.y";
    case psdPerceptZ: return "psdPercept.z";
    case robotPoseX: return "robotPose.x";
    case robotPoseY: return "robotPose.y";
    case robotPoseAngle: return "robotPose.angle";
    case robotPoseSpeed: return "robotPose.speed";
    case specialPerceptRobotPoseX: return "specialPerceptRobotPose.x";
    case specialPerceptRobotPoseY: return "specialPerceptRobotPose.y";
    case specialPerceptRobotPoseAngle: return "specialPerceptRobotPose.angle";
    case walkSpeedX: return "walkSpeedX";
    case walkSpeedY: return "walkSpeedY";
    case walkSpeedRotation:return "walkSpeedRotation";

    default: return "unknown";
    }
  }

  /** Default constructor. */
  CValueHistoryDlgBar();

  /** Destructor */
	~CValueHistoryDlgBar();   
  
  /** 
  * Called from a MessageQueue to distribute messages 
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message);


  /** 
  * Adds a new value to display
  * @param id the id of the value
  * @param timeStamp the time when the value was generated
  * @param value the value
  * @param halfSiz specifies whether the value shall be painted half as large as usual
  */
  void addValue(valueID id, unsigned long timeStamp, double value, bool halfSiz=false);

	//{{AFX_DATA(CValueHistoryDlgBar)
	enum { IDD = IDD_DIALOG_BAR_VALUE_HISTORY };
	CSliderCtrl	m_scaleSlider;
	CSliderCtrl	m_timeRangeSlider;
  CStatic m_timeRangeStatic;
  CButton m_freezeButton;
	//}}AFX_DATA


	//{{AFX_VIRTUAL(CValueHistoryDlgBar)
	protected:
	virtual void DoDataExchange(CDataExchange* pDX);    
  virtual BOOL OnInitDialog();
	//}}AFX_VIRTUAL

protected:

	//{{AFX_MSG(CValueHistoryDlgBar)
	afx_msg void OnPaint();
	afx_msg void OnSize(UINT nType, int cx, int cy);
	afx_msg void OnContextMenu(CWnd* pWnd, CPoint point);
	afx_msg void OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
  afx_msg void OnFreezeButton();
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP();
private:

  //display ////////
  CRect rect;
  CDC dcOffScreen;
  CBitmap* bmpOffScreen;
  CBitmap* oldBitmap;

  /** saves all data to a CSV (comma separated values) file (separate time lines for 
  * all data */
  void saveAllCSV();



protected:
  /** the range of time that is displayed in the dialog (in seconds ) */
  int timeRange;

  /** zoom factor for the values displayed in the dialog */
  double scale;

  /** The last received time stamp */
  unsigned long latestTimeStamp;

  /** The maximum numbers of entries that can be stored per value */
  enum {maxNumberOfEntries=1000};

  /** The valueHistory for each value */
  RingBuffer<double,maxNumberOfEntries> valueHistory[numberOfValueIDs];

  /** The time stamp history for each value */
  RingBuffer<unsigned long,maxNumberOfEntries> timeStamps[numberOfValueIDs];
  
  /** The half size history for each value */
  RingBuffer<bool,maxNumberOfEntries> halfSize[numberOfValueIDs];

  /** If the values shall be dislplayed */
  bool displayValue[numberOfValueIDs];

  /** The paint style for each value */
  PaintMethodsWin32::PaintStyle paintStyles[numberOfValueIDs];

  /** Paints a value to a given position */
  void paintValue(int x, int y, int size, PaintMethodsWin32::PaintStyle style);

  /** if true, no new values are added */
  bool freeze;

  /** called to delete every pieces of data added so far */
  void clear();
};

//{{AFX_INSERT_LOCATION}}

#endif //__ValueHistoryDlgBar_h_

/*
 * Change log :
 * 
 * $Log: ValueHistoryDlgBar.h,v $
 * Revision 1.15  2004/04/22 09:41:52  goehring
 * Added BallSpeedDirection
 *
 * Revision 1.14  2004/04/21 20:13:14  goehring
 * Added slider
 *
 * Revision 1.13  2004/03/16 14:00:23  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.2  2004/03/09 18:42:10  juengel
 * Moved PaintStyle to PaintMethodsWin32
 *
 * Revision 1.12  2004/03/05 15:49:09  dueffert
 * robotPoseSpeed added
 *
 * Revision 1.11  2004/02/25 13:34:33  dueffert
 * half size marks for low validity added
 *
 * Revision 1.10  2004/02/05 21:36:26  goehring
 * Bug removed
 *
 * Revision 1.9  2004/01/28 08:29:40  dueffert
 * changed enum order to match data flow in modules
 *
 * Revision 1.8  2003/12/06 16:44:14  loetzsch
 * renamed notify() to addValue()
 * messages are now handled inside the dialog
 *
 * Revision 1.7  2003/11/24 15:25:00  dueffert
 * resetting corrected
 *
 * Revision 1.6  2003/11/18 13:24:42  goehring
 * Different BallDistance measures notified
 *
 * Revision 1.5  2003/11/10 13:32:26  dueffert
 * special percept visualization added
 *
 * Revision 1.4  2003/11/06 16:42:59  goehring
 * CSV button added (BETA)
 *
 * Revision 1.3  2003/11/05 13:41:25  goehring
 * JointStates removed
 *
 * Revision 1.2  2003/10/21 14:40:37  goehring
 * ballPerceptX and Y available
 *
 * Revision 1.1  2003/10/07 10:09:38  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/07/06 00:44:24  loetzsch
 * added option playing-goalie2
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.9  2003/06/03 09:26:41  goehring
 * Review for CollisionDetectorValues
 *
 * Revision 1.8  2003/05/11 23:46:36  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.7  2003/04/06 20:34:22  juengel
 * CameraMatrix.isValid is displayed in ValueHistoryDlgBar.
 *
 * Revision 1.6  2003/04/06 16:40:03  juengel
 * Added PSDPercept.tooFarAway to ValueHistoryDlgBar.
 *
 * Revision 1.5  2003/03/28 14:28:08  juengel
 * Added PSDPercept
 *
 * Revision 1.4  2003/03/07 19:04:07  juengel
 * Added style horizontalLine, and values for motionRequest, and green level
 *
 * Revision 1.3  2003/03/05 12:16:19  loetzsch
 * added the "freeze" button to the value history dialog bar
 *
 * Revision 1.2  2003/03/03 23:08:01  loetzsch
 * first working version of the value history dialog bar
 *
 * Revision 1.1  2003/02/27 15:30:47  loetzsch
 * added the value history dialog
 *
 */
