#include "MotionOptimizerRobot.h"

#include "Platform/SystemCall.h"
#include "RobotControlQueues.h"

/////////////////////////////////////////////////////////////////////////////
// Motion Optimizer Physical Robot

MotionOptimizerRobot::MotionOptimizerRobot ( MotionOptimizerController* dlg, int robotId, bool isPhysical, int listItem )
{
  dialog = dlg;
  id = robotId;
  this->physical = isPhysical;
  item = listItem;
  messages = 0;
  paramSetId = 0;
  state = MotionRatingBehaviorMessage::unknown;
  realState = MotionRatingBehaviorMessage::unknown;
  currSpeed = 0.0;
  avgSpeed = 0.0;
}

bool MotionOptimizerRobot::handleMessage ( MotionRatingBehaviorMessage& message )
{
  messages++;
  lastMessage = SystemCall::getCurrentSystemTime();

  battery = message.remainingPower;
  paramSetId = message.paramSetId;
  currentWalk = message.currentWalk;
  walkCount = message.walkCount;

  if ( realState == message.state ) state = message.state;
  realState = message.state;

  if ( message.type == MotionRatingBehaviorMessage::report )
  {
    currSpeed = message.currentSpeed;
    avgSpeed = message.averageSpeed;

    if ( message.finished ) 
    {
      char str[50];
      sprintf ( str, "robot %d finished test %d (speed: %.2fcm/s)\r\n", id, message.paramSetId, message.averageSpeed * 100.0 );
      dialog->printMessage ( str );
      dialog->setResult ( message.paramSetId, avgSpeed );
    }
  }
  dialog->updateList ( this );
  return true;
}

bool MotionOptimizerRobot::notify ()
{
  bool dead = false;
  if ( SystemCall::getTimeSince ( lastMessage ) > 10000 ) dead = true;
  if ( dead ) dialog->printMessage ( "robot dead\r\n" );
  return ! dead;
}

bool MotionOptimizerRobot::init ( int path ) 
{
  MotionRatingBehaviorMessage message;
  message.type = MotionRatingBehaviorMessage::init;
  message.data = path;

  sendMessage ( message );

  return true;
}

bool MotionOptimizerRobot::test ( int paramSetId, InvKinWalkingParameters& paramSet ) 
{
  MessageQueue& queue = getQueueToRobot();
  queue.out.bin << paramSet;
  queue.out.finishMessage ( idInvKinWalkingParameters );
  
  MotionRatingBehaviorMessage message;
  message.type = MotionRatingBehaviorMessage::request;
  message.data = paramSetId;

  sendMessage ( message );

  char str[50];
  sprintf ( str, "robot %d started test %d\r\n", id, paramSetId );
  dialog->printMessage ( str );

  return true;
}

MessageQueue& MotionOptimizerRobot::getQueueToRobot ( )
{
  if ( physical ) return getQueues().toPhysical.robot[id];
  else return getQueues().toSimulated.robot[id];
}

void MotionOptimizerRobot::sendMessage ( MotionRatingBehaviorMessage& message )
{
  MessageQueue& queue = getQueueToRobot();
  queue.out.bin << message;
  queue.out.finishMessage ( idMotionRatingBehaviorMessage );

  state = MotionRatingBehaviorMessage::unknown;
  realState = state;
}
