/**
* @file KickEditorDlgBar.h
*
* Declaration of class CKickEditorDlgBar
*
* @author Matthias Jngel
*/

#ifndef __KickEditorDlgBar_h_
#define __KickEditorDlgBar_h_

#include "StdAfx.h"
#include "RobotControlDialogBar.h"

#include "Tools/KickSelectionTable.h"
#include "Visualization/PaintMethodsWin32.h"
#include "Representations/Motion/MotionRequest.h"

/**
* Represents the results and the preconditions of a single kick experiment
* @author Martin Ltzsch
*/
class KickCase
{
public:
  /** The relative x position of the ball when the kick was requested */
  double ballXR;

  /** The relative y position of the ball when the kick was requested */
  double ballYR;

  /** The time since the ball was seen last when the kick was requested */
  unsigned long ballTR;

  /** The relative x position of the ball when the kick was started */
  double ballXS;

  /** The relative y position of the ball when the kick was started */
  double ballYS;

  /** The time since the ball was seen last when the kick was started */
  unsigned long ballTS;

  /** The relative x position of the ball after the kick */
  double ballXA;

  /** The relative y position of the ball after the kick */
  double ballYA;

  /** The time since the kick was finished when the record was written */
  unsigned long tA;

};

/**
* @class CKickEditorDlgBar 
* 
* A dialog bar for creating the data base for the kick selector
*
* @author Matthias Jngel
*/
class CKickEditorDlgBar : public CRobotControlDialogBar
{
public:
  /** Default constructor.*/
  CKickEditorDlgBar();

  /** Destructor */
  ~CKickEditorDlgBar();

  /** Returns an action id for a special action id */
  KickSelectionTable::ActionID getActionIDFromSpecialActionID(MotionRequest::SpecialActionID id)
  {
    switch (id)
    {
    // new athErs7-Kicks
    case MotionRequest::athErs7ChestKick2: return KickSelectionTable::athErs7ChestKick2;
    case MotionRequest::athErs7fastDiveKick: return KickSelectionTable::athErs7fastDiveKick;
    case MotionRequest::athErs7ForwardLeapingKick: return KickSelectionTable::athErs7ForwardLeapingKick;
    case MotionRequest::athErs7HeadKickLeft: return KickSelectionTable::athErs7HeadKickLeft;
    case MotionRequest::athErs7HeadKickRight: return KickSelectionTable::athErs7HeadKickRight;
    case MotionRequest::athErs7KickForwardWithRightPaw: return KickSelectionTable::athErs7KickForwardWithRightPaw;
    case MotionRequest::athErs7KickForwardWithLeftPaw: return KickSelectionTable::athErs7KickForwardWithLeftPaw;
    case MotionRequest::athErs7SideHeadKickL: return KickSelectionTable::athErs7SideHeadKickL;
    case MotionRequest::athErs7SideHeadKickR: return KickSelectionTable::athErs7SideHeadKickR;
    case MotionRequest::armKickLeft: return KickSelectionTable::armKickLeft;
    case MotionRequest::armKickRight: return KickSelectionTable::armKickRight;
    case MotionRequest::MSH7AnyLeft: return KickSelectionTable::msh7AnyLeft;
    case MotionRequest::MSH7AnyRight: return KickSelectionTable::msh7AnyRight;
    case MotionRequest::MSH7LeftHook: return KickSelectionTable::msh7LeftHook;
    case MotionRequest::MSH7RightHook: return KickSelectionTable::msh7RightHook;
    case MotionRequest::MSH7ForwardLeft: return KickSelectionTable::msh7ForwardLeft;
    case MotionRequest::MSH7ForwardRight: return KickSelectionTable::msh7ForwardRight;
    

     default: return KickSelectionTable::nothing;
    }
  }
  //{{AFX_DATA(CKickEditorDlgBar)
	enum { IDD = IDD_DIALOG_BAR_KICK_EDITOR };
	CSliderCtrl	m_sizeSlider;
	CComboBox	m_tableCombo;
	CComboBox	m_actionCombo;
	//}}AFX_DATA

  //{{AFX_VIRTUAL(CKickEditorDlgBar)
  protected:
    virtual void DoDataExchange(CDataExchange* pDX);    
    virtual BOOL OnInitDialog();
  //}}AFX_VIRTUAL
  
protected:
  /** A case base of recored kick experiments */
  CArray<KickCase,KickCase&> caseBase[KickSelectionTable::numberOfActions];

  /** Loads the case base from a file */
  void loadCaseBaseFromFile(const char* fileName);

  //{{AFX_MSG(CKickEditorDlgBar)
	afx_msg void OnPaint();
	afx_msg void OnSize(UINT nType, int cx, int cy);
	afx_msg void OnLButtonDown(UINT nFlags, CPoint point);
	afx_msg void OnLButtonUp(UINT nFlags, CPoint point);
	afx_msg void OnMouseMove(UINT nFlags, CPoint point);
	afx_msg void OnContextMenu(CWnd* pWnd, CPoint point);
	afx_msg void OnSave();
	afx_msg void OnSelchangeActionCombo();
	afx_msg void OnSelchangeTableCombo();
	afx_msg void OnOpen();
	afx_msg void OnLButtonDblClk(UINT nFlags, CPoint point);
	afx_msg void OnSend();
	afx_msg void OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
	//}}AFX_MSG
  
  DECLARE_MESSAGE_MAP();

private:
  /** 
  * Is called when a configuration is loaded. Can be used by the dialogs and 
  * bars to load configuration dependand settings.
  * @param sectionName The registry section for the current configuration
  */
  virtual void OnConfigurationLoad(CString sectionName);

  /** 
  * Is called when a configuration is saved. Can be used by the dialogs and 
  * bars to save configuration dependand settings.
  * @param sectionName The registry section for the current configuration
  */
  virtual void OnConfigurationSave(CString sectionName);

  void setSliders();

  void paintKickSelectionTableToCDC(
    CDC& dc, 
    double scale,
    const KickSelectionTable& kickSelectionTable,
    KickSelectionTable::KickSelectionTableID kickSelectionTableID,
    int sector
    );

  void paintLegend(CDC& dc);
  void changeKickPerLegend(CPoint point);

  void paintDot(CDC& dc, double scale, int x, int y, PaintMethodsWin32::PaintStyle style);


  void paintCurrentSelectionToCDC(CDC& dc, bool paintFrame = false);
  void copyToClipBoard();

  /** The scale for all drawings */
  double scale;

  // Mouse
	CPoint m_LeftDownPos;
	bool m_LeftButtonDown;
  
  int highlightedSector;
  int selectedSector;

  int highlightedX, highlightedY;

  /** The rect where the image is painted*/
  CRect rect;

  bool showCoordinateSystem;
  bool showRobot;
  bool showKickCases;

  int currentWidth;
  int currentHeight;
  int horizontalOffset;
  int verticalOffset;

  KickSelectionTable kickSelectionTable;
  KickSelectionTable kickSelectionTableForCopies;

  /** The paint style for each action */
  PaintMethodsWin32::PaintStyle paintStyles[KickSelectionTable::numberOfActions];

private:
  KickSelectionTable::KickSelectionTableID currentTable;
  KickSelectionTable::ActionID currentAction;

  int frameSize;


  // for legend and changeKickPerLegend
  int columnsCount;
  int columnWidth;
  int xOffset, yOffset, ySpacing;
  int actionsPerColumn;


  int selectedSectorWhenCopied;
  void copy();
  void pasteForCurrentSector();
  void pasteForAllSectors();
  void mirrorForCurrentSector();
  void mirrorForAllSectors();

  void paintKickCase(CDC& dc, double scale, KickCase& kickCase);
  void paintCaseBase(CDC& dc, double scale, KickSelectionTable::ActionID action);

  void floodFill(int x, int y, KickSelectionTable::ActionID oldAction, KickSelectionTable::ActionID newAction);
  KickSelectionTable::ActionID oldAction;  
  void setFloodFillColor(int x, int y, KickSelectionTable::ActionID newAction);

};

//{{AFX_INSERT_LOCATION}}

#endif //__KickEditorDlgBar_h_

/*
 * Change log :
 * 
 * $Log: KickEditorDlgBar.h,v $
 * Revision 1.10  2004/05/07 17:25:39  goetzke
 * improved floodfill
 *
 * Revision 1.9  2004/05/04 18:10:25  juengel
 * Added msh kicks
 *
 * Revision 1.8  2004/04/21 05:50:34  juengel
 * improved
 *
 * Revision 1.7  2004/04/05 17:56:50  loetzsch
 * merged the local German Open CVS of the aibo team humboldt with the tamara CVS
 *
 * Revision 1.2  2004/04/03 02:30:36  jumped
 * added more kicks
 *
 * Revision 1.1.1.1  2004/03/31 11:16:56  loetzsch
 * created ATH repository for german open 2004
 *
 * Revision 1.6  2004/03/19 23:01:52  altmeyer
 * added changeKickPerLegend
 *
 * Revision 1.5  2004/03/19 21:18:28  altmeyer
 * SideHeadKick and SideHeadKick2 changed and renamed to SideHeadKick L and SideHeadKickR
 * a few ATH ERS7-kicks removed
 *
 * Revision 1.4  2004/03/19 11:26:22  juengel
 * recursion depth of floodFill bounded
 *
 * Revision 1.3  2004/03/17 19:55:26  juengel
 * Added OnSend() and floodFill()
 *
 * Revision 1.2  2004/03/16 16:25:02  altmeyer
 * improved visualization using columns and different colors/symbol for each special action
 *
 * Revision 1.1  2004/03/16 14:00:22  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.11  2004/03/11 17:28:02  juengel
 * Added visualization of KickCases.
 *
 * Revision 1.10  2004/03/11 11:41:21  loetzsch
 * implemented loading of case base
 *
 * Revision 1.9  2004/03/10 22:02:03  juengel
 * Added copy and mirror methods.
 *
 * Revision 1.8  2004/03/10 21:17:00  juengel
 * Added Buttons to KickEditor.
 *
 * Revision 1.7  2004/03/10 20:56:52  juengel
 * Moved origin of KickEditor.
 *
 * Revision 1.6  2004/03/10 16:37:58  juengel
 * Added combo boxes to KickEditor.
 *
 * Revision 1.5  2004/03/10 13:47:49  juengel
 * Added save button to KickEditorDlg.
 *
 * Revision 1.4  2004/03/09 18:44:11  juengel
 * Some improvements.
 *
 * Revision 1.3  2004/03/09 13:41:54  juengel
 * Moved KickSelectionTable to Tools.
 *
 * Revision 1.2  2004/03/09 01:14:44  juengel
 * Added class KickDirections.
 *
 * Revision 1.1  2004/03/08 00:17:04  juengel
 * Added the KickEditorDlgBar.
 *
*/
