/**
* @file ColorTable64DlgBar.cpp
*
* Implementation of class CColorTable64DlgBar
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/

#include "StdAfx.h"
#include "ColorTable64DlgBar.h"

#include "Visualization/DebugDrawingManager/DebugDrawingManager.h"
#include "Visualization/ImageMethods.h"
#include "Visualization/PaintMethodsWin32.h"

#include "RobotControlQueues.h"
#define HAVE_BOOLEAN
#include "Representations/Perception/JPEGImage.h"
#include "Platform/Sensors.h"
#include "Tools/Location.h"

CColorTable64DlgBar::CColorTable64DlgBar()
: CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CColorTable64DlgBar)
  //}}AFX_DATA_INIT
}

CColorTable64DlgBar::~CColorTable64DlgBar()
{
}

bool CColorTable64DlgBar::handleMessage(InMessage& message)
{
  if (message.getMessageID() == idImage || message.getMessageID() == idJPEGImage)
  {
    if (debugImageID == Images::rawImage)
    {
      if(message.getMessageID() == idImage)
        message.bin >> rawImage[0];
      else
      {
        JPEGImage jpeg;
        message.bin >> jpeg;
        jpeg.toImage(rawImage[0]);
      }

      yPosSegmentedImage = rawImage[0].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1) + 1;
      for(int i = 0; i < 5; i++)
        xPosSegmentedImage[i] = xPosRawImage[i] = 120 + i * (rawImage[0].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) + 2);

      if (this->IsWindowVisible())
      {
        generateSegmentedImage();
        RedrawWindow(NULL, NULL, RDW_INVALIDATE);
        if ( m_autoremove.GetCheck() ) this->removeColorClassFromImage();
        if ( m_automedian.GetCheck() ) this->smoothImage();
      }
    }
    return true;
  }
  else if (message.getMessageID() == idDebugImage)
  {
    int id;
    message.bin >> id;
    if (id == debugImageID)
    {
      message.bin >> rawImage[0];

      yPosSegmentedImage = rawImage[0].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1) + 1;
      for(int i = 0; i < 5; i++)
        xPosSegmentedImage[i] = xPosRawImage[i] = 120 + i * (rawImage[0].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) + 2);

      if (this->IsWindowVisible())
      {
        generateSegmentedImage();
        RedrawWindow(NULL, NULL, RDW_INVALIDATE);
        if ( m_autoremove.GetCheck() ) this->removeColorClassFromImage();
        if ( m_automedian.GetCheck() ) this->smoothImage();
      }
    }
    return true;
  }
  else 
    return false;
}

void CColorTable64DlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CColorTable64DlgBar)
	DDX_Control(pDX, IDC_CT64_SELECTED_COLOR_ONLY, m_selectedColorOnlyCheck);
  DDX_Control(pDX, IDC_CT64_SEND_TO_ROBOT, m_sendButton);
  DDX_Control(pDX, IDC_CT64_RANGE_SLIDER, m_rangeSlider);
  DDX_Control(pDX, IDC_CT64_RANGE_STATIC, m_rangeStatic);
  DDX_Control(pDX, IDC_CT64_PEN_SLIDER, m_penSlider);
  DDX_Control(pDX, IDC_CT64_PEN_STATIC, m_penStatic);
  DDX_Control(pDX, IDC_CT64_COLOR_CLASS_COMBO, m_ct64ColorClassCombo);
	DDX_Control(pDX, IDC_CT64_AUTO_MEDIAN, m_automedian);
	DDX_Control(pDX, IDC_CT64_AUTO_REMOVE, m_autoremove);
	DDX_Control(pDX, IDC_CT64_MEDIAN_ALL_COLORS, m_medianAllColors);
	DDX_Control(pDX, IDC_CT64_MEDIAN_ALLOW_REASSIGN, m_medianAllowReassign);
  DDX_Control(pDX, IDC_CT64_MEDIAN_RANGE_SLIDER, m_medianRangeSlider);
  DDX_Control(pDX, IDC_CT64_MEDIAN_RANGE_STATIC, m_medianRangeStatic);
  DDX_Control(pDX, IDC_CT64_MEDIAN_DISTANCE_SLIDER, m_medianDistanceSlider);
  DDX_Control(pDX, IDC_CT64_MEDIAN_DISTANCE_STATIC, m_medianDistanceStatic);
  DDX_Control(pDX, IDC_CT64_MEDIAN_NEIGHBOUR_SLIDER, m_medianNeighbourSlider);
  DDX_Control(pDX, IDC_CT64_MEDIAN_NEIGHBOUR_STATIC, m_medianNeighbourStatic);
  DDX_Control(pDX, IDC_CT64_IMAGE_COMBO, m_ct64ImageCombo);
  DDX_Control(pDX, IDC_CT64_HIGH_RES, m_highRes);
  //}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CColorTable64DlgBar, CDynamicBarDlg)
//{{AFX_MSG_MAP(CColorTable64DlgBar)
ON_WM_PAINT()
ON_WM_SIZE()
ON_BN_CLICKED(IDC_CT64_CLEAR_COLOR_TABLE, OnCt64ClearColorTable)
ON_BN_CLICKED(IDC_CT64_CLEAR_CHANNEL, OnCt64ClearChannel)
ON_BN_CLICKED(IDC_CT64_OPEN, OnCt64Open)
ON_BN_CLICKED(IDC_CT64_SAVE, OnCt64Save)
ON_CBN_SELCHANGE(IDC_CT64_COLOR_CLASS_COMBO, OnSelchangeCt64ColorClassCombo)
ON_WM_LBUTTONDOWN()
ON_WM_RBUTTONDOWN()
ON_WM_HSCROLL()
ON_BN_CLICKED(IDC_CT64_UNDO, OnCt64Undo)
ON_BN_CLICKED(IDC_CT64_SEND_TO_ROBOT, OnCt64SendToRobot)
ON_BN_CLICKED(IDC_CT64_REMOVE, OnCt64Remove)
ON_BN_CLICKED(IDC_CT64_MEDIAN, OnCt64Median)
	ON_BN_CLICKED(IDC_CT64_CLEAR_CHANNEL_IN_IMAGE, OnClearChannelInImage)
	ON_BN_CLICKED(IDC_CT64_SELECTED_COLOR_ONLY, OnSelectedColorOnly)
ON_WM_TIMER()
ON_BN_CLICKED(IDC_CT64_SEND_TO_ROBOT, OnCt64SendToRobot)
ON_CBN_SELCHANGE(IDC_CT64_IMAGE_COMBO, OnSelchangeCt64ImageCombo)
ON_WM_MOUSEMOVE()
ON_BN_CLICKED(IDC_CT64_CAPTURE_1, OnCapture1)
ON_BN_CLICKED(IDC_CT64_CAPTURE_2, OnCapture2)
ON_BN_CLICKED(IDC_CT64_CAPTURE_3, OnCapture3)
ON_BN_CLICKED(IDC_CT64_CAPTURE_4, OnCapture4)
ON_BN_CLICKED(IDC_CT64_HIGH_RES, OnHighRes)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


void CColorTable64DlgBar::sendLocal()
{
  getQueues().toSimulated.allRobots.out.bin << colorTable;
  getQueues().toSimulated.allRobots.out.finishMessage(idColorTable64);

  getQueues().toGUI.out.bin << colorTable;
  getQueues().toGUI.out.finishMessage(idColorTable64);
}

void CColorTable64DlgBar::OnCt64SendToRobot() 
{
  getQueues().toPhysical.allRobots.out.bin << colorTable;
  getQueues().toPhysical.allRobots.out.finishMessage(idColorTable64);
}

BOOL CColorTable64DlgBar::OnInitDialog() 
{
  CDynamicBarDlg::OnInitDialog();
  
  selectedColorClass = orange;
  showSelectedColorClassOnly = false;
  // initialize the entries in the color selector
  int i;
  for(i = 0; i < numOfColors; i++)
  {
    colorClass currentColorClass = (colorClass)i;
    
    m_ct64ColorClassCombo.AddString(
      ColorClasses::getColorName(currentColorClass)
      );
  }
  m_ct64ColorClassCombo.SetCurSel(1);

  m_highRes.SetCheck(1);
  useHighRes = true;

  //initialize image combo
  debugImageID = Images::rawImage;
  for(i = 1; i < Images::numberOfImageIDs; i++)
  {
    m_ct64ImageCombo.AddString(Images::getImageIDName((Images::ImageID)i));
  }
  m_ct64ImageCombo.SetCurSel(debugImageID - 1);
  
  // write something to the color table
  InBinaryFile fin(getLocation().getFilename("coltable.c64"));
  if (fin.exists())
  {
    fin >> colorTable;
  }
  else
  {
    colorTable.clear();
  }
  
  previousColorTable=colorTable; // normally obsolete init for undo
  undoColorTablePossible=FALSE; // this is a new color table
 
  repaintRawImage = true;
  repaintSegmentedImage = true;
  
  yPosRawImage = 0;
  
  yPosSegmentedImage = rawImage[0].cameraInfo.resolutionHeight + 1;
  
  for(i = 0; i < 5; i++)
  {
    ImageMethods::writeRobotControlLogoToImage(rawImage[i]);
    xPosSegmentedImage[i] = xPosRawImage[i] = 120 + i * (rawImage[0].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) + 2);
    imageCaptured[i] = (i == 0);
  }

  generateSegmentedImage();
  
  m_rangeSlider.SetRange(0, 30);
  setClickRangeSlider(1);
  m_penSlider.SetRange(0, 5);
  setPenSlider(0);
  
  m_medianRangeSlider.SetRange(0, 30);
  setMedianRangeSlider(0);
  m_medianDistanceSlider.SetRange(0, 30);
  setMedianDistanceSlider(2);
  m_medianNeighbourSlider.SetRange(3, 8);
  setMedianNeighbourSlider(7);

  for (i = 0; i < 5; i++)
  {
    xPosPreviewPixel[i] = 0;
    yPosPreviewPixel[i] = 0;
  }

  return TRUE;
}

void CColorTable64DlgBar::OnPaint() 
{
  CPaintDC dc(this); // device context for painting
  
  for (int i = 0; i < 5; i++)
    if (imageCaptured[i])
    {
      ImageMethods::paintImage2CDCAsYUV(
        dc, rawImage[i],
        CRect(xPosRawImage[i], yPosRawImage, xPosRawImage[i] + rawImage[i].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1), yPosRawImage + rawImage[i].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1)), 
        useHighRes );
      ImageMethods::paintColorClassImage2CDC(
        dc, 
        segmentedImage[i], 
        CRect(xPosSegmentedImage[i], yPosSegmentedImage, xPosSegmentedImage[i] + segmentedImage[i].width, yPosSegmentedImage + segmentedImage[i].height) );

      // paint preview pixel
      RECT rect;
      rect.top = yPosSegmentedImage + segmentedImage[i].height + 1; //rect.bottom + 10;
      rect.left = xPosSegmentedImage[i];
      rect.bottom = rect.top + 50;
      rect.right = rect.left + 50;
      int Y,U,V;
      if (useHighRes)
      {
        Y = rawImage[i].getHighResY(xPosPreviewPixel[i], yPosPreviewPixel[i]);
        U = rawImage[i].image[yPosPreviewPixel[i] / 2][1][xPosPreviewPixel[i] / 2] - 128;
        V = rawImage[i].image[yPosPreviewPixel[i] / 2][2][xPosPreviewPixel[i] / 2] - 128;
      } else {
        Y = rawImage[i].image[yPosPreviewPixel[i]][0][xPosPreviewPixel[i]];
        U = rawImage[i].image[yPosPreviewPixel[i]][1][xPosPreviewPixel[i]] - 128;
        V = rawImage[i].image[yPosPreviewPixel[i]][2][xPosPreviewPixel[i]] - 128;
      }
      int B = (int)(Y + 1.140 * V);
      int G = (int)(Y - 0.394 * U - 0.581 * V);
      int R = (int)(Y + 2.028 * U);
      if(R < 0) R = 0; if(R > 255) R = 255;
      if(G < 0) G = 0; if(G > 255) G = 255;
      if(B < 0) B = 0; if(B > 255) B = 255;
      COLORREF color = RGB(R,G,B);
      dc.FillSolidRect(&rect, color);
      rect.left = rect.right + 1;
      rect.right = rect.left + 50;
      color = ColorClasses::colorClassToRGB((colorClass)segmentedImage[i].image[yPosPreviewPixel[i]][xPosPreviewPixel[i]]);
      dc.FillSolidRect(&rect, color);
    }
}

void CColorTable64DlgBar::OnSize(UINT nType, int cx, int cy) 
{
}

void CColorTable64DlgBar::OnCt64Undo() 
{
  if (this->undoColorTablePossible) 
  {
    this->colorTable=this->previousColorTable;
    this->undoColorTablePossible=FALSE;
    generateSegmentedImage();
    
    sendLocal();
    
    RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  }
}


void CColorTable64DlgBar::OnCt64ClearColorTable() 
{
  this->previousColorTable=this->colorTable;
  this->undoColorTablePossible=TRUE;
  this->colorTable.clear();
  generateSegmentedImage();
  
  sendLocal();
  
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnCt64ClearChannel() 
{
  this->previousColorTable=this->colorTable;
  this->undoColorTablePossible=TRUE;
  this->colorTable.clearChannel(selectedColorClass);
  generateSegmentedImage();
  
  sendLocal();
  
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnCt64Open() 
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("ColorTable64Dlg", "openPath", defaultPath);
  pathName += "*.c64";

  CFileDialog fileDialog(true, ".c64",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "color tables (*.c64)|*.c64|cmu color tables (*.tm)|*.tm|unsw color tables (*.cal)|*.cal||", this);

  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("ColorTable64Dlg", "openPath", pathName);
  
    InBinaryFile fin(pathAndFileName);
    if (fin.exists())
    {
      if (this->undoColorTablePossible) // there have been actions
      {
        this->previousColorTable=this->colorTable;
      }
      if ( fileDialog.GetFileExt() == "c64" )
      {
	      fin >> colorTable;
		  }
      else if ( fileDialog.GetFileExt() == "tm" )
      {
        // read cmu colortable - Y 4bit, U 6bit, V 6bit
        char header[19];
			  fin.read(&header, 19);
        colorClass colorclass;
        char idx;
        for ( int y = 0; y < 256; y = y + 16 ) {
          for ( int u = 0; u < 256; u = u + 4 ) {
            for ( int v = 0; v < 256; v = v + 4 ) {
              fin >> idx;
              switch ( idx ) {
                case 0:
                  colorclass = noColor;
                  break;
                case 1:
                  colorclass = orange;
                  break;
                case 2:
                  colorclass = green;
                  break;
                case 3:
                  colorclass = pink;
                  break;
                case 4:
                  colorclass = skyblue;
                  break;
                case 5:
                  colorclass = yellow;
                  break;
                case 6:
                  colorclass = blue;
                  break;
                case 7:
                  colorclass = red;
                  break;
                case 8:
                  colorclass = white;
                  break;
                default:
                  colorclass = noColor;
                  break;
						  }
              colorTable.addCuboidToColorClass(colorclass, y, u, v, y+15, u+3, v+3);
            }
          }
        }
		  }
      else if ( fileDialog.GetFileExt() == "cal" )
      {
        // read unsw colortable - Y 7bit, U 7bit, V 7bit
        colorClass colorclass;
        char idx;
        for ( int y = 0; y < 256; y = y + 2 ) {
          for ( int u = 0; u < 256; u = u + 2 ) {
            for ( int v = 0; v < 256; v = v + 2 ) {
              fin >> idx;
              switch ( idx ) {
                case 0:
                  colorclass = orange;
                  break;
                case 1:
                  colorclass = skyblue;
                  break;
                case 2: // green landmark
                  colorclass = green;
                  break;
                case 3:
                  colorclass = yellow;
                  break;
                case 4:
                  colorclass = pink;
                  break;
                case 5:
                  colorclass = blue;
                  break;
                case 6:
                  colorclass = red;
                  break;
                case 7: // green floor
                  colorclass = green;
                  break;
							  case 8: // white and nocolor
                  colorclass = white;
                  break;
              }
              colorTable.addCuboidToColorClass(colorclass, y, u, v, y+1, u+1, v+1);
            }
          }
        }
      }
      else
      {
        CString message;
        message.Format( "Unknown file-extension %s.", fileDialog.GetFileExt());
        AfxMessageBox( message, MB_OK);
      }
    }
    else
    {
      CString message;
      message.Format( "File %s not found.", pathAndFileName);
      AfxMessageBox( message, MB_OK);
    }
  }
  generateSegmentedImage();

  sendLocal();

  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnCt64Save() 
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("ColorTable64Dlg", "savePath", defaultPath);
  pathName += "*.c64";

  CFileDialog fileDialog(false, ".c64",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "color tables (*.c64)|*.c64|cmu color tables (*.tm)|*.tm|unsw color tables (*.cal)|*.cal||", this);

  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("ColorTable64Dlg", "savePath", pathName);
  
    OutBinaryFile fout(pathAndFileName);
    if (fout.exists())
    {
      if ( fileDialog.GetFileExt() == "c64" )
      {
        fout << colorTable;
      }
      else if ( fileDialog.GetFileExt() == "tm" )
      {
        // create cmu colortable - Y 4bit, U 6bit, V 6bit
        char* header;
        header = "TMAP\nYUV8\n16 64 64\n";
        fout.write ( header, 19 );
        colorClass colorclass;
        char idx;
        for ( int y = 0; y < 256; y = y + 16 ) {
          for ( int u = 0; u < 256; u = u + 4 ) {
            for ( int v = 0; v < 256; v = v + 4 ) {
              colorclass = (colorClass) colorTable.getColorClass ( y, u, v );
              switch ( colorclass ) {
                case noColor:
                  idx = (char) 0;
                  break;
                case orange:
                  idx = (char) 1;
                  break;
                case green:
                  idx = (char) 2;
                  break;
                case pink:
                  idx = (char) 3;
                  break;
                case skyblue:
                  idx = (char) 4;
                  break;
                case yellow:
                  idx = (char) 5;
                  break;
                case blue:
                  idx = (char) 6;
                  break;
                case red:
                  idx = (char) 7;
                  break;
                case white:
                  idx = (char) 8;
                  break;
                default:
                  idx = (char) 0;
                  break;
              }
              fout << idx;
            }
          }
        }
      }
      else if ( fileDialog.GetFileExt() == "cal" )
      {
        // create unsw colortable - Y 7bit, U 7bit, V 7bit
        colorClass colorclass;
        char idx;
        for ( int y = 0; y < 256; y = y + 2 ) {
          for ( int u = 0; u < 256; u = u + 2 ) {
            for ( int v = 0; v < 256; v = v + 2 ) {
              colorclass = (colorClass) colorTable.getColorClass ( y, u, v );
              switch ( colorclass ) {
                case orange:
                  idx = (char) 0;
                  break;
                case skyblue:
                  idx = (char) 1;
                  break;
			  // green landmark = 2
                case yellow:
                  idx = (char) 3;
                  break;
                case pink:
                  idx = (char) 4;
                  break;
                case blue:
                  idx = (char) 5;
                  break;
                case red:
                  idx = (char) 6;
                  break;
                case green:
                  idx = (char) 7;
                  break;
                case white:
                case noColor:
                default:
                  idx = (char) 8;
                  break;
              }
              fout << idx;
            }
          }
        }
      }
      else
      {
        CString message;
        message.Format( "Unknown file-extension %s.", fileDialog.GetFileExt());
        AfxMessageBox( message, MB_OK);
      }
    }
    else
    {
      CString message;
      message.Format( "File %s not found.", pathAndFileName);
      AfxMessageBox( message, MB_OK);
    }
  }
}

void CColorTable64DlgBar::OnSelchangeCt64ColorClassCombo() 
{
  selectedColorClass = (colorClass)m_ct64ColorClassCombo.GetCurSel();
  generateSegmentedImage();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnSelchangeCt64ImageCombo() 
{
  debugImageID = m_ct64ImageCombo.GetCurSel() + 1;
}

void CColorTable64DlgBar::OnLButtonDown(UINT nFlags, CPoint point) 
{
  bool somethingWasSelected = false;
  
  int selectedX = 0;
  int selectedY = 0;
  
  for (int c = 0; c < 5; c++)
    if (imageCaptured[c])
    {
      //is the selected Point in the raw image?
      if(
        point.x >= xPosRawImage[c] &&
        point.x < xPosRawImage[c] + rawImage[c].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) &&
        point.y >= yPosRawImage &&
        point.y < yPosRawImage + rawImage[c].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1)
        )
      {
        somethingWasSelected = true;
        selectedX = point.x - xPosRawImage[c];
        selectedY = point.y - yPosRawImage;
      }
  
      //is the selected Point in the segmented image?
      if(
        point.x >= xPosSegmentedImage[c] &&
        point.x < xPosSegmentedImage[c] + rawImage[c].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) &&
        point.y >= yPosSegmentedImage &&
        point.y < yPosSegmentedImage + rawImage[c].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1)
        )
      {
        somethingWasSelected = true;
        selectedX = point.x - xPosSegmentedImage[c];
        selectedY = point.y - yPosSegmentedImage;
      }
  
      if (somethingWasSelected)
      {
        this->previousColorTable=this->colorTable;
        this->undoColorTablePossible=TRUE;
    
        for ( int i = selectedY - penSize; i <= selectedY + penSize; i++ ) {
          for ( int j = selectedX - penSize; j <= selectedX + penSize; j++ ) {
            if ( i >= 0 && 
                 i < rawImage[c].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1) && 
                 j >= 0 && 
                 j < rawImage[c].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) ) 
            {
              this->colorTable.addColorClass(
                selectedColorClass,
                (useHighRes ? rawImage[c].getHighResY(j, i) : rawImage[c].image[i][0][j]),
                rawImage[c].image[i / (useHighRes ? 2 : 1)][1][j / (useHighRes ? 2 : 1)],
                rawImage[c].image[i / (useHighRes ? 2 : 1)][2][j / (useHighRes ? 2 : 1)],
                clickRange
              );
            }
          }
        }
        generateSegmentedImage();
    
        sendLocal();

        RedrawWindow(NULL, NULL, RDW_INVALIDATE);
        break;
      }
    }
  
  CDynamicBarDlg::OnLButtonDown(nFlags, point);
}

void CColorTable64DlgBar::OnRButtonDown(UINT nFlags, CPoint point) 
{
  bool somethingWasSelected = false;
  
  int selectedX = 0;
  int selectedY = 0;
  
  for (int c = 0; c < 5; c++)
    if (imageCaptured[c])
    {
      //is the selected Point in the raw image?
      if(
        point.x >= xPosRawImage[c] &&
        point.x < xPosRawImage[c] + rawImage[c].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) &&
        point.y >= yPosRawImage &&
        point.y < yPosRawImage + rawImage[c].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1)
        )
      {
        somethingWasSelected = true;
        selectedX = point.x - xPosRawImage[c];
        selectedY = point.y - yPosRawImage;
      }
  
      //is the selected Point in the segmented image?
      if(
        point.x >= xPosSegmentedImage[c] &&
        point.x < xPosSegmentedImage[c] + rawImage[c].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) &&
        point.y >= yPosSegmentedImage &&
        point.y < yPosSegmentedImage + rawImage[c].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1)
        )
      {
        somethingWasSelected = true;
        selectedX = point.x - xPosSegmentedImage[c];
        selectedY = point.y - yPosSegmentedImage;
      }
  
      if (somethingWasSelected)
      {
        this->previousColorTable=this->colorTable;
        this->undoColorTablePossible=TRUE;
    
        for ( int i = selectedY - penSize; i <= selectedY + penSize; i++ ) {
          for ( int j = selectedX - penSize; j <= selectedX + penSize; j++ ) {
            if ( i >= 0 && 
                 i < rawImage[c].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1) && 
                 j >= 0 && 
                 j < rawImage[c].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) &&
                 (!showSelectedColorClassOnly || (colorClass)this->segmentedImage[0].image[i][j] == selectedColorClass))
            {
              this->colorTable.removeColorClass(
                (useHighRes ? rawImage[c].getHighResY(j, i) : rawImage[c].image[i][0][j]),
                rawImage[c].image[i / (useHighRes ? 2 : 1)][1][j / (useHighRes ? 2 : 1)],
                rawImage[c].image[i / (useHighRes ? 2 : 1)][2][j / (useHighRes ? 2 : 1)],
                clickRange
              );
            }
          }
        }
    
        generateSegmentedImage();
    
        sendLocal();
    
        RedrawWindow(NULL, NULL, RDW_INVALIDATE);
        break;
      }
    }
  
  CDynamicBarDlg::OnRButtonDown(nFlags, point);
}

void CColorTable64DlgBar::OnMouseMove(UINT nFlags, CPoint point) 
{
  bool somethingWasSelected = false;
  
  int selectedX = 0;
  int selectedY = 0;
  
  for (int c = 0; c < 5; c++)
    if (imageCaptured[c])
    {
      //is the selected Point in the raw image?
      if(
        point.x >= xPosRawImage[c] &&
        point.x < xPosRawImage[c] + rawImage[c].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) &&
        point.y >= yPosRawImage &&
        point.y < yPosRawImage + rawImage[c].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1)
        )
      {
        somethingWasSelected = true;
        selectedX = point.x - xPosRawImage[c];
        selectedY = point.y - yPosRawImage;
      }
  
      //is the selected Point in the segmented image?
      if(
        point.x >= xPosSegmentedImage[c] &&
        point.x < xPosSegmentedImage[c] + rawImage[c].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) &&
        point.y >= yPosSegmentedImage &&
        point.y < yPosSegmentedImage + rawImage[c].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1)
        )
      {
        somethingWasSelected = true;
        selectedX = point.x - xPosSegmentedImage[c];
        selectedY = point.y - yPosSegmentedImage;
      }
  
      if (somethingWasSelected)
      {
        xPosPreviewPixel[c] = selectedX;
        yPosPreviewPixel[c] = selectedY;
        RedrawWindow(NULL, NULL, RDW_INVALIDATE);
        break;
      }
    }
}

void CColorTable64DlgBar::OnCt64Remove()
{
  this->removeColorClassFromImage();
  sendLocal();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnCt64Median()
{
  this->smoothImage();
  sendLocal();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::smoothImage()
{
  HCURSOR lhCursor = AfxGetApp()->LoadStandardCursor(IDC_APPSTARTING);
  SetCursor(lhCursor);

  for ( int i = 1; i < rawImage[0].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1) - 1; i++ ) {
    for ( int j = 1; j < rawImage[0].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) - 1 - 1; j++ ) {

      int y1 = (useHighRes ? this->rawImage[0].getHighResY(j, i) : this->rawImage[0].image[i][0][j]);
      int u1 = this->rawImage[0].image[i / (useHighRes ? 2 : 1)][1][j / (useHighRes ? 2 : 1)];
      int v1 = this->rawImage[0].image[i / (useHighRes ? 2 : 1)][2][j / (useHighRes ? 2 : 1)];

      colorClass colorclass = (colorClass)this->segmentedImage[0].image[i][j];
      if ( (colorclass == noColor) || m_medianAllowReassign.GetCheck() ) {
        
        for ( int m = 0; m < numOfColors; m++ ) {
          colorClass c1 = (colorClass)m;
          
          if ( ( c1 != noColor ) && ( (c1 == selectedColorClass) || (m_medianAllColors.GetCheck()) ) ) {
            int counter = 0;
            double minDistance = 10000.0;
            for ( int k = i-1; k <= i+1; k++ ) {
              for ( int l = j-1; l <= j+1; l++ ) {
                if ( (k != i) || (l != j) ) {
                  colorClass c = (colorClass)this->segmentedImage[0].image[k][l];
                  if ( c1 == c ) {
                    counter++;
                    int y2 = (useHighRes ? this->rawImage[0].getHighResY(l, k) : this->rawImage[0].image[i][0][j]);;
                    int u2 = this->rawImage[0].image[k / (useHighRes ? 2 : 1)][1][l / (useHighRes ? 2 : 1)];
                    int v2 = this->rawImage[0].image[k / (useHighRes ? 2 : 1)][2][l / (useHighRes ? 2 : 1)];
                    double distance = sqrt ( sqr ((double)(y1-y2)) + sqr ((double)(u1-u2)) + sqr ((double)(v1-v2)) );
                    if ( distance < minDistance ) minDistance = distance;
                  }
                }
              }
            }
            if ( ( counter >= medianNeighbour ) && ( minDistance < medianDistance ) ) {
              this->colorTable.addColorClass( c1, y1, u1, v1, medianRange );
            } // if ( counter >= 7 )
          } // if ( c1 != noColor )
        } // for ( int m = 0; m < numOfColors; m++ )
      } // if ( colorclass == noColor )
    }
  }

  lhCursor = AfxGetApp()->LoadStandardCursor(IDC_ARROW);
  SetCursor(lhCursor);
}

  
void CColorTable64DlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  setClickRangeSlider(m_rangeSlider.GetPos());
  setPenSlider(m_penSlider.GetPos());
  setMedianRangeSlider(m_medianRangeSlider.GetPos());
  setMedianDistanceSlider(m_medianDistanceSlider.GetPos());
  setMedianNeighbourSlider(m_medianNeighbourSlider.GetPos());
  CDynamicBarDlg::OnHScroll(nSBCode, nPos, pScrollBar);
}

void CColorTable64DlgBar::setClickRangeSlider(unsigned char newPosition)
{
  CString string;
  m_rangeSlider.SetPos(newPosition);
  clickRange = newPosition;
  string.Format("%d", newPosition);
  m_rangeStatic.SetWindowText(string);
}

void CColorTable64DlgBar::setPenSlider(unsigned char newPosition)
{
  CString string;
  m_penSlider.SetPos(newPosition);
  penSize = newPosition;
  string.Format("%d", newPosition * 2 + 1);
  m_penStatic.SetWindowText(string);
}

void CColorTable64DlgBar::setMedianRangeSlider(unsigned char newPosition)
{
  CString string;
  m_medianRangeSlider.SetPos(newPosition);
  medianRange = newPosition;
  string.Format("%d", newPosition);
  m_medianRangeStatic.SetWindowText(string);
}

void CColorTable64DlgBar::setMedianDistanceSlider(unsigned char newPosition)
{
  CString string;
  m_medianDistanceSlider.SetPos(newPosition);
  medianDistance = newPosition;
  string.Format("%d", newPosition);
  m_medianDistanceStatic.SetWindowText(string);
}

void CColorTable64DlgBar::setMedianNeighbourSlider(unsigned char newPosition)
{
  CString string;
  m_medianNeighbourSlider.SetPos(newPosition);
  medianNeighbour = newPosition;
  string.Format("%d", newPosition);
  m_medianNeighbourStatic.SetWindowText(string);
}

void CColorTable64DlgBar::OnClearChannelInImage() 
{
  removeColorClassFromImage();
}

void CColorTable64DlgBar::removeColorClassFromImage()
{
  this->previousColorTable=this->colorTable;
  this->undoColorTablePossible=TRUE;

  for(int x = 0; x < rawImage[0].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1); x++)
  {
    for(int y = 0; y < rawImage[0].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1); y++)
    {
      if(colorTable.getColorClass(
        (useHighRes ? rawImage[0].getHighResY(x, y) : rawImage[0].image[y][0][x]),
        rawImage[0].image[y / (useHighRes ? 2 : 1)][1][x / (useHighRes ? 2 : 1)],
        rawImage[0].image[y / (useHighRes ? 2 : 1)][2][x / (useHighRes ? 2 : 1)]) == selectedColorClass
        )
      {
        this->colorTable.removeColorClass(
          (useHighRes ? rawImage[0].getHighResY(x, y) : rawImage[0].image[y][0][x]),
          rawImage[0].image[y / (useHighRes ? 2 : 1)][1][x / (useHighRes ? 2 : 1)],
          rawImage[0].image[y / (useHighRes ? 2 : 1)][2][x / (useHighRes ? 2 : 1)],
          clickRange
        );
      }
    }
  }

  generateSegmentedImage();
  
  sendLocal();
    
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnSelectedColorOnly() 
{
	showSelectedColorClassOnly = (m_selectedColorOnlyCheck.GetCheck() == 1);
  generateSegmentedImage();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::generateSegmentedImage()
{
  for (int i = 0; i < 5; i++)
  {
    if (useHighRes)
    {
      if(showSelectedColorClassOnly)
        colorTable.generateHighResColorClassImage(rawImage[i], segmentedImage[i], selectedColorClass);
      else
        colorTable.generateHighResColorClassImage(rawImage[i], segmentedImage[i]);
    } else {
      if(showSelectedColorClassOnly)
        colorTable.generateColorClassImage(rawImage[i], segmentedImage[i], selectedColorClass);
      else
        colorTable.generateColorClassImage(rawImage[i], segmentedImage[i]);
    }
  }
}

void CColorTable64DlgBar::OnCapture1()
{
  rawImage[1] = rawImage[0];
  imageCaptured[1] = true;
  generateSegmentedImage();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnCapture2()
{
  rawImage[2] = rawImage[0];
  imageCaptured[2] = true;
  generateSegmentedImage();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnCapture3()
{
  rawImage[3] = rawImage[0];
  imageCaptured[3] = true;
  generateSegmentedImage();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnCapture4()
{
  rawImage[4] = rawImage[0];
  imageCaptured[4] = true;
  generateSegmentedImage();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CColorTable64DlgBar::OnHighRes()
{
  useHighRes = (m_highRes.GetCheck() == 1);

  yPosSegmentedImage = rawImage[0].cameraInfo.resolutionHeight * (useHighRes ? 2 : 1) + 1;
  for(int i = 0; i < 5; i++)
    xPosSegmentedImage[i] = xPosRawImage[i] = 120 + i * (rawImage[0].cameraInfo.resolutionWidth * (useHighRes ? 2 : 1) + 2);

  generateSegmentedImage();
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

/*
* Change log :
* 
* $Log: ColorTable64DlgBar.cpp,v $
* Revision 1.16  2004/05/03 12:31:12  thomas
* fix removing of selected color only works now as intended
*
* Revision 1.15  2004/04/23 12:53:22  risler
* imageProcessorGeneral should not be default for ColorTableToolBar, was accidentally checked in from DDDGO stuff
*
* Revision 1.14  2004/04/17 12:10:20  risler
* bugfix: add color class did not work in low resolution
*
* Revision 1.13  2004/04/07 13:00:46  risler
* ddd checkin after go04 - second part
*
* Revision 1.6  2004/04/06 13:19:35  risler
* cleaned up and improved high resolution image support
*
* Revision 1.5  2004/04/05 14:58:34  risler
* colortable dialog now can show both resolutions
*
* Revision 1.4  2004/04/02 14:46:19  risler
* added segmented high resolution image
* colortable dialog supports high resolution images
*
* Revision 1.3  2004/03/30 08:12:48  risler
* bugfix multiple image positions
*
* Revision 1.2  2004/03/29 13:32:26  risler
* imageProcessorGeneral as default image
*
* Revision 1.1.1.1  2004/03/29 08:28:44  Administrator
* initial transfer from tamara
*
* Revision 1.12  2004/03/28 20:30:19  risler
* corrected YUV conversion for pixel preview
*
* Revision 1.11  2004/03/28 18:39:35  risler
* colortable64 dialog multiple image support
*
* Revision 1.10  2004/03/28 14:16:40  risler
* left pixel preview corrected
*
* Revision 1.9  2004/03/18 15:03:50  risler
* added pixel preview
*
* Revision 1.8  2004/03/18 10:35:36  risler
* simplified image selection
*
* Revision 1.7  2004/03/17 17:24:47  risler
* ColorTable64Dlg can segment debug images
*
* Revision 1.6  2004/01/09 15:56:14  thomas
* added: support for importing cmu/unsw ct's and exporting unsw ct's
*
* Revision 1.5  2004/01/04 16:36:41  juengel
* adaptation for larger images
*
* Revision 1.4  2003/12/15 11:46:14  juengel
* Introduced CameraInfo
*
* Revision 1.3  2003/12/09 19:49:24  loetzsch
* Renamed some of the main queues of RobotControl.
*
* Added possibility to send messages to specific simulated or physical robots.
*
* Revision 1.2  2003/12/09 16:25:29  jhoffman
* - addColorClass( , , , , ) now works like addColorClass( , , , , , range)
* - 176x144 --> 144 x ...Width
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/09/26 15:28:23  juengel
* Renamed DataTypes to representations.
*
* Revision 1.3  2003/09/26 11:40:12  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.2  2003/07/07 07:38:04  roefer
* JPEG support added
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.14  2003/05/26 13:09:49  thomas
* added: option to save color table in cmu-format
*
* Revision 1.13  2003/05/11 23:46:32  dueffert
* Depend now works with RobotControl too
*
* Revision 1.12  2003/05/11 17:03:00  risler
* added location.cfg
*
* Revision 1.11  2003/04/16 17:46:46  juengel
* clean up in ColorTable64DlgBar
*
* Revision 1.10  2003/04/15 15:52:07  risler
* DDD GO 2003 code integrated
*
* Revision 1.10  2003/04/08 21:34:43  Charlie
* bugfix
*
* Revision 1.9  2003/04/07 21:13:09  Charlie
* added more functionality to ct64 dialog
*
* Revision 1.9  2003/04/14 16:17:07  loetzsch
* ATH after GermanOpen CVS merge
*
* improved dialog
*
* Revision 1.3  2003/04/10 12:08:01  juengel
* Activated showSelectedColorClassOnly
*
* Revision 1.2  2003/04/10 07:57:37  juengel
* no message
*
* Revision 1.8  2003/04/01 14:00:42  juengel
* color table is sent to gui
*
* Revision 1.7  2003/03/31 17:02:55  osterhues
* Disabled sendLocal() at init because it has overridden color table selection from coltable.* file
*
* Revision 1.6  2003/03/05 17:09:27  loetzsch
* redesign of the queues and debug key tables in RobotControl
*
* Revision 1.5  2002/10/04 10:22:16  loetzsch
* Invalidate and RedrawWindow are only called in the handleMessage
* function when the dialog is visible
*
* Revision 1.4  2002/09/20 23:28:34  juengel
* Moved instance of DebugDrawingManager to RobotControlMainFrame.
*
* Revision 1.3  2002/09/18 16:35:13  loetzsch
* made GT2003 compilable,
* rechanged message ids
*
* Revision 1.2  2002/09/17 23:55:23  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.1  2002/09/10 15:49:04  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.4  2002/08/22 15:20:10  loetzsch
* debug messages to the local processes are now sent in a better synchronized way.
*
* Revision 1.3  2002/08/07 10:50:21  loetzsch
* - removed direct access to the dialogs
* - faster starting of RobotControl application and main window
* - general clean up
*
* Revision 1.2  2002/07/23 13:43:36  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - empty constructors in bars / dialogs
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
* Revision 1.1.1.1  2002/05/10 12:40:20  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.23  2002/05/04 12:22:55  juengel
* Moved global instance of the ColorTable64 for segmented image views from CRobotControlApp to DebugDrawingManager.cpp
*
* Revision 1.22  2002/05/04 11:40:40  juengel
* Added << endl after colorTable.
*
* Revision 1.21  2002/04/23 17:45:17  loetzsch
* - splitted debugKeyTable into debugKeyTableForRobot and debugKeyTableForLocalProcesses
* - removed Modules Toolbar
* - removed access for dialogs and toolbars to solutionRequest.
* - changed access for dialogs and toolbars to debug queues
* - removed the instance of SolutionRequest in CRobotControlApp
* - only the log player, local processes and the robot put messages into queueToRobotControl
*
* Revision 1.20  2002/04/16 15:46:36  dueffert
* no message
*
* Revision 1.2  2002/04/12 02:06:06  dueffert
* we do not want <<endl after coltable64, it has its own stopsign
*
* Revision 1.19  2002/03/22 16:56:39  loetzsch
* added a send to robot button to the color table 64 tool
*
* Revision 1.18  2002/02/12 16:34:49  risler
* finished MofTester
*
* Revision 1.17  2002/02/11 00:53:33  loetzsch
* ::Paint Methoden in ::paint umbenannt
*
* Revision 1.16  2002/02/07 16:27:12  loetzsch
* colortables are sent twice on every change of the color table
* for synchronization with the processes
*
* Revision 1.15  2002/01/26 18:09:20  juengel
* DebugDrawingManager umstrukturiert.
*
* Revision 1.14  2002/01/22 14:55:48  juengel
* Save Dialog speichert nun und ffnet nicht mehr.
*
* Revision 1.13  2002/01/19 21:36:26  risler
* added HeadMotionTester, HeadControlSelector
*
* Revision 1.12  2002/01/15 13:50:34  bach
* Undo button in color table tool added (Joscha)
*
* Revision 1.11  2002/01/08 15:50:10  juengel
* Zeichnungsmethoden fr Image und colorClassImage  knnen jetzt skalieren.
*
* Revision 1.10  2002/01/04 14:35:18  juengel
* no message
*
* Revision 1.9  2001/12/22 20:35:35  loetzsch
* removed ColorTable::segmentImage, replaced by
* ColorTable64::generateColorClassImage
*
* Revision 1.8  2001/12/22 08:25:30  roefer
* Enhanced constructor of class File, removed second one
*
* Revision 1.7  2001/12/21 12:38:37  juengel
* Jetzt kann auch in das segmented Image geklickt werden.
* Immer wenn sich die Farbtabelle ndert, wird sie in die QueueToRobotControl gestellt.
*
* Revision 1.6  2001/12/20 19:19:52  juengel
* Slider eingebaut
*
* Revision 1.5  2001/12/20 14:36:34  juengel
* Methoden zum Zeichnen der ColorTable zeichnen jetzt in Bitmaps und nicht mehr in CDCs
*
* Revision 1.4  2001/12/19 14:04:47  loetzsch
* In ColorTables Pointer gegen Referenzen ausgetauscht,
* Funktion segmentImage hinzugefgt und implementiert
*
* Revision 1.3  2001/12/19 10:45:21  juengel
* An der Anzeige gefrickelt.
*
* Revision 1.2  2001/12/17 19:22:58  juengel
* Grundfunktionalitt hergestellt.
*
* Revision 1.1  2001/12/15 19:26:13  juengel
* ColorTable, ColorTable64, ColorTable64DlgBar hinzugefgt.
*
*
*/
