/**
* @file ColorSpaceDlgBar.cpp
* 
* Implementation of class CColorSpaceDlgBar.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/

#include "StdAfx.h"
#include "ColorSpaceDlgBar.h"

#include <gl/gl.h>
#include <gl/glu.h>
#include "Visualization/OpenGLMethods.h"
#define HAVE_BOOLEAN
#include "Representations/Perception/JPEGImage.h"
#include "Platform/Sensors.h"
#include "Tools/Math/Common.h"
#include "RobotControlDebugKeyTables.h"

CColorSpaceDlgBar::CColorSpaceDlgBar()
: CRobotControlDialogBar(IDD),
  m_hGLContext(NULL),
  m_xRotate(pi / 3.0),
  m_yRotate(0),
  m_zRotate(0),
  colorModel(yuv),
  backGroundColor(gray),
  paintFullCube(true),
  showColorTable(false),
  showColorTableCuboids(false),
  showColorTableReferenceColor(false),
  width_y(image.cameraInfo.resolutionHeight),
  y1(0),
  width_x(image.cameraInfo.resolutionWidth),
  x1(0),
  m_LeftButtonDown(false),
  initialized(false)
{
  //{{AFX_DATA_INIT(CColorSpaceDlgBar)
		// NOTE: the ClassWizard will add member initialization here
  //}}AFX_DATA_INIT
}

void CColorSpaceDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDynamicBarDlg::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CColorSpaceDlgBar)
  DDX_Control(pDX, IDC_CSPACE_STATIC4, m_static4);
  DDX_Control(pDX, IDC_CSPACE_STATIC3, m_static3);
  DDX_Control(pDX, IDC_CSPACE_STATIC2, m_static2);
  DDX_Control(pDX, IDC_CSPACE_STATIC1, m_static1);
  DDX_Control(pDX, IDC_CSPACE_SLIDER4, m_slider4);
  DDX_Control(pDX, IDC_CSPACE_SLIDER3, m_slider3);
  DDX_Control(pDX, IDC_CSPACE_SLIDER2, m_slider2);
  DDX_Control(pDX, IDC_CSPACE_SLIDER1, m_slider1);
  //}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CColorSpaceDlgBar, CDynamicBarDlg)
ON_WM_CONTEXTMENU()
//{{AFX_MSG_MAP(CColorSpaceDlgBar)
ON_WM_PAINT()
ON_WM_CREATE()
ON_WM_DESTROY()
ON_WM_SIZE()
ON_WM_LBUTTONDOWN()
ON_WM_LBUTTONUP()
ON_WM_MOUSEMOVE()
ON_WM_CONTEXTMENU()
ON_WM_HSCROLL()
//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CColorSpaceDlgBar message handlers

BOOL CColorSpaceDlgBar::OnInitDialog()
{
  CDynamicBarDlg::OnInitDialog();
  
  m_slider1.SetRange(0, image.cameraInfo.resolutionHeight);
  m_slider2.SetRange(1, image.cameraInfo.resolutionHeight);
  m_slider3.SetRange(0, image.cameraInfo.resolutionWidth);
  m_slider4.SetRange(1, image.cameraInfo.resolutionWidth);
  
  setSliders();
  return TRUE;
}

void CColorSpaceDlgBar::OnPaint() 
{
  wglMakeCurrent(m_hDC,m_hGLContext);

  if (!initialized)
  {
    paintColorTable();
  
    OpenGLMethods::paintImagePixelsToOpenGLList(
      image, colorModel, selection - 1, displayMode==polygons, OpenGLMethods::colorSpacePixelList,
      x1, x1 + width_x, y1, y1 + width_y);
    initialized=true;
  }
  
  // ** Draw scene **
  CPaintDC dc(this);
  renderScene();
  glFinish();
  SwapBuffers(dc.m_hDC);
  wglMakeCurrent(NULL,NULL);
}


int CColorSpaceDlgBar::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
  //  HDC hDC = ::GetDC(m_hWnd);
  m_hDC = ::GetDC(GetSafeHwnd());
  if(OpenGLMethods::setWindowPixelFormat(m_hDC) == false)
    return 0;
  
  if(OpenGLMethods::createViewGLContext(m_hDC, m_hGLContext) == false)
    return 0;
  
  // Antialiasing
  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
  glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);
  glLineWidth(1.5); // required
  glPointSize(2.5);
  glPolygonMode(GL_FRONT,GL_LINE);
  glPolygonMode(GL_BACK,GL_LINE);
  glShadeModel(GL_SMOOTH);
  
  selection = yuvUsage;
  displayMode = pixels;

  paintBoundingCube();
  //  wglMakeCurrent(NULL,NULL);
  
  return 0;
}

void CColorSpaceDlgBar::OnDestroy() 
{
  CDynamicBarDlg::OnDestroy();
  
  if(wglGetCurrentContext() != NULL)
    wglMakeCurrent(NULL,NULL);
  
  if(m_hGLContext != NULL)
  {
    wglDeleteContext(m_hGLContext);
    m_hGLContext = NULL;
  }
  
}

void CColorSpaceDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  CDynamicBarDlg::OnSize(nType, cx, cy);
  
  wglMakeCurrent(m_hDC,m_hGLContext);
  
  GLsizei width = cx;
  GLsizei height = cy;
  GLdouble aspect;
  
  if(cy==0)
    aspect = (GLdouble)width;
  else
    aspect = (GLdouble)width/(GLdouble)height;
  
  glViewport(0,0,width,height);
  
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  
  gluPerspective(30,aspect,1,100.0);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  
  glDrawBuffer(GL_BACK);
  glEnable(GL_DEPTH_TEST);
  
  InvalidateRect(NULL,FALSE);
}



void CColorSpaceDlgBar::renderScene()
{
  if(backGroundColor == white)
  {
    glClearColor(1.0, 1.0, 1.0, 1.0);
  }
  if(backGroundColor == gray)
  {
    glClearColor(0.5, 0.5, 0.5, 1.0);
  }
  
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  
  glPushMatrix();
  if (selection == yuvUsage)
    glTranslated(0.0,0.0,-8.0);
  else
    glTranslated(0.0,0.0,-5.0);
  
  glRotated(m_xRotate, 1.0, 0.0, 0.0);
  glRotated(m_yRotate, 0.0, 1.0, 0.0);
  glRotated(m_zRotate, 0.0, 0.0, 1.0);
  glCallList(OpenGLMethods::colorSpaceCubeList); // cube
  if(displayMode != nothing)
    glCallList(OpenGLMethods::colorSpacePixelList); // clouds

  if(showColorTable == true && colorModel == yuv && selection == yuvUsage)
    glCallList(OpenGLMethods::colorSpaceColorTableList); //colortable
  
  if(showColorTableCuboids == true && colorModel == yuv)
    glCallList(OpenGLMethods::colorSpaceColorTableCuboidsList); // colortable
  
  if(showColorTableReferenceColor == true && colorModel == yuv && selection == yuvUsage)
    glCallList(OpenGLMethods::colorSpaceColorTableReferenceColorList); //colortable
  glPopMatrix();
}

// MOUSE INTERACTION

void CColorSpaceDlgBar::OnLButtonDown(UINT nFlags, CPoint point) 
{
  m_LeftButtonDown = TRUE;
  m_LeftDownPos = point;
  SetCapture();
  CDynamicBarDlg::OnLButtonDown(nFlags, point);
}
void CColorSpaceDlgBar::OnLButtonUp(UINT nFlags, CPoint point) 
{
  m_LeftButtonDown = FALSE;
  ReleaseCapture();
  CDynamicBarDlg::OnLButtonUp(nFlags, point);
}


void CColorSpaceDlgBar::OnMouseMove(UINT nFlags, CPoint point) 
{
  if(m_LeftButtonDown)
  {
    CSize rotate = m_LeftDownPos - point;
    m_LeftDownPos = point;
    //    m_yRotate -= rotate.cx;
    m_zRotate -= rotate.cx;
    m_xRotate -= rotate.cy;
    InvalidateRect(NULL,FALSE);
  }
  CDynamicBarDlg::OnMouseMove(nFlags, point);
}


bool CColorSpaceDlgBar::handleMessage(InMessage& message) 
{
  if (this->IsWindowVisible())
  {
    if (message.getMessageID() == idImage || message.getMessageID() == idJPEGImage)
    {
      if(message.getMessageID() == idImage)
        message.bin >> image;
      else
      {
        JPEGImage jpeg;
        message.bin >> jpeg;
        jpeg.toImage(image);
      }

      if(width_x == cameraResolutionWidth_ERS210 && image.cameraInfo.resolutionWidth == cameraResolutionWidth_ERS7)
        width_x = cameraResolutionWidth_ERS7;
      if(width_y == cameraResolutionHeight_ERS210 && image.cameraInfo.resolutionHeight == cameraResolutionHeight_ERS7)
        width_y = cameraResolutionHeight_ERS7;

      if(width_x == cameraResolutionWidth_ERS7 && image.cameraInfo.resolutionWidth == cameraResolutionWidth_ERS210)
        width_x = cameraResolutionWidth_ERS210;
      if(width_y == cameraResolutionHeight_ERS7 && image.cameraInfo.resolutionHeight == cameraResolutionHeight_ERS210)
        width_y = cameraResolutionHeight_ERS210;


      m_slider1.SetRange(0, image.cameraInfo.resolutionHeight);
      m_slider2.SetRange(1, image.cameraInfo.resolutionHeight);
      m_slider3.SetRange(0, image.cameraInfo.resolutionWidth);
      m_slider4.SetRange(1, image.cameraInfo.resolutionWidth);
      
      setSliders();
      paintBoundingCube();

      wglMakeCurrent(m_hDC,m_hGLContext);

      OpenGLMethods::paintImagePixelsToOpenGLList(
        image, colorModel, selection - 1, displayMode==polygons , OpenGLMethods::colorSpacePixelList,
        x1, x1 + width_x, y1, y1 + width_y);
      
      InvalidateRect(NULL,FALSE);
      return true;
    }
    else if (message.getMessageID() == idParametersFromGridImageProcessor2)
    {
      message.bin >> gridImageProcessor2Parameters;
      wglMakeCurrent(m_hDC,m_hGLContext);

      paintColorTable();

      InvalidateRect(NULL,FALSE);
    
      return true;
    }
    else if(message.getMessageID() == idColorTable64)
    {
      message.bin >> colorTable64;

      wglMakeCurrent(m_hDC,m_hGLContext);
      paintColorTable();

      InvalidateRect(NULL,FALSE);
      return true;
    }
    else if(message.getMessageID() == idColorTableCuboids)
    {
      message.bin >> colorTableCuboids;

      wglMakeCurrent(m_hDC,m_hGLContext);

      paintColorTable();

      InvalidateRect(NULL,FALSE);
    
      return true;
    }
    else if(message.getMessageID() == idColorTableReferenceColor)
    {
      message.bin >> colorTableReferenceColor;

      wglMakeCurrent(m_hDC,m_hGLContext);

      paintColorTable();

      InvalidateRect(NULL,FALSE);
    
      return true;
    }
    else
      return false;
  }
  else 
    return false;
}

void CColorSpaceDlgBar::paintColorTable()
{
  if(selection == yuvUsage)
  {
    if(showColorTable)
    {
      OpenGLMethods::paintColorTable(colorTable64, OpenGLMethods::colorSpaceColorTableList);
    }
    if(showColorTableCuboids)
    {
      OpenGLMethods::paintColorTableCuboids(
        colorTableCuboids,
        OpenGLMethods::colorSpaceColorTableCuboidsList);
    }
    if(showColorTableReferenceColor)
    {
      OpenGLMethods::paintColorTableReferenceColor(
        colorTableReferenceColor,
        OpenGLMethods::colorSpaceColorTableReferenceColorList);
    }
  }
}

void CColorSpaceDlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
  CMenu menu;
  VERIFY( menu.CreatePopupMenu() );
  
  UINT flags = 0;

  if(colorModel == yuv) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2000,"YUV");

  if(colorModel == rgb) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2001,"RGB");

  if(colorModel == tsl) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2002,"TSL");

  if(colorModel == hsi) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2003,"HSI");

  menu.AppendMenu(MF_SEPARATOR );
  
  if(selection == yuvUsage) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1000,"Color Space Usage");
  
  menu.AppendMenu(MF_SEPARATOR );
  
  if(selection == yUsage) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1001,"Channel 1 Color Space Usage");
  
  if(selection == uUsage) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1002,"Channel 2 Color Space Usage");
  
  if(selection == vUsage) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1003,"Channel 3 Color Space Usage");
  
  menu.AppendMenu(MF_SEPARATOR );
  
  if(displayMode == pixels) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1100,"show pixels");
  
  if(displayMode == polygons) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1101,"show polygons");

  if(displayMode == nothing) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1102,"show nothing");

  menu.AppendMenu(MF_SEPARATOR );
  
  if(backGroundColor == white) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2100,"white background");
  
  if(backGroundColor == gray) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2101,"gray background");

  menu.AppendMenu(MF_SEPARATOR );
  
  if(paintFullCube == true) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2200,"cube");

  if(paintFullCube == false) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2201,"rectangle");

  menu.AppendMenu(MF_SEPARATOR );
  
  if(showColorTable == true) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2299,"show color table '64'");

  if(showColorTableCuboids == true) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2300,"show color table 'cuboids'");

  if(showColorTableReferenceColor == true) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,2301,"show color table 'reference-color'");

  UINT nID = menu.TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
    point.x, point.y, this );
  
  switch(nID)
  {
  case 1000:
    selection = yuvUsage;
    break;
  case 1001:
    selection = yUsage;
    break;
  case 1002:
    selection = uUsage;
    break;
  case 1003:
    selection = vUsage;
    break;
  case 1100:
    displayMode = pixels;
    break;
  case 1101:
    displayMode = polygons;
    break;
  case 1102:
    displayMode = nothing;
    break;
  case 2000:
    colorModel = yuv;
    break;
  case 2001:
    colorModel = rgb;
    break;
  case 2002:
    colorModel = tsl;
    break;
  case 2003:
    colorModel = hsi;
    break;
  case 2100:
    backGroundColor = white;
    break;
  case 2101:
    backGroundColor = gray;
    break;
  case 2200:
    paintFullCube = true;
    break;
  case 2201:
    paintFullCube = false;
    break;
  case 2299:
    showColorTable = !showColorTable;
    
    if(showColorTable) {
      getDebugKeyTables().forPhysicalRobots.set(DebugKeyTable::sendColorTable64, DebugKey::always);
      getDebugKeyTables().forSimulatedRobots.set(DebugKeyTable::sendColorTable64, DebugKey::always);
    }
    else {
      getDebugKeyTables().forPhysicalRobots.set(DebugKeyTable::sendColorTable64, DebugKey::disabled);
      getDebugKeyTables().forSimulatedRobots.set(DebugKeyTable::sendColorTable64, DebugKey::disabled);
    }
    
    getDebugKeyTables().sendForPhysicalRobots();
    getDebugKeyTables().sendForSimulatedRobots();
    
    break;
  case 2300:
    showColorTableCuboids = !showColorTableCuboids;

    if(showColorTableCuboids) {
      getDebugKeyTables().forPhysicalRobots.set(DebugKeyTable::sendColorTableCuboids, DebugKey::always);
      getDebugKeyTables().forSimulatedRobots.set(DebugKeyTable::sendColorTableCuboids, DebugKey::always);
    }
    else {
      getDebugKeyTables().forPhysicalRobots.set(DebugKeyTable::sendColorTableCuboids, DebugKey::disabled);
      getDebugKeyTables().forSimulatedRobots.set(DebugKeyTable::sendColorTableCuboids, DebugKey::disabled);
    }
    
    getDebugKeyTables().sendForPhysicalRobots();
    getDebugKeyTables().sendForSimulatedRobots();
    break;
  case 2301:
    showColorTableReferenceColor = !showColorTableReferenceColor;

    if(showColorTableReferenceColor) {
      getDebugKeyTables().forPhysicalRobots.set(DebugKeyTable::sendColorTableReferenceColor, DebugKey::always);
      getDebugKeyTables().forSimulatedRobots.set(DebugKeyTable::sendColorTableReferenceColor, DebugKey::always);
    }
    else {
      getDebugKeyTables().forPhysicalRobots.set(DebugKeyTable::sendColorTableReferenceColor, DebugKey::disabled);
      getDebugKeyTables().forSimulatedRobots.set(DebugKeyTable::sendColorTableReferenceColor, DebugKey::disabled);
    }
    
    getDebugKeyTables().sendForPhysicalRobots();
    getDebugKeyTables().sendForSimulatedRobots();
    break;
  }
  
  wglMakeCurrent(m_hDC,m_hGLContext);
  
  paintBoundingCube();
  paintColorTable();
  
  OpenGLMethods::paintImagePixelsToOpenGLList(
    image, colorModel, selection - 1, displayMode==polygons, OpenGLMethods::colorSpacePixelList,
    x1, x1 + width_x, y1, y1 + width_y);
  InvalidateRect(NULL, false);
}

void CColorSpaceDlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  y1 = m_slider1.GetPos();
  width_y = m_slider2.GetPos();
  x1 = m_slider3.GetPos();
  width_x = m_slider4.GetPos();
  
  if(y1 + width_y > image.cameraInfo.resolutionHeight) y1 = image.cameraInfo.resolutionHeight - width_y;
  if(x1 + width_x > image.cameraInfo.resolutionWidth) x1 = image.cameraInfo.resolutionWidth - width_x;
  
  setSliders();
  wglMakeCurrent(m_hDC,m_hGLContext);

  OpenGLMethods::paintImagePixelsToOpenGLList(
    image, colorModel, selection - 1, displayMode==polygons , OpenGLMethods::colorSpacePixelList,
    x1, x1 + width_x, y1, y1 + width_y);
  InvalidateRect(NULL,FALSE);
  
  CDynamicBarDlg::OnHScroll(nSBCode, nPos, pScrollBar);
}

void CColorSpaceDlgBar::setSliders()
{
  m_slider1.SetPos(y1);
  m_slider2.SetPos(width_y);
  m_slider3.SetPos(x1);
  m_slider4.SetPos(width_x);
  
  CString string;
  
  string.Format("%d", y1);
  m_static1.SetWindowText(string);
  string.Format("%d", width_y);
  m_static2.SetWindowText(string);
  string.Format("%d", x1);
  m_static3.SetWindowText(string);
  string.Format("%d", width_x);
  m_static4.SetWindowText(string);
}

void CColorSpaceDlgBar::paintBoundingCube()
{
  if(selection == yuvUsage)
  {
    OpenGLMethods::paintCubeToOpenGLList(
      256,256,256, 
      OpenGLMethods::colorSpaceCubeList, 
      paintFullCube,
      127, //scale
      -127,-127,-127, // offsets
      0,0,0 // black
      );
  }
  else
  {
    OpenGLMethods::paintCubeToOpenGLList(
      image.cameraInfo.resolutionWidth, image.cameraInfo.resolutionHeight, 128, 
      OpenGLMethods::colorSpaceCubeList, paintFullCube,
      127, //scale
      -image.cameraInfo.resolutionWidth / 2, -image.cameraInfo.resolutionHeight / 2, -65, // offsets
      0,0,0 // black
      );
  }
}

void CColorSpaceDlgBar::OnConfigurationLoad(CString sectionName)
{
  paintFullCube                = AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "paintFullCube",0)==1;
  showColorTable               = AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "showColorTable",0)==1;
  showColorTableCuboids        = AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "showColorTableCuboids",0)==1;
  showColorTableReferenceColor = AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "showColorTableReferenceColor",0)==1;

  colorModel      = (ColorModel)AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "colorModel",0);
  selection       = (Selection)AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "selection",0);
  displayMode     = (DisplayMode)AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "displayMode",0);
  backGroundColor = (BackGroundColor)AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "backGroundColor",0);

  m_xRotate = AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "m_xRotate",1 * 1000 + 1000000);
  m_yRotate = AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "m_yRotate",1000000);
  m_zRotate = AfxGetApp()->GetProfileInt(sectionName + "\\ColorSpace", "m_zRotate",1000000);

  m_xRotate -= 1000000; m_xRotate /= 1000.0;
  m_yRotate -= 1000000; m_yRotate /= 1000.0;
  m_zRotate -= 1000000; m_zRotate /= 1000.0;

  setSliders();
  wglMakeCurrent(m_hDC,m_hGLContext);
  
  paintBoundingCube();
  InvalidateRect(NULL, false);
}


void CColorSpaceDlgBar::OnConfigurationSave(CString sectionName)
{
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "paintFullCube",paintFullCube?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "showColorTable",showColorTable?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "showColorTableCuboids",showColorTableCuboids?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "showColorTableReferenceColor",showColorTableReferenceColor?1:0);

  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "m_xRotate", (int)(m_xRotate * 1000 + 1000000));
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "m_yRotate", (int)(m_yRotate * 1000 + 1000000));
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "m_zRotate", (int)(m_zRotate * 1000 + 1000000));

  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "y1", y1);
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "width_y", width_y);
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "x1", x1);
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "width_x", width_x);

  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "colorModel", colorModel);
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "selection", selection);
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "displayMode", displayMode);
  AfxGetApp()->WriteProfileInt(sectionName + "\\ColorSpace", "backGroundColor", backGroundColor);
}

/*
* Change log :
* $Log: ColorSpaceDlgBar.cpp,v $
* Revision 1.9  2004/01/31 00:07:45  dueffert
* dialog now works with nvidia cards too
*
* Revision 1.8  2004/01/10 12:39:57  juengel
* Selections are stored dependent on configuration.
*
* Revision 1.7  2004/01/04 12:08:28  juengel
* adaptation for larger images
*
* Revision 1.6  2003/12/15 11:46:14  juengel
* Introduced CameraInfo
*
* Revision 1.5  2003/10/30 18:29:15  juengel
* Improved paint of ColorTableCuboids.
*
* Revision 1.4  2003/10/29 13:24:42  juengel
* Added addCuboidToColorClass to ColorTable64
*
* Revision 1.3  2003/10/23 07:21:20  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.2  2003/10/12 11:50:30  juengel
* Added DebugKey sendColorTable64.
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.6  2003/09/26 15:28:23  juengel
* Renamed DataTypes to representations.
*
* Revision 1.5  2003/08/18 11:52:31  juengel
* Improved display of colorTableReferenceColor.
*
* Revision 1.4  2003/07/21 13:43:16  juengel
* Moved ColorTableReferenceColor and GoalRecognizer to ImageProcessorTools.
*
* Revision 1.3  2003/07/19 19:31:34  juengel
* Improved AutoCalibration
*
* Revision 1.2  2003/07/07 07:38:04  roefer
* JPEG support added
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.17  2003/05/11 23:46:32  dueffert
* Depend now works with RobotControl too
*
* Revision 1.16  2003/05/05 12:18:08  juengel
* Changes in GIP2Parameters and ColorTableReferenceColor.
*
* Revision 1.15  2003/04/03 18:50:09  juengel
* Improved AutoCalibration.
*
* Revision 1.14  2003/04/01 19:28:12  juengel
* Added displayMode nothing
*
* Revision 1.13  2003/04/01 14:00:25  juengel
* Added visualization of color tables.
*
* Revision 1.12  2003/03/04 14:53:10  juengel
* Added handling for messges of type idParametersFromGridImageProcessor2
*
* Revision 1.11  2003/02/27 12:56:56  juengel
* Added sub cube functionality.
*
* Revision 1.10  2003/02/24 22:33:04  juengel
* ColorSpace Dialog can convert images to RGB, HSI and TSL.
*
* Revision 1.9  2002/10/10 16:18:39  loetzsch
* some minor improvements and better doxygen comments
*
* Revision 1.8  2002/10/10 13:09:50  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
* Revision 1.7  2002/10/04 10:22:16  loetzsch
* Invalidate and RedrawWindow are only called in the handleMessage
* function when the dialog is visible
*
* Revision 1.6  2002/09/22 18:40:51  risler
* added new math functions, removed GTMath library
*
* Revision 1.5  2002/09/20 23:28:34  juengel
* Moved instance of DebugDrawingManager to RobotControlMainFrame.
*
* Revision 1.4  2002/09/18 16:35:13  loetzsch
* made GT2003 compilable,
* rechanged message ids
*
* Revision 1.3  2002/09/17 23:55:23  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.2  2002/09/15 07:52:27  juengel
* Added some sliders in ColorSpaceDlgBar.
*
* Revision 1.1  2002/09/10 15:49:04  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.2  2002/07/23 13:43:36  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - empty constructors in bars / dialogs
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
* Revision 1.1.1.1  2002/05/10 12:40:20  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.3  2002/02/12 13:05:34  juengel
* Added OpenGLMethods to project.
*
* Revision 1.2  2002/02/10 12:19:35  juengel
* Polygon surface added.
*
* Revision 1.1  2002/02/09 15:13:37  juengel
* ColorSpaceDlgBar added.
*
*
*/
