/**
* @file Representations/WLan/TeamMessageCollection.cpp
*
* Implementation of class TeamMessageCollection.
*
* @author <A href=mailto:sebastian.schmidt@udo.edu>Sebastian Schmidt</A>
* @author <A href=mailto:m_wachter@gmx.de>Michael Wachter</A>
*/

#include "TeamMessageCollection.h"
#include "Platform/SystemCall.h"
#include "Tools/Player.h"
#include "Tools/Debugging/Debugging.h"

TeamMessageCollection::TeamMessageCollection()
{
  lastSendingTimeStamp = 0;
  for (int a=0; a< maxNumberOfTeamMessages; a++)
  {
    incomingTimeStamp[a] = teamMessageTimeStamp[a] = 0;
  }
  
  int i;
   
  // all ringbuffer elements are set to zero. 
  for(int j=0;j< maxNumberOfTeamMessages ;j++) 
  {
    for(int i=0;i<100;i++) offsets[j].add(0);
    offsets[j].init();
    timeOffsetSum[j]=0;
    offsets[j].add(0);
  }
  
  messagesSent = false;
  for(i=0;i< maxNumberOfTeamMessages ;i++) 
  {
    teamMessages[i] = NULL;
  }

  for(i=0;i< maxNumberOfTeamMessages ;i++) 
  {
	teamMessagesForActualTeamMessages[i] = 0;
  }

  numberOfInTeamMessages = 0;
  numberOfOutTeamMessages = 0;
  lastSendTime=0;
  sendNow = false;
  delay = 100;
 }

void TeamMessageCollection::processTimeStamps()
{
  signed long actualOffset;
  for (int i=0; i < numberOfInTeamMessages; i++)
  {
  /*  Offset = (1/2) * ( T2 - T1 + T3 - T4 )  =  T5 - T4
  *  
  *  T1 = LastReceivedTimeStamp
  *  T2 = IncomimgTimeStamp
  *  T3 = TimeStamp
  *  T4 = System-Time
  *  T5 = System-Time of the other Robot
  *  
    */
    
    actualOffset= ( (signed long)teamMessages[i]->incomingTimeStamp -
      (signed long)teamMessages[i]->lastReceivedTimeStamp +
      (signed long)teamMessages[i]->timeStamp -
      (signed long)SystemCall::getCurrentSystemTime() ) / 2 ;
    
    // calculating the average of the values in the ringBuffer
    timeOffsetSum[i] += actualOffset;
    timeOffsetSum[i] -= offsets[i].getEntry(-1);
    
    (const_cast<TeamMessage*>(teamMessages[i]))->timeOffset =
      //((TeamMessage*)teamMessages[i])->timeOffset =
      timeOffsetSum[i] / offsets[i].getNumberOfEntries();
    offsets[i].add(actualOffset);
    
    // Setting timeStamps in teamMessages
    outTeamMessages[i]->incomingTimeStamp = SystemCall::getCurrentSystemTime();
    outTeamMessages[i]->lastReceivedTimeStamp = teamMessages[i]->timeStamp;
    
  }
  INFO(sendTimeOffsets,idText,text,"TimeOffsets : " 
    << teamMessages[0]->timeOffset << " " << teamMessages[0]->playerNumberOfSender <<" "
    << teamMessages[1]->timeOffset << " " << teamMessages[1]->playerNumberOfSender <<" "
    << teamMessages[2]->timeOffset << " " << teamMessages[2]->playerNumberOfSender);
}


void TeamMessageCollection::processMessages()
{   

  numberOfTeamMessages = 0;
  for(int a=0;a< numberOfInTeamMessages;a++) {
    
    // Reset sendXxx fields in the TeamMessages if they are send
    if (messagesSent) {
      ((TeamMessage*) outTeamMessages[a])->sendSeenBallPosition
        = outTeamMessages[a] -> sendBehaviorTeamMessage 
        = outTeamMessages[a] -> sendRobotPose 
        = false;
      messagesSent = false;
    }
    
    // Set the teamMessagesForActualTeamMessages so that operator[] only returns actual 
    // teamMessages. 
    if (teamMessages[a]->isActual()) {
      teamMessagesForActualTeamMessages[numberOfTeamMessages++] = a;
    }
  }
  
  processTimeStamps();
}

void TeamMessageCollection::setOutTeamMessages(TeamMessage& tm)
{
  outTeamMessages[numberOfOutTeamMessages] = &tm;
  numberOfOutTeamMessages++;  

}

void TeamMessageCollection::setInTeamMessages(TeamMessage& tm)
{
  teamMessages[numberOfInTeamMessages] = &tm;
  numberOfInTeamMessages++;
} 


void TeamMessageCollection::send(const RobotPose& robotPose,bool sendNow) const
{
  for (int i=0;i< numberOfOutTeamMessages;i++)
  {
     outTeamMessages[i]->robotPose = robotPose ;
     outTeamMessages[i]->sendRobotPose = true;
  }
  this->sendNow |= sendNow;
}

void TeamMessageCollection::send(const SeenBallPosition& seenBallPosition, bool sendNow) const
{
  for (int i=0;i< numberOfOutTeamMessages;i++)
  {
      outTeamMessages[i]->seenBallPosition = seenBallPosition;
      outTeamMessages[i]->sendSeenBallPosition = true;
  }
  this->sendNow |= sendNow;
}

void TeamMessageCollection::send(const BehaviorTeamMessage& behaviorTeamMessage,bool sendNow) const
{
  for (int i=0;i< numberOfOutTeamMessages;i++)
  {
      outTeamMessages[i]->behaviorTeamMessage = behaviorTeamMessage;
      outTeamMessages[i]->sendBehaviorTeamMessage = true;
  }
  this->sendNow |= sendNow;
}

bool TeamMessageCollection::processOutMessages() {
  
  // is te delay over ?
  sendNow |= ((signed)(SystemCall::getCurrentSystemTime() - (signed) lastSendTime) > delay);
  if (sendNow) 
  {
    // set timestamps for all outgoing TeamMessages
    for(int a=0;a< numberOfOutTeamMessages;a++) {
      outTeamMessages[a]->timeStamp = SystemCall::getCurrentSystemTime();
    }

    messagesSent = true;
    lastSendTime = SystemCall::getCurrentSystemTime();
    sendNow = false;
    return (true);
  }
  else return (false);
}


void TeamMessageCollection::setDelay(int delay)
{
  this->delay = delay;
}

int TeamMessageCollection::getDelay() const
{
  return (delay);
}


/*
* Change log :
*
* $Log: TeamMessageCollection.cpp,v $
* Revision 1.4  2004/05/14 14:12:08  wachter
* - Added communication support for 5 robots
* - rewrote parts of team-communication to be faster and more stable
*
* Revision 1.3  2004/02/18 12:09:20  hamerla
* add playernumber to Debugkey Timeoffset
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/07 10:09:36  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/26 15:27:27  juengel
* Renamed DataTypes to representations.
*
* Revision 1.2  2003/07/02 19:14:23  loetzsch
* bug fixes, removed unused functions
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/05/02 12:57:13  loetzsch
* TeamMessage now contains a SeenBallPosition instead of a BallPercept
*
* Revision 1.4  2003/03/24 12:10:45  wachter
* Fixed warning.
*
* Revision 1.3  2003/02/28 17:02:55  wachter
* Reenabled trainer-syncronisation for RuhrpottHellhound-Behavior
*
* Revision 1.2  2003/02/25 12:55:40  wachter
* made some attributes muteable
*
* Revision 1.1  2003/02/18 13:24:36  wachter
* added new TeamMessageCollection and TeamMessage
*
*
*/





