/**
 * @file SensorData.h
 * 
 * Declaration of class SensorData
 */ 

#ifndef __SensorData_h_
#define __SensorData_h_

#include "Tools/Streams/InOut.h"

/**
* A class representing a sensor data vector. 
*/
class SensorData
{
public:
  /**
  * constructs a SensorData.
  */ 
  SensorData();

  /**
  * deconstructs a SensorData.
  */
  ~SensorData();

  enum sensors 
  {
    // ERS-210
    headTilt, 
    headPan, 
    headRoll,
    headBack, 
    headFront,
    psd, 
    mouth,
    chin,
    legFL1, legFL2, legFL3, pawFL,
    legHL1, legHL2, legHL3, pawHL,
    legFR1, legFR2, legFR3, pawFR,
    legHR1, legHR2, legHR3, pawHR,
    tailPan, tailTilt,
    thermo,
    back,
    accelerationX, accelerationY, accelerationZ,
    numOfSensor_ERS210,
    
    // ERS-7
    head = numOfSensor_ERS210,
    headPsdFar,
    bodyPsd,
    numOfSensor_ERS7,

    // synonyms
    headTilt1 = headTilt,
    headTilt2 = headRoll,
    headPsdNear = psd,
    backF = headFront, 
    backM = back,
    backR = headBack,
    wlan = thermo
  };

  /** The sensor data value */
  long data[numOfSensor_ERS7];

  /** Sensor data reference values.
   * Only valid for joint sensors. */
  long refValue[numOfSensor_ERS7];

  /** The frame number of that sensor data. */
  unsigned long frameNumber; 

  void init();

  /** returns the name of a sensor */
  static const char* getSensorName(sensors sensor)
  {
    switch (sensor)
    {
    case headTilt: return "headTilt(1)";
    case headRoll: return "headRoll/Tilt2";
    case headBack: return "head/backRear";
    case headFront: return "head/backFront";
    case psd: return "headPsd(Near)";
    case thermo: return "thermo/wlan";
    case back: return "back(Middle)";
    default: return getSensorNameERS7(sensor);
    }
  }

  /** returns the name of a sensor of the ERS-210 */
  static const char* getSensorNameERS210(sensors sensor)
  {
    switch (sensor)
    {
    case headTilt: return "headTilt";
    case headPan: return "headPan";
    case headRoll: return "headRoll";
    case headBack: return "headBack";
    case headFront: return "headFront";
    case psd: return "psd";
    case mouth: return "mouth";
    case chin: return "chin";
    case legFL1: return "legFL1";
    case legFL2: return "legFL2";
    case legFL3: return "legFL3";
    case pawFL: return "pawFL";
    case legHL1: return "legHL1";
    case legHL2: return "legHL2";
    case legHL3: return "legHL3";
    case pawHL: return "pawHL";
    case legFR1: return "legFR1";
    case legFR2: return "legFR2";
    case legFR3: return "legFR3";
    case pawFR: return "pawFR";
    case legHR1: return "legHR1";
    case legHR2: return "legHR2";
    case legHR3: return "legHR3";
    case pawHR: return "pawHR";
    case tailPan: return "tailPan";
    case tailTilt: return "tailTilt";
    case thermo: return "thermo";
    case back: return "back";
    case accelerationX: return "accelerationX";
    case accelerationY: return "accelerationY";
    case accelerationZ: return "accelerationZ";
    default: return "unknown sensor";
    }
  }

  /** returns the name of a sensor of the ERS-7 */
  static const char* getSensorNameERS7(sensors sensor)
  {
    switch (sensor)
    {
    case headTilt1: return "headTilt1";
    case headTilt2: return "headTilt2";
    case backR: return "backRear";
    case backF: return "backFront";
    case headPsdNear: return "headPsdNear";
    case wlan: return "wlan";
    case backM: return "backMiddle";
    case head: return "head";
    case headPsdFar: return "headPsdFar";
    case bodyPsd: return "bodyPsd";
    default: return getSensorNameERS210(sensor);
    }
  }
};

/**
 * Streaming operator that writes a SensorData to a stream.
 * @param stream The stream to write on.
 * @param sensorData The SensorData object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const SensorData& sensorData);

/**
 * Streaming operator that reads a SensorData from a stream.
 * @param stream The stream to read from.
 * @param sensorData The SensorData object.
 * @return The stream.
 */ 
In& operator>>(In& stream, SensorData& sensorData);

#endif //__SensorData_h_

/*
 * Change log :
 * 
 * $Log: SensorData.h,v $
 * Revision 1.3  2004/04/12 18:00:45  roefer
 * SensorView added
 *
 * Revision 1.2  2003/12/31 20:16:14  roefer
 * SensorData for ERS-7
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2003/05/05 11:59:57  dueffert
 * comment corrected
 *
 * Revision 1.3  2003/05/03 15:26:29  risler
 * added refValue to SensorData
 *
 * Revision 1.2  2003/05/02 18:26:18  risler
 * SensorDataBuffer added
 * replaced SensorData with SensorDataBuffer
 * full SensorData resolution now accessible
 *
 * Revision 1.1  2002/09/10 15:26:40  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.9  2002/02/01 15:02:41  risler
 * Acceleration sensor adjusted to coordinate system
 *
 * Revision 1.8  2002/01/25 16:52:40  loetzsch
 * function for enum names added
 *
 * Revision 1.7  2001/12/28 09:00:02  roefer
 * Streaming operator and frameNumber inserted
 *
 * Revision 1.6  2001/12/23 23:38:04  petters
 * method init added
 *
 * Revision 1.5  2001/12/12 20:21:12  petters
 * Streaming for SensorData / Image implemented; Conflict solved
 *
 * Revision 1.4  2001/12/10 17:47:06  risler
 * change log added
 *
 */
