/**
 * @file ObstaclesPercept.h
 * 
 * Declaration of class ObstaclesPercept
 * @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
 */ 

#ifndef __ObstaclesPercept_h_
#define __ObstaclesPercept_h_


#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector2.h"

/**
* The class represents perceived obstacles on the field.
*/
class ObstaclesPercept
{
  public:
    enum ObstacleType {border, goal, teammate, opponent, unknown};
    enum {maxNumberOfPoints = 200}; /**< Specifies the maximum number of points per line type. */
  
    Vector2<int> nearPoints[maxNumberOfPoints]; /**< The nearest green points*/
    Vector2<int> farPoints[maxNumberOfPoints]; /**< The most distant green points*/
    bool farPointIsOnImageBorder[maxNumberOfPoints]; /**< Indicates if the far point is an obstacle or the image border */
    ObstacleType obstacleType[maxNumberOfPoints];

    int numberOfPoints; /**< The number of pairs of points */

    enum{ownGoal = 0, opponentGoal = 1};
    bool angleToFreePartOfGoalWasDetermined[2]; 
    double angleToFreePartOfGoal[2],
           distanceToFreePartOfGoal[2],
           widthOfFreePartOfGoal[2];
    /**
     * Constructor.
     */
    ObstaclesPercept() {reset(0);}

    /**
     * The function empties the obstacle percept.
     */
    void reset(unsigned long frameNumber);

    /**
     * The function adds a new pair of points to the obstacle percept.
     * @param nearPoint The near point.
     * @param farPoint The far point.
     * @param farPointIsOnImageBorder Specifies whether farPoint was found on the image border
     */
    void add(const Vector2<int>& nearPoint, const Vector2<int>& farPoint, bool farPointIsOnImageBorder);

    /**
     * The function adds a new pair of points to the obstacle percept.
     * @param nearPoint The near point.
     * @param farPoint The far point.
     * @param farPointIsOnImageBorder Specifies whether farPoint was found on the image border
     * @param obstacleType The type of the obstacle
     */
    void add(const Vector2<int>& nearPoint, const Vector2<int>& farPoint, bool farPointIsOnImageBorder, ObstacleType obstacleType);

    /** The frame number when perceived. */
    unsigned long frameNumber;
};

/**
 * Streaming operator that reads a ObstaclesPercept from a stream.
 * @param stream The stream from which is read.
 * @param obstaclesPercept The ObstaclesPercept object.
 * @return The stream.
 */ 
In& operator>>(In& stream,ObstaclesPercept& obstaclesPercept);
 
/**
 * Streaming operator that writes a ObstaclePercept to a stream.
 * @param stream The stream to write on.
 * @param obstaclesPercept The ObstaclesPercept object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const ObstaclesPercept& obstaclesPercept);


#endif //__ObstaclesPercept_h_

/*
 * Change log :
 * 
 * $Log: ObstaclesPercept.h,v $
 * Revision 1.5  2004/03/01 11:47:50  juengel
 * Moved enum ObstacleType to class ObstaclesPercept.
 *
 * Revision 1.4  2004/02/28 13:50:36  juengel
 * Added ObstacleType.
 *
 * Revision 1.3  2004/01/19 14:53:46  dueffert
 * all frameNumbers (and not only some of them) are unsigned long now
 *
 * Revision 1.2  2003/11/12 16:19:35  goehring
 * frameNumber added to percepts
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/08/08 11:37:30  dueffert
 * doxygen docu corrected
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.3  2003/06/25 18:42:14  juengel
 * Added nearPoints, farPoints[maxNumberOfPoints] and farPointIsOnImageBorder to ObstaclesPercept.
 *
 * Revision 1.2  2003/05/26 08:12:11  juengel
 * Moved angleToFreePartOfGoal from specialPercept to obstaclesPercept.
 *
 * Revision 1.1  2003/03/10 13:52:09  juengel
 * Added ObstaclesPercept
 *
 *
 */
