/** 
* @file ColorTable64.h
* Declaration of class ColorTable64.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
* @author <A href=mailto:martin@martin-loetzsch.de>Martin Ltzsch</A>
*/

#ifndef _ColorTable64_h_
#define _ColorTable64_h_

// forward declaration
class ColorTable64;

#include "ColorTable.h"
#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector3.h"
#include "ColorClassImage.h"
#include "Image.h"

/**
* @class ColorTable64
*
* Contains a ColorTable64 which can decode the color for
* every 4 * 4 * 4 cube in the 255 * 255 * 255 YUV color space.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
* @author <A href=mailto:martin@martin-loetzsch.de>Martin Ltzsch</A>
*/
class ColorTable64 : public ColorTable
{
public:
  /** Constructor */
  ColorTable64();

  /** Destructor */
  ~ColorTable64();

  /** 
  * Calculates the color class of a pixel.
  * @param y the y value of the pixel
  * @param u the u value of the pixel
  * @param v the v value of the pixel
  * @return the color class
  */
  virtual colorClass getColorClass (const unsigned char y, 
    const unsigned char u, 
    const unsigned char v) const
  {
    return (colorClass) colorClasses[y/4][u/4][v/4];
  }

  /**
  * Computes an axis aligned box around all positions of a specified color
  * in YUV space
  * @param color The color
  * @param pNear The corner of the box nearest to the origin
  * @param pFar The corner of the box farthest to the origin
  */
  void getBoxAroundColorClass(colorClass color, 
                              Vector3<int>& pNear, Vector3<int>& pFar);
  /**
  * Segments an image to an color class image.
  * 
  * This doesn't need to used in the image processor, but is needed for visualisation
  * of color tables.
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  */
  virtual void generateColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage) const;

  /**
  * Generates an image that contains all pixels that have the specified color class.
  * 
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  * @param colorClass The color class. 
  */
  virtual void generateColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage,
    colorClass colorClass
    ) const;

  /**
  * Segments an image to an color class image.
  * 
  * This doesn't need to used in the image processor, but is needed for visualisation
  * of color tables.
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  */
  virtual void generateHighResColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage) const;

  /**
  * Generates an image that contains all pixels that have the specified color class.
  * 
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  * @param colorClass The color class. 
  */
  virtual void generateHighResColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage,
    colorClass colorClass
    ) const;

  /** 
  * The color table.
  * Each element in the array contains the color class of a 4x4x4 cube in the color space.
  */
  unsigned char colorClasses[64][64][64];

  //!@name Members to be used by the ColorTable64 dialog
  //!@{

  /** Sets the color class of every 4x4x4 to noColor */
  void clear();
  
  /** Sets all cubes that have the given color class to noColor */
  void clearChannel(colorClass colorClass);
  
  /** 
  * Sets the color class for a pixel in the color space given by y, u, v
  * to the given color class.
  */
  void addColorClass(
    colorClass colorClass,
    unsigned char y,
    unsigned char u,
    unsigned char v
    );

  /** 
  * Sets the color class for a cube with the size "range" around a pixel
  * given by y,u,v to the given color class.
  */
  void addColorClass(
    colorClass colorClass,
    unsigned char y, 
    unsigned char u, 
    unsigned char v, 
    unsigned char range
    );

  /**
  * Sets the color class for all sub cubes within the specified cuboid to the given color class.
  */
  void addCuboidToColorClass(
    colorClass colorClass,
    unsigned char yMin, 
    unsigned char uMin, 
    unsigned char vMin, 
    unsigned char yMax, 
    unsigned char uMax, 
    unsigned char vMax
    );
  
  /**
  * Sets the color class for a cube with the size "range" around a pixel
  * given by y,u,v to noColor.
  */
  void removeColorClass(
    unsigned char y, 
    unsigned char u, 
    unsigned char v, 
    unsigned char range
    );

  //!@}
  
};

/**
 * Streaming operator that reads a ColorTable64 from a stream.
 * @param stream The stream from which is read.
 * @param colorTable64 The ColorTable64 object.
 * @return The stream.
 */ 
In& operator>>(In& stream,ColorTable64& colorTable64);
 
/**
 * Streaming operator that writes a ColorTable64 to a stream.
 * @param stream The stream to write on.
 * @param colorTable64 The ColorTable64 object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const ColorTable64& colorTable64);

#endif   //  _ColorTable64_h_

/*
 * Change log :
 * 
 * $Log: ColorTable64.h,v $
 * Revision 1.5  2004/04/07 13:00:44  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.2  2004/04/02 14:46:18  risler
 * added segmented high resolution image
 * colortable dialog supports high resolution images
 *
 * Revision 1.1.1.1  2004/03/29 08:28:45  Administrator
 * initial transfer from tamara
 *
 * Revision 1.4  2004/03/05 17:56:39  tim
 * corrected doxygen comments
 *
 * Revision 1.3  2004/02/16 18:23:47  tim
 * Added getBoxAroundColorClass
 *
 * Revision 1.2  2003/10/29 13:24:42  juengel
 * Added addCuboidToColorClass to ColorTable64
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/25 11:23:52  juengel
 * Removed BlobCollection.
 *
 * Revision 1.2  2003/07/21 13:43:51  juengel
 * generateColorClassImage returns void
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2003/04/14 16:12:09  loetzsch
 * ATH after GermanOpen CVS merge
 * improved color table dialog bar
 *
 * Revision 1.2  2003/04/10 08:29:19  juengel
 * no message
 *
 * Revision 1.3  2003/02/18 21:29:17  osterhues
 * Changed all instances of ColorTable64 to new base class ColorTable
 *
 * Revision 1.2  2003/01/09 14:11:31  jhoffman
 * added some stuff needed for ISL
 *
 * Revision 1.1  2002/09/10 15:26:40  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/07/23 13:29:28  loetzsch
 * removed base class of ColorTable
 *
 * new streaming classes
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.7  2001/12/22 20:35:35  loetzsch
 * removed ColorTable::segmentImage, replaced by
 * ColorTable64::generateColorClassImage
 *
 * Revision 1.6  2001/12/21 14:22:41  roefer
 * warnings fixed: inline/eof
 *
 * Revision 1.5  2001/12/21 00:49:48  loetzsch
 * segmentImage in die Basisklasse verschoben
 *
 * Revision 1.4  2001/12/20 16:01:37  loetzsch
 * Streaming operators added
 *
 * Revision 1.3  2001/12/19 14:04:47  loetzsch
 * In ColorTables Pointer gegen Referenzen ausgetauscht,
 * Funktion segmentImage hinzugefgt und implementiert
 *
 * Revision 1.2  2001/12/17 19:24:42  juengel
 * no message
 *
 */
